﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestIpc.h"
#include "util_TestIpcData.h"
#include "util_TestMemory.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

extern nn::Result ClientSendLightRequest(nn::svc::Handle handle);
extern "C" void nnMain();

namespace {

const char* PortName = "TestSeq";
const int64_t SleepTime = 100 * 1000 * 1000;
const size_t MapSendSize = 0x1000;
const uint8_t SendMapSendData = 0xa;

char g_ServerBuffer[2][0x1000] __attribute__((aligned(0x1000)));
char g_ClientBuffer[3][0x1000] __attribute__((aligned(0x1000)));
char g_ClientMapBuffer[3][0x1000] __attribute__((aligned(0x1000)));
char g_ServerPointerBuffer[2][0x1000] __attribute__((aligned(0x1000)));

nn::svc::Handle g_ReceiveList[10];

int32_t g_Sequence;

enum PortType
{
    PortType_Port,
    PortType_NamedPort,
    PortType_Session,
};

enum SendMethod
{
    SendMethod_SyncTls,
    SendMethod_SyncUserBuffer,
    SendMethod_AsyncUserBuffer,
};

enum RecvMethod
{
    RecvMethod_Tls,
    RecvMethod_UserBuffer,
};

struct ServerData
{
    nn::svc::Handle session;
    nn::Result result;
    PortType portType;
    RecvMethod method;
    int32_t recvNum;
    int32_t bufferIndex;
    DataType dataType;
};

struct ClientData
{
    nn::svc::Handle session;
    nn::Result result;
    PortType portType;
    SendMethod method;
    int32_t bufferIndex;
    DataType dataType;
    bool doCheckAsyncFailure;
};

void SetupTestDataWithNamedPort(ServerData* serverData, ClientData* clientData)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    serverData->portType = PortType_NamedPort;
    clientData->portType = PortType_NamedPort;

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);

    result = nn::svc::ConnectToNamedPort(&clientData->session, PortName);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverData->session, serverPort);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void SetupTestDataWithPort(ServerData* serverData, ClientData* clientData)
{
    nn::Result result;
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;

    serverData->portType = PortType_Port;
    clientData->portType = PortType_Port;

    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    result = nn::svc::ConnectToPort(&clientData->session, clientPort);
    ASSERT_RESULT_SUCCESS(result);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::AcceptSession(&serverData->session, serverPort);
    ASSERT_RESULT_SUCCESS(result);
}

void SetupTestDataWithSession(ServerData* serverData, ClientData* clientData)
{
    nn::Result result;

    serverData->portType = PortType_Session;
    clientData->portType = PortType_Session;

    result = nn::svc::CreateSession(&serverData->session, &clientData->session, false, 0);
    ASSERT_RESULT_SUCCESS(result);
}

void SetupTestData(ServerData* serverData, ClientData* clientData, PortType port)
{
    clientData->doCheckAsyncFailure = false;

    switch(port)
    {
    case PortType_Port:
        SetupTestDataWithPort(serverData, clientData);
        break;
    case PortType_NamedPort:
        SetupTestDataWithNamedPort(serverData, clientData);
        break;
    case PortType_Session:
        SetupTestDataWithSession(serverData, clientData);
        break;
    default: FAIL();
    }
}

const PortType PortTypeArray[] = {
    PortType_Port,
    PortType_NamedPort,
    PortType_Session,
};
const int32_t PortTypeArraySize = static_cast<int32_t>(sizeof(PortTypeArray) / sizeof(PortType));

const SendMethod SendMethodArray[] = {
    SendMethod_SyncTls,
    SendMethod_SyncUserBuffer,
    SendMethod_AsyncUserBuffer,
};
const int32_t SendMethodArraySize = static_cast<int32_t>(sizeof(SendMethodArray) / sizeof(SendMethod));

const RecvMethod RecvMethodArray[] = {
    RecvMethod_Tls,
    RecvMethod_UserBuffer,
};
const int32_t RecvMethodArraySize = static_cast<int32_t>(sizeof(RecvMethodArray) / sizeof(RecvMethod));

class IpcTestCondition
{
    public:
        IpcTestCondition() : m_Port(0), m_RecvMethod(0), m_SendMethod(0) {};

        bool IsEnd() const
        {
            return m_Port >= PortTypeArraySize;
        }

        void NextTest(ServerData* serverData, ClientData* clientData)
        {
            SetupTestData(serverData, clientData, PortTypeArray[m_Port]);

            serverData->method = RecvMethodArray[m_RecvMethod];
            clientData->method = SendMethodArray[m_SendMethod];

            m_SendMethod++;

            if (m_SendMethod >= SendMethodArraySize)
            {
                m_SendMethod = 0;
                m_RecvMethod++;
            }

            if (m_RecvMethod >= RecvMethodArraySize)
            {
                m_RecvMethod = 0;
                m_Port++;
            }
        }

        void PrintCurrentCondition() const
        {
            NN_LOG("PortIndex: %d, RecvMethodIndex: %d, SendMethodIndex: %d\n"
                    , m_Port, m_RecvMethod, m_SendMethod);
        }

        PortType GetPort() const { return PortTypeArray[m_Port]; }
        RecvMethod GetRecvMethod() const { return RecvMethodArray[m_RecvMethod]; }
        SendMethod GetSendMethod() const { return SendMethodArray[m_SendMethod]; }

    private:
        int32_t m_Port;
        int32_t m_RecvMethod;
        int32_t m_SendMethod;
};

void GetSendIpcBuffer(nn::Bit32** ppMsgBuffer, ClientData* data)
{
    switch(data->method)
    {
    case SendMethod_SyncTls:
        *ppMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        return;
    case SendMethod_SyncUserBuffer:
    case SendMethod_AsyncUserBuffer:
        *ppMsgBuffer = reinterpret_cast<nn::Bit32*>(g_ClientBuffer[data->bufferIndex]);
        return;
    default: FAIL();
    }
}

void GetReceiveIpcBuffer(nn::Bit32** ppMsgBuffer, ServerData* data)
{
    switch(data->method)
    {
    case RecvMethod_Tls:
        *ppMsgBuffer = nn::svc::ipc::GetMessageBuffer();
        return;
    case RecvMethod_UserBuffer:
        *ppMsgBuffer = reinterpret_cast<nn::Bit32*>(g_ServerBuffer[data->bufferIndex]);
        return;
    default: FAIL();
    }
}

void GetReceiveIpcBufferSize(size_t* pOutSize, ServerData* data)
{
    switch(data->method)
    {
    case RecvMethod_Tls:
        *pOutSize = sizeof(nn::svc::GetThreadLocalRegion()->messageBuffer);
        return;
    case RecvMethod_UserBuffer:
        *pOutSize = sizeof(g_ServerBuffer[data->bufferIndex]);
        return;
    default: FAIL();
    }
}

void TestSendRequest(ClientData* data)
{
    nn::svc::Handle eventHandle;
    nn::Result tmpResult;
    int32_t index;

    switch(data->method)
    {
    case SendMethod_SyncTls:
        data->result = nn::svc::SendSyncRequest(data->session);
        break;
    case SendMethod_SyncUserBuffer:
        data->result = nn::svc::SendSyncRequestWithUserBuffer(
                        reinterpret_cast<uintptr_t>(g_ClientBuffer[data->bufferIndex]),
                        sizeof(g_ClientBuffer[data->bufferIndex]),
                        data->session);
        break;
    case SendMethod_AsyncUserBuffer:
        data->result =  nn::svc::SendAsyncRequestWithUserBuffer(
                        &eventHandle,
                        reinterpret_cast<uintptr_t>(g_ClientBuffer[data->bufferIndex]),
                        sizeof(g_ClientBuffer[data->bufferIndex]),
                        data->session);
        if (data->result.IsSuccess())
        {
            data->result = nn::svc::WaitSynchronization(&index, &eventHandle, 1, -1);
            tmpResult = nn::svc::CloseHandle(eventHandle);
            NN_ASSERT_RESULT_SUCCESS(tmpResult);
        }
        break;
    default: FAIL();
    }
}

void TestReceiveRequest(int32_t* index, ServerData* data)
{
    switch(data->method)
    {
    case RecvMethod_Tls:
        data->result = nn::svc::ReplyAndReceive(
                        index, g_ReceiveList, data->recvNum, nn::svc::INVALID_HANDLE_VALUE, -1);
        break;
    case RecvMethod_UserBuffer:
        data->result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        index,
                        reinterpret_cast<uintptr_t>(g_ServerBuffer[data->bufferIndex]),
                        sizeof(g_ServerBuffer[data->bufferIndex]),
                        g_ReceiveList, data->recvNum, nn::svc::INVALID_HANDLE_VALUE, -1);
        break;
    default: FAIL();
    }
}

void TestReplyRequest(nn::svc::Handle target, ServerData* data)
{
    int32_t index;
    switch(data->method)
    {
    case RecvMethod_Tls:
        data->result = nn::svc::ReplyAndReceive(
                        &index, g_ReceiveList, 0, target, 0);
        break;
    case RecvMethod_UserBuffer:
        data->result = nn::svc::ReplyAndReceiveWithUserBuffer(
                        &index,
                        reinterpret_cast<uintptr_t>(g_ServerBuffer[data->bufferIndex]),
                        sizeof(g_ServerBuffer[data->bufferIndex]),
                        g_ReceiveList, 0, target, 0);
        break;
    default: FAIL();
    }
}

void ClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ClientData* data = reinterpret_cast<ClientData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetSendIpcBuffer(&pMsgBuffer, data);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0001, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);

    TestSendRequest(data);
    if (data->doCheckAsyncFailure)
    {
        CheckAsyncFailure(pMsgBuffer);
    }
}

void LightClientThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    ClientData* data = reinterpret_cast<ClientData*>(arg);
    data->result = ClientSendLightRequest(data->session);
}

void SendMapThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ClientData* data = reinterpret_cast<ClientData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetSendIpcBuffer(&pMsgBuffer, data);

    MakeOnlyMapSendData(
            pMsgBuffer, g_ClientMapBuffer[data->bufferIndex], MapSendSize, SendMapSendData);

    TestSendRequest(data);

    nn::svc::MemoryInfo info;
    GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(g_ClientMapBuffer[data->bufferIndex]));
    NN_ASSERT(info.state == nn::svc::MemoryState_CodeData);
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);
    GetMemoryInfo(&info, reinterpret_cast<uintptr_t>(pMsgBuffer));
    NN_ASSERT(info.permission == nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT(info.attribute == 0);
}

void WaitThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    int32_t index;
    data->result = nn::svc::WaitSynchronization(&index, g_ReceiveList, data->recvNum, 0);
}

void PrepareServerReceiveBuffer(nn::Bit32* pMsgBuffer, const ServerData* data)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0000, 0, 0, 0, 0, 0, 0, 2);
    nn::svc::ipc::MessageBuffer::ReceiveListEntry
        receiveEntry(reinterpret_cast<char*>(g_ServerPointerBuffer[data->bufferIndex]), sizeof(g_ServerPointerBuffer[data->bufferIndex]));
    int offset = ipcMsg.Set(ipcHeader);
    offset = ipcMsg.Set(offset, receiveEntry);
}

void PrepareServerReplyBuffer(nn::Bit32* pMsgBuffer)
{
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);
}

void ReceiveAndReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetReceiveIpcBuffer(&pMsgBuffer, data);

    PrepareServerReceiveBuffer(pMsgBuffer, data);

    int32_t index;
    TestReceiveRequest(&index, data);
    NN_ASSERT_RESULT_SUCCESS(data->result);

    PrepareServerReplyBuffer(pMsgBuffer);

    TestReplyRequest(g_ReceiveList[index], data);
}

void ReceiveAndSleepThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetReceiveIpcBuffer(&pMsgBuffer, data);

    PrepareServerReceiveBuffer(pMsgBuffer, data);

    int32_t index;
    TestReceiveRequest(&index, data);
    NN_ASSERT_RESULT_SUCCESS(data->result);

    // Wait Thread に切り替える
    nn::svc::SleepThread(SleepTime);

    PrepareServerReplyBuffer(pMsgBuffer);

    TestReplyRequest(g_ReceiveList[index], data);
}

void MultiReceiveAndReplyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetReceiveIpcBuffer(&pMsgBuffer, data);

    PrepareServerReceiveBuffer(pMsgBuffer, data);

    int32_t index;
    TestReceiveRequest(&index, data);
    NN_ASSERT_RESULT_SUCCESS(data->result);

    // WaitThread に処理を移す
    nn::svc::SleepThread(SleepTime);


    // 要求受信中に他のセッションからの要求を受信する
    for (int32_t i = 0; i < data->recvNum - 1; i++)
    {
        PrepareServerReceiveBuffer(pMsgBuffer, data);
        TestReceiveRequest(&index, data);
        NN_ASSERT_RESULT_SUCCESS(data->result);
    }

    // 全ての要求の返答を返す
    for (int32_t i = 0; i < data->recvNum; i++)
    {
        PrepareServerReplyBuffer(pMsgBuffer);
        TestReplyRequest(g_ReceiveList[i], data);
        NN_ASSERT_RESULT_FAILURE_VALUE(data->result, nn::svc::ResultTimeout());
    }
}

void LoopReceiveAndSleepThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    GetReceiveIpcBuffer(&pMsgBuffer, data);

    NN_ASSERT(g_Sequence == 0);

    for(;;)
    {
        PrepareServerReceiveBuffer(pMsgBuffer, data);

        int32_t index;
        TestReceiveRequest(&index, data);
        if (data->result <= nn::svc::ResultCancelled())
        {
            break;
        }
        NN_ASSERT_RESULT_SUCCESS(data->result);

        // WaitThread に切り替える
        if (g_Sequence == 0)
        {
            g_Sequence = 1;
        }
        while(g_Sequence == 1)
        {
            nn::svc::SleepThread(SleepTime);
        }

        PrepareServerReplyBuffer(pMsgBuffer);

        TestReplyRequest(g_ReceiveList[index], data);

        NN_ASSERT_RESULT_FAILURE_VALUE(data->result, nn::svc::ResultTimeout());
    }
}

void TestReceiveDataServer(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ServerData* data = reinterpret_cast<ServerData*>(arg);

    nn::Bit32* pMsgBuffer = nullptr;
    size_t msgSize = 0;
    GetReceiveIpcBuffer(&pMsgBuffer, data);
    GetReceiveIpcBufferSize(&msgSize, data);

    PrepairServerIpc(pMsgBuffer, data->dataType);

    int32_t index;
    TestReceiveRequest(&index, data);
    ASSERT_RESULT_SUCCESS(data->result);

    TestIpcMapData mapData = { 0 };
    ServerCheckIpcData(&mapData, pMsgBuffer, msgSize, data->dataType);

    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 0, 0, 0, 0, 0, 0, 0);
    ipcMsg.Set(ipcHeader);

    TestReplyRequest(g_ReceiveList[index], data);
}

void FinalizeClientData(const ClientData* testData, nn::svc::Handle handle)
{
    nn::Result result;
    switch(testData->dataType)
    {
    case DataType_ProcessId:
        break;
    case DataType_CopyHandle:
        result = nn::svc::CloseHandle(handle);
        NN_ASSERT_RESULT_SUCCESS(result);
        break;
    case DataType_MoveHandle:
        break;
    case DataType_PointerWithIpc:
        break;
    case DataType_PointerWithOne:
        break;
    case DataType_PointerWithMulti:
        break;
    case DataType_MapSend:
        CheckClientBaseMapArea(testData->dataType);
        break;
    case DataType_MapRecv:
        CheckClientBaseMapArea(testData->dataType);
        break;
    case DataType_MapExch:
        CheckClientBaseMapArea(testData->dataType);
        break;
    case DataType_Raw:
        break;
    case DataType_All:
        result = nn::svc::CloseHandle(handle);
        NN_ASSERT_RESULT_SUCCESS(result);
        CheckClientBaseMapArea(testData->dataType);
        break;
    default: FAIL();
    }
}

void TestErrorHandlingClient(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ClientData* data = reinterpret_cast<ClientData*>(arg);

    nn::Bit32* pMsgBuffer;
    GetSendIpcBuffer(&pMsgBuffer, data);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::Handle handle;

    // 失敗する要求を投げる
    {
        nn::svc::ipc::MessageBuffer::MessageHeader ipcHeader(0x0002, 1, 0, 0, 0, 0, 0, 0);
        ipcMsg.Set(ipcHeader);
        int offset = ipcMsg.Set(nn::svc::ipc::MessageBuffer::SpecialHeader(false, 0, 1));
        offset = ipcMsg.SetHandle(offset, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);

        TestSendRequest(data);
        if (data->method == SendMethod_AsyncUserBuffer)
        {
            ASSERT_RESULT_SUCCESS(data->result);
            CheckAsyncFailure(pMsgBuffer);
        }
        else
        {
            ASSERT_RESULT_FAILURE_VALUE(data->result, nn::svc::ResultInvalidCombination());
        }
    }

    MakeClientData(&handle, pMsgBuffer, data->dataType);

    TestSendRequest(data);

    FinalizeClientData(data, handle);
}

void TestSendDataClient(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    ClientData* data = reinterpret_cast<ClientData*>(arg);

    nn::Bit32* pMsgBuffer;
    GetSendIpcBuffer(&pMsgBuffer, data);
    nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
    nn::svc::Handle handle;

    MakeClientData(&handle, pMsgBuffer, data->dataType);

    TestSendRequest(data);

    FinalizeClientData(data, handle);

    if (data->doCheckAsyncFailure)
    {
        CheckAsyncFailure(pMsgBuffer);
    }
}

} // namespace

// TEST 901-34
// 先にWait してしまう
TEST(IpcSequence, WaitBeforeReceiveIpc)
{
    IpcTestCondition cond;

    while(!cond.IsEnd())
    {
        ServerData waitServerData;
        ServerData receiveServerData;
        ClientData clientData;

        cond.NextTest(&receiveServerData, &clientData);
        AutoHandleClose sSession(receiveServerData.session);
        AutoHandleClose cSession(clientData.session);

        nn::Result result;
        uintptr_t pc;
        uintptr_t sp;
        uintptr_t param;
        uintptr_t priority;
        uintptr_t idealCore;

        TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * 3);

        uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

        // Client
        clientData.bufferIndex = 0;
        pc = reinterpret_cast<uintptr_t>(ClientThread);
        sp = stackBottom;
        param = reinterpret_cast<uintptr_t>(&clientData);
        priority = TestLowestThreadPriority;
        idealCore = 0;
        TestThread clientThread(pc, param, sp, priority, idealCore);
        clientThread.Start();

        g_ReceiveList[0] = receiveServerData.session;
        g_ReceiveList[1] = nn::svc::INVALID_HANDLE_VALUE;
        receiveServerData.recvNum = 1;
        receiveServerData.bufferIndex = 0;
        waitServerData = receiveServerData;

        // Client がデータを Send するまで待つ
        nn::svc::SleepThread(SleepTime);

        // Wait Server Thread
        pc = reinterpret_cast<uintptr_t>(WaitThread);
        sp = stackBottom - DefaultStackSize;
        param = reinterpret_cast<uintptr_t>(&waitServerData);
        priority = TestLowestThreadPriority - 1;
        idealCore = 0;
        TestThread waitThread(pc, param, sp, priority, idealCore);
        waitThread.Start();

        // Receive Server Thread
        pc = reinterpret_cast<uintptr_t>(ReceiveAndReplyThread);
        sp = stackBottom - DefaultStackSize * 2;
        param = reinterpret_cast<uintptr_t>(&receiveServerData);
        priority = TestLowestThreadPriority - 1;
        idealCore = 0;
        TestThread receiveThread(pc, param, sp, priority, idealCore);
        receiveThread.Start();

        clientThread.Wait();
        waitThread.Wait();
        receiveThread.Wait();

        ASSERT_RESULT_SUCCESS(clientData.result);
        ASSERT_RESULT_SUCCESS(waitServerData.result);
        ASSERT_RESULT_FAILURE_VALUE(receiveServerData.result, nn::svc::ResultTimeout());
    }
}

// TEST 901-35
// Receive 後にWait を行い、その後に Reply を返す
TEST(IpcSequence, WaitAfterReceiveIpc)
{
    IpcTestCondition cond;

    while(!cond.IsEnd())
    {
        ServerData waitServerData;
        ServerData receiveServerData;
        ClientData clientData;

        cond.NextTest(&receiveServerData, &clientData);
        AutoHandleClose sSession(receiveServerData.session);
        AutoHandleClose cSession(clientData.session);

        nn::Result result;
        uintptr_t pc;
        uintptr_t sp;
        uintptr_t param;
        uintptr_t priority;
        uintptr_t idealCore;

        TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * 3);

        uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

        // Client
        clientData.bufferIndex = 0;
        pc = reinterpret_cast<uintptr_t>(ClientThread);
        sp = stackBottom;
        param = reinterpret_cast<uintptr_t>(&clientData);
        priority = TestLowestThreadPriority;
        idealCore = 0;
        TestThread clientThread(pc, param, sp, priority, idealCore);
        clientThread.Start();

        g_ReceiveList[0] = receiveServerData.session;
        g_ReceiveList[1] = nn::svc::INVALID_HANDLE_VALUE;
        receiveServerData.recvNum = 1;
        receiveServerData.bufferIndex = 0;
        waitServerData = receiveServerData;

        // Client がデータを Send するまで待つ
        nn::svc::SleepThread(SleepTime);

        // Receive Server Thread
        pc = reinterpret_cast<uintptr_t>(ReceiveAndSleepThread);
        sp = stackBottom - DefaultStackSize * 2;
        param = reinterpret_cast<uintptr_t>(&receiveServerData);
        priority = TestLowestThreadPriority - 1;
        idealCore = 0;
        TestThread receiveThread(pc, param, sp, priority, idealCore);
        receiveThread.Start();

        // Wait Server Thread
        pc = reinterpret_cast<uintptr_t>(WaitThread);
        sp = stackBottom - DefaultStackSize;
        param = reinterpret_cast<uintptr_t>(&waitServerData);
        priority = TestLowestThreadPriority;
        idealCore = 0;
        TestThread waitThread(pc, param, sp, priority, idealCore);
        waitThread.Start();

        clientThread.Wait();
        waitThread.Wait();
        receiveThread.Wait();

        ASSERT_RESULT_SUCCESS(clientData.result);
        ASSERT_RESULT_FAILURE_VALUE(waitServerData.result, nn::svc::ResultTimeout());
        ASSERT_RESULT_FAILURE_VALUE(receiveServerData.result, nn::svc::ResultTimeout());
    }
}

// TEST 901-36
// 要求の受信中に他のセッションから要求を受け付ける
TEST(IpcSequence, ReceiveFromMultipleSession)
{
    IpcTestCondition cond1;

    while(!cond1.IsEnd())
    {
        ServerData serverData1;
        ClientData clientData1;
        cond1.NextTest(&serverData1, &clientData1);
        AutoHandleClose sSession1(serverData1.session);
        AutoHandleClose cSession1(clientData1.session);

        IpcTestCondition cond2;
        while(!cond2.IsEnd())
        {
            ServerData serverData2; // Session を作るために、用意
            ClientData clientData2;
            cond2.NextTest(&serverData2, &clientData2);
            AutoHandleClose sSession2(serverData2.session);
            AutoHandleClose cSession2(clientData2.session);

            nn::Result result;
            uintptr_t pc;
            uintptr_t sp;
            uintptr_t param;
            uintptr_t priority;
            uintptr_t idealCore;

            TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * 4);

            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            // Client1
            clientData1.bufferIndex = 0;
            pc = reinterpret_cast<uintptr_t>(ClientThread);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData1);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread1(pc, param, sp, priority, idealCore);
            clientThread1.Start();

            // Client2
            clientData2.bufferIndex = 1;
            pc = reinterpret_cast<uintptr_t>(ClientThread);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&clientData2);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread2(pc, param, sp, priority, idealCore);
            clientThread2.Start();

            g_ReceiveList[0] = serverData1.session;
            g_ReceiveList[1] = serverData2.session;
            g_ReceiveList[2] = nn::svc::INVALID_HANDLE_VALUE;
            serverData1.recvNum = 2;
            serverData1.bufferIndex = 0;
            serverData2.recvNum = 2;

            // Client がデータを Send するまで待つ
            nn::svc::SleepThread(SleepTime);

            // Receive Server Thread
            pc = reinterpret_cast<uintptr_t>(MultiReceiveAndReplyThread);
            sp = stackBottom - DefaultStackSize * 2;
            param = reinterpret_cast<uintptr_t>(&serverData1);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread receiveThread(pc, param, sp, priority, idealCore);

            // Wait Server Thread
            pc = reinterpret_cast<uintptr_t>(WaitThread);
            sp = stackBottom - DefaultStackSize * 3;
            param = reinterpret_cast<uintptr_t>(&serverData2);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread waitThread(pc, param, sp, priority, idealCore);

            receiveThread.Start();
            waitThread.Start();

            clientThread1.Wait();
            clientThread2.Wait();
            receiveThread.Wait();
            waitThread.Wait();

            ASSERT_RESULT_SUCCESS(clientData1.result);
            ASSERT_RESULT_SUCCESS(clientData2.result);
            ASSERT_RESULT_FAILURE_VALUE(serverData1.result, nn::svc::ResultTimeout());
            ASSERT_RESULT_SUCCESS(serverData2.result);
        }
    }
}

// TEST 901-37
// Receive 後に同じセッションを使ってリクエストを出し、その後にWait を行う
TEST(IpcSequence, ReceiveFromMultipleSessionAndWait)
{
    nn::Result result;

    IpcTestCondition cond1;

    while(!cond1.IsEnd())
    {
        ServerData serverData1;
        ClientData clientData1;
        cond1.NextTest(&serverData1, &clientData1);
        AutoHandleClose sSession1(serverData1.session);
        AutoHandleClose cSession1(clientData1.session);

        IpcTestCondition cond2;
        while(!cond2.IsEnd())
        {
            ServerData serverData2; // Session を作るために、用意
            ClientData clientData2;
            cond2.NextTest(&serverData2, &clientData2);
            AutoHandleClose sSession2(serverData2.session);
            AutoHandleClose cSession2(clientData2.session);

            // 送信メソッドの情報のみを使う
            clientData2.session = clientData1.session;

            nn::Result result;
            uintptr_t pc;
            uintptr_t sp;
            uintptr_t param;
            uintptr_t priority;
            uintptr_t idealCore;

            TestHeap heap((DefaultStackSize) * 4);

            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            // Client1
            clientData1.bufferIndex = 0;
            pc = reinterpret_cast<uintptr_t>(ClientThread);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData1);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread1(pc, param, sp, priority, idealCore);
            clientThread1.Start();

            // Client2
            clientData2.bufferIndex = 1;
            pc = reinterpret_cast<uintptr_t>(ClientThread);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&clientData2);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread2(pc, param, sp, priority, idealCore);
            clientThread2.Start();

            g_ReceiveList[0] = serverData1.session;
            g_ReceiveList[1] = nn::svc::INVALID_HANDLE_VALUE;
            serverData1.recvNum = 1;
            serverData1.bufferIndex = 0;

            // Client がデータを Send するまで待つ
            nn::svc::SleepThread(SleepTime);

            // Receive Server Thread
            g_Sequence = 0;
            pc = reinterpret_cast<uintptr_t>(LoopReceiveAndSleepThread);
            sp = stackBottom - DefaultStackSize * 2;
            param = reinterpret_cast<uintptr_t>(&serverData1);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread receiveThread(pc, param, sp, priority, idealCore);
            receiveThread.Start();

            // Wait Server Thread
            serverData2 = serverData1;
            pc = reinterpret_cast<uintptr_t>(WaitThread);
            sp = stackBottom - DefaultStackSize * 3;
            param = reinterpret_cast<uintptr_t>(&serverData2);
            priority = TestLowestThreadPriority - 2;
            idealCore = 0;
            TestThread waitThread(pc, param, sp, priority, idealCore);

            while(g_Sequence != 1)
            {
                nn::svc::SleepThread(SleepTime);
            }

            waitThread.Start();
            waitThread.Wait();
            ASSERT_RESULT_FAILURE_VALUE(serverData2.result, nn::svc::ResultTimeout());
            g_Sequence = 2;

            clientThread1.Wait();
            clientThread2.Wait();

            ASSERT_TRUE(clientData1.result.IsSuccess() && clientData2.result.IsSuccess());
            result = nn::svc::CancelSynchronization(receiveThread.GetHandle());
            ASSERT_RESULT_SUCCESS(result);

            receiveThread.Wait();

            ASSERT_RESULT_FAILURE_VALUE(serverData1.result, nn::svc::ResultCancelled());
        }
    }
}

// TEST 901-38
// 複数のクライアントスレッドから同じセッションを使って要求を複数送信し、
// 要求がサーバーに受信される前にサーバーセッションが閉じられる
TEST(IpcSequence, ServerSessionClose)
{
    nn::Result result;

    IpcTestCondition cond1;

    while(!cond1.IsEnd())
    {
        ServerData serverData1;
        ClientData clientData1;
        cond1.NextTest(&serverData1, &clientData1);
        AutoHandleClose sSession1(serverData1.session);
        AutoHandleClose cSession1(clientData1.session);

        IpcTestCondition cond2;
        while(!cond2.IsEnd())
        {
            ServerData serverData2; // Session を作るために、用意
            ClientData clientData2;
            cond2.NextTest(&serverData2, &clientData2);
            AutoHandleClose sSession2(serverData2.session);
            AutoHandleClose cSession2(clientData2.session);

            // 送信メソッドの情報のみを使う
            // serverData2 のセッションを使う
            clientData1.session = clientData2.session;

            nn::Result result;
            uintptr_t pc;
            uintptr_t sp;
            uintptr_t param;
            uintptr_t priority;
            uintptr_t idealCore;

            TestHeap heap((DefaultStackSize) * 4);

            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            // Client1
            clientData1.bufferIndex = 0;
            pc = reinterpret_cast<uintptr_t>(SendMapThread);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData1);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread1(pc, param, sp, priority, idealCore);
            clientThread1.Start();

            // Client2
            clientData2.bufferIndex = 1;
            pc = reinterpret_cast<uintptr_t>(SendMapThread);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&clientData2);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread2(pc, param, sp, priority, idealCore);
            clientThread2.Start();

            g_ReceiveList[0] = serverData2.session;
            g_ReceiveList[1] = nn::svc::INVALID_HANDLE_VALUE;
            serverData1.recvNum = 1;
            serverData1.bufferIndex = 0;

            // Client がデータを Send するまで待つ
            nn::svc::SleepThread(SleepTime);

            // サーバーセッションを閉じる
            sSession2.Close();

            clientThread1.Wait();
            clientThread2.Wait();
        }
    }
}

// TEST 901-39
// 複数のクライアントスレッドから同じセッションを使って要求を複数送信し、
// １つの要求がサーバーに受信された後にサーバーセッションが閉じられる
TEST(IpcSequence, ServerSessionCloseWithRequest)
{
    nn::Result result;

    IpcTestCondition cond1;

    while(!cond1.IsEnd())
    {
        ServerData serverData1;
        ClientData clientData1;
        cond1.NextTest(&serverData1, &clientData1);
        AutoHandleClose sSession1(serverData1.session);
        AutoHandleClose cSession1(clientData1.session);

        IpcTestCondition cond2;
        while(!cond2.IsEnd())
        {
            ServerData serverData2; // Session を作るために、用意
            ClientData clientData2;
            cond2.NextTest(&serverData2, &clientData2);
            AutoHandleClose sSession2(serverData2.session);
            AutoHandleClose cSession2(clientData2.session);

            // 送信メソッドの情報のみを使う
            // serverData2 のセッションを使う
            clientData1.session = clientData2.session;

            nn::Result result;
            nn::svc::MemoryInfo info;
            uintptr_t pc;
            uintptr_t sp;
            uintptr_t param;
            uintptr_t priority;
            uintptr_t idealCore;

            TestHeap heap((DefaultStackSize) * 2);

            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            // Client1
            clientData1.bufferIndex = 0;
            pc = reinterpret_cast<uintptr_t>(SendMapThread);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData1);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread1(pc, param, sp, priority, idealCore);
            clientThread1.Start();

            // Client2
            clientData2.bufferIndex = 1;
            pc = reinterpret_cast<uintptr_t>(SendMapThread);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&clientData2);
            priority = TestLowestThreadPriority - 1;
            idealCore = 0;
            TestThread clientThread2(pc, param, sp, priority, idealCore);
            clientThread2.Start();

            g_ReceiveList[0] = serverData2.session;
            g_ReceiveList[1] = nn::svc::INVALID_HANDLE_VALUE;
            serverData1.recvNum = 1;
            serverData1.bufferIndex = 0;

            // Client がデータを Send するまで待つ
            nn::svc::SleepThread(SleepTime);

            nn::Bit32* pMsgBuffer = nullptr;
            GetReceiveIpcBuffer(&pMsgBuffer, &serverData1);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();

            int32_t index;
            TestReceiveRequest(&index, &serverData1);
            NN_ASSERT_RESULT_SUCCESS(serverData1.result);

            uintptr_t addr;
            CheckOnlyMapSendData(&addr, MapSendSize, SendMapSendData, pMsgBuffer);

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Ipc);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_Read);
            ASSERT_TRUE(info.attribute == 0);

            // サーバーセッションを閉じる
            sSession2.Close();

            GetMemoryInfo(&info, addr);
            ASSERT_TRUE(info.state == nn::svc::MemoryState_Free);
            ASSERT_TRUE(info.permission == nn::svc::MemoryPermission_None);
            ASSERT_TRUE(info.attribute == 0);

            ipcMsg.SetNull();
            TestReplyRequest(g_ReceiveList[index], &serverData1);

            clientThread1.Wait();
            clientThread2.Wait();
        }
    }
}

// Wait 後に timeout=0 で受信できる
TEST(IpcSequence, WaitAndReceive)
{
    IpcTestCondition cond;

    while(!cond.IsEnd())
    {
        ServerData receiveServerData;
        ClientData clientData;

        cond.NextTest(&receiveServerData, &clientData);
        AutoHandleClose sSession(receiveServerData.session);
        AutoHandleClose cSession(clientData.session);

        nn::Result result;
        uintptr_t pc;
        uintptr_t sp;
        uintptr_t param;
        uintptr_t priority;
        uintptr_t idealCore;

        TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * 3);

        uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

        // Client
        clientData.bufferIndex = 0;
        pc = reinterpret_cast<uintptr_t>(ClientThread);
        sp = stackBottom;
        param = reinterpret_cast<uintptr_t>(&clientData);
        priority = TestLowestThreadPriority;
        idealCore = 0;
        TestThread clientThread(pc, param, sp, priority, idealCore);
        clientThread.Start();

        receiveServerData.recvNum = 1;
        receiveServerData.bufferIndex = 0;
        g_ReceiveList[0] = receiveServerData.session;

        // Client からのシグナルを待つ
        int32_t index;
        result = nn::svc::WaitSynchronization(&index, &receiveServerData.session, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        // timeout=0 で受信できる
        {
            ServerData* data = &receiveServerData;
            nn::Bit32* pMsgBuffer = nullptr;
            GetReceiveIpcBuffer(&pMsgBuffer, data);

            nn::svc::ipc::MessageBuffer ipcMsg(pMsgBuffer);
            ipcMsg.SetNull();

            int32_t index;
            TestReceiveRequest(&index, data);
            NN_ASSERT_RESULT_SUCCESS(data->result);

            ipcMsg.SetNull();
            TestReplyRequest(g_ReceiveList[index], data);
            ASSERT_RESULT_FAILURE_VALUE(data->result, nn::svc::ResultTimeout());
        }

        clientThread.Wait();
        ASSERT_RESULT_SUCCESS(clientData.result);
    }
}

// クライアントが異常な要求を送った後に、正常な要求を送る
TEST(IpcSequence, ErrorHandling)
{
    nn::Result result;
    IpcTestCondition cond;

    while(!cond.IsEnd())
    {
        ServerData receiveServerData;
        ClientData clientData;

        cond.NextTest(&receiveServerData, &clientData);
        AutoHandleClose sSessionCloser(receiveServerData.session);
        AutoHandleClose cSessionCloser(clientData.session);

        for (int i = 0; i < sizeof(DataTypeArray) / sizeof(DataTypeArray[0]); i++)
        {
#ifndef SUPPORT_POINTER_AT_FAIL_IPC
            if (i == DataType_PointerWithIpc ||
                    i == DataType_PointerWithOne ||
                    i == DataType_PointerWithMulti ||
                    i == DataType_All)
            {
                continue;
            }
#endif // SUPPORT_POINTER_AT_FAIL_IPC
            uintptr_t pc;
            uintptr_t sp;
            uintptr_t param;
            uintptr_t priority;
            uintptr_t idealCore;

            TestHeap heap((sizeof(nn::svc::Handle) + DefaultStackSize) * 3);

            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            // Client
            clientData.bufferIndex = 0;
            clientData.dataType = DataTypeArray[i];
            pc = reinterpret_cast<uintptr_t>(TestErrorHandlingClient);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread(pc, param, sp, priority, idealCore);
            clientThread.Start();

            receiveServerData.recvNum = 1;
            receiveServerData.bufferIndex = 0;
            g_ReceiveList[0] = receiveServerData.session;

            // Server
            receiveServerData.dataType = DataTypeArray[i];
            pc = reinterpret_cast<uintptr_t>(TestReceiveDataServer);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&receiveServerData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread serverThread(pc, param, sp, priority, idealCore);
            serverThread.Start();

            clientThread.Wait();
            serverThread.Wait();

            ASSERT_RESULT_SUCCESS(clientData.result);
            ASSERT_RESULT_FAILURE_VALUE(receiveServerData.result, nn::svc::ResultTimeout());
        }

        result = nn::svc::CloseHandle(clientData.session);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(receiveServerData.session);
        ASSERT_RESULT_SUCCESS(result);
    }
}

// クライアントセッションを作成前にサーバーポートがクローズ
TEST(IpcSequence, CloseServerPortBeforeCreatingClientSession)
{
    nn::Result result;

    // Normal Port
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cPortCloser(clientPort);

        result = nn::svc::CloseHandle(serverPort);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle clientSession;
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE(result);
    }

    // Light Port
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cPortCloser(clientPort);

        result = nn::svc::CloseHandle(serverPort);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle clientSession;
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_FAILURE(result);
    }
}

// クライアントセッションを作成後にサーバーポートがクローズ
TEST(IpcSequence, CloseServerPortAfterCreatingClientSession)
{
    nn::Result result;

    // Normal Port
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cPortCloser(clientPort);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToPort(&clientSession, clientPort);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            clientData.doCheckAsyncFailure = false;

            nn::Bit32* pMsgBuffer = nullptr;
            GetSendIpcBuffer(&pMsgBuffer, &clientData);

            nn::svc::Handle handle;
            MakeClientData(&handle, pMsgBuffer, clientData.dataType);

            TestSendRequest(&clientData);

            ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());

            FinalizeClientData(&clientData, handle);
        }
    }

    // Named Port
    // Port は閉じるけど、ManageNamedPortで名前の解放はしない
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            clientData.doCheckAsyncFailure = false;

            nn::Bit32* pMsgBuffer = nullptr;
            GetSendIpcBuffer(&pMsgBuffer, &clientData);

            nn::svc::Handle handle;
            MakeClientData(&handle, pMsgBuffer, clientData.dataType);

            TestSendRequest(&clientData);

            ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());

            FinalizeClientData(&clientData, handle);

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    // Named Port
    // Port も閉じて、ManageNamedPortで名前の解放も行う
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            clientData.doCheckAsyncFailure = false;

            nn::Bit32* pMsgBuffer = nullptr;
            GetSendIpcBuffer(&pMsgBuffer, &clientData);

            nn::svc::Handle handle;
            MakeClientData(&handle, pMsgBuffer, clientData.dataType);

            TestSendRequest(&clientData);

            ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());

            FinalizeClientData(&clientData, handle);
        }
    }

    // Named Port
    // ManageNamedPortで名前の解放は行うが、Port は閉じない
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            nn::svc::Handle tmpHandle;
            result = nn::svc::ManageNamedPort(&tmpHandle, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle serverSession;
            result = nn::svc::AcceptSession(&serverSession, serverPort);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sSessionCloser(serverSession);

            ServerData receiveServerData;
            receiveServerData.session = serverSession;
            receiveServerData.method = RecvMethod_Tls;
            receiveServerData.bufferIndex = 0;
            receiveServerData.recvNum = 1;
            receiveServerData.dataType = DataTypeArray[j];
            g_ReceiveList[0] = serverSession;

            TestHeap heap(DefaultStackSize);
            uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

            uintptr_t pc = reinterpret_cast<uintptr_t>(TestReceiveDataServer);
            uintptr_t sp = stackBottom;
            uintptr_t param = reinterpret_cast<uintptr_t>(&receiveServerData);
            int32_t priority = TestLowestThreadPriority;
            int32_t idealCore = 0;
            TestThread receiveThread(pc, param, sp, priority, idealCore);
            receiveThread.Start();

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            clientData.doCheckAsyncFailure = false;

            nn::Bit32* pMsgBuffer = nullptr;
            GetSendIpcBuffer(&pMsgBuffer, &clientData);

            nn::svc::Handle handle;
            MakeClientData(&handle, pMsgBuffer, clientData.dataType);

            TestSendRequest(&clientData);

            ASSERT_RESULT_SUCCESS(clientData.result);

            FinalizeClientData(&clientData, handle);

            receiveThread.Wait();

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    // Light Port
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle clientSession;
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::CloseHandle(serverPort);
        ASSERT_RESULT_SUCCESS(result);

        result = ClientSendLightRequest(clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultSessionClosed());
    }
} // NOLINT(impl/function_size)

// クライアントセッションが通信中にサーバーポートがクローズ
TEST(IpcSequence, CloseServerPortWhileSendingRequest)
{
    nn::Result result;
    uintptr_t pc;
    uintptr_t sp;
    uintptr_t param;
    uintptr_t priority;
    uintptr_t idealCore;

    TestHeap heap(DefaultStackSize * 2);
    uintptr_t stackBottom = reinterpret_cast<uintptr_t>(heap.GetAddress() + heap.GetSize());

    // Normal Port
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;
            result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cPortCloser(clientPort);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToPort(&clientSession, clientPort);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                clientData.doCheckAsyncFailure = true;
            }
            else
            {
                clientData.doCheckAsyncFailure = false;
            }

            pc = reinterpret_cast<uintptr_t>(TestSendDataClient);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread(pc, param, sp, priority, idealCore);
            clientThread.Start();

            nn::svc::SleepThread(SleepTime);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            clientThread.Wait();

            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                ASSERT_RESULT_SUCCESS(clientData.result);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());
            }
        }
    }

    // Named Port
    // Port は閉じるけど、ManageNamedPortで名前の解放はしない
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;
            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                clientData.doCheckAsyncFailure = true;
            }
            else
            {
                clientData.doCheckAsyncFailure = false;
            }


            pc = reinterpret_cast<uintptr_t>(TestSendDataClient);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread(pc, param, sp, priority, idealCore);
            clientThread.Start();

            nn::svc::SleepThread(SleepTime);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            clientThread.Wait();

            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                ASSERT_RESULT_SUCCESS(clientData.result);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());
            }

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    // Named Port
    // Port も閉じて、ManageNamedPortで名前の解放も行う
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                clientData.doCheckAsyncFailure = true;
            }
            else
            {
                clientData.doCheckAsyncFailure = false;
            }

            pc = reinterpret_cast<uintptr_t>(ClientThread);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&clientData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread(pc, param, sp, priority, idealCore);
            clientThread.Start();

            nn::svc::SleepThread(SleepTime);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);

            clientThread.Wait();

            if (clientData.method == SendMethod_AsyncUserBuffer)
            {
                ASSERT_RESULT_SUCCESS(clientData.result);
            }
            else
            {
                ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());
            }
        }
    }

    // Named Port
    // ManageNamedPortで名前の解放は行うが、Port は閉じない
    for (int i = 0; i < SendMethodArraySize; i++)
    {
        for (int j = 0; j < DataTypeArraySize; j++)
        {
            nn::svc::Handle serverPort;
            nn::svc::Handle clientPort;

            result = nn::svc::ManageNamedPort(&serverPort, PortName, 1);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle clientSession;
            result = nn::svc::ConnectToNamedPort(&clientSession, PortName);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose cSessionCloser(clientSession);

            ClientData clientData;
            clientData.session = clientSession;
            clientData.bufferIndex = 0;
            clientData.method = SendMethodArray[i];
            clientData.dataType = DataTypeArray[j];
            clientData.doCheckAsyncFailure = false;

            pc = reinterpret_cast<uintptr_t>(TestSendDataClient);
            sp = stackBottom - DefaultStackSize;
            param = reinterpret_cast<uintptr_t>(&clientData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread clientThread(pc, param, sp, priority, idealCore);
            clientThread.Start();

            nn::svc::Handle tmpHandle;
            result = nn::svc::ManageNamedPort(&tmpHandle, PortName, 0);
            ASSERT_RESULT_SUCCESS(result);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
            ASSERT_RESULT_SUCCESS(result);

            nn::svc::Handle serverSession;
            result = nn::svc::AcceptSession(&serverSession, serverPort);
            ASSERT_RESULT_SUCCESS(result);
            AutoHandleClose sSessionCloser(serverSession);

            ServerData receiveServerData;
            receiveServerData.session = serverSession;
            receiveServerData.method = RecvMethod_Tls;
            receiveServerData.bufferIndex = 0;
            receiveServerData.recvNum = 1;
            receiveServerData.dataType = DataTypeArray[j];
            g_ReceiveList[0] = serverSession;

            pc = reinterpret_cast<uintptr_t>(TestReceiveDataServer);
            sp = stackBottom;
            param = reinterpret_cast<uintptr_t>(&receiveServerData);
            priority = TestLowestThreadPriority;
            idealCore = 0;
            TestThread receiveThread(pc, param, sp, priority, idealCore);
            receiveThread.Start();

            clientThread.Wait();
            receiveThread.Wait();

            ASSERT_RESULT_SUCCESS(clientData.result);

            result = nn::svc::CloseHandle(serverPort);
            ASSERT_RESULT_SUCCESS(result);
        }
    }

    // Light Port
    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;
        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, true, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cPortCloser(clientPort);

        nn::svc::Handle clientSession;
        result = nn::svc::ConnectToPort(&clientSession, clientPort);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose cSessionCloser(clientSession);

        ClientData clientData;
        clientData.session = clientSession;

        pc = reinterpret_cast<uintptr_t>(LightClientThread);
        sp = stackBottom;
        param = reinterpret_cast<uintptr_t>(&clientData);
        priority = TestLowestThreadPriority;
        idealCore = 0;
        TestThread clientThread(pc, param, sp, priority, idealCore);
        clientThread.Start();

        nn::svc::SleepThread(SleepTime);

        result = nn::svc::CloseHandle(serverPort);
        ASSERT_RESULT_SUCCESS(result);

        clientThread.Wait();

        ASSERT_RESULT_FAILURE_VALUE(clientData.result, nn::svc::ResultSessionClosed());
    }
} // NOLINT(impl/function_size)
