﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Result.h>

namespace {
char g_Buffer[DefaultStackSize] __attribute__((aligned(0x1000)));

const int NumTest0 = 1000;
const int NumTest1 = 100;
int g_Sequence;
struct TestData
{
    nn::svc::Handle writableEvent;
    nn::svc::Handle readableEvent;
};

void TestAutoEvent(const int x, nn::svc::Handle readableEvent, nn::svc::Handle writableEvent)
{
    for (int i = 0; i < NumTest1; i++)
    {
        nn::Result result;
        if (x == 1)
        {
            ASSERT_TRUE(g_Sequence == 2 * i);
            g_Sequence++;
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 0)
        {
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);
            result = nn::svc::ResetSignal(readableEvent);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_Sequence == 2 * i + 1);
            g_Sequence++;
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 1)
        {
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);
            result = nn::svc::ResetSignal(readableEvent);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_Sequence == 2 * i + 2);
        }
    }
}

void TestAutoEventThread(TestData* pData)
{
    AutoThreadExit autoExit;
    TestAutoEvent(0, pData->readableEvent, pData->writableEvent);
}

void TestManualEvent(const int x, nn::svc::Handle readableEvent, nn::svc::Handle writableEvent)
{
    for (int i = 0; i < NumTest1; i++)
    {
        nn::Result result;
        if (x == 1)
        {
            ASSERT_TRUE(g_Sequence == 2 * i);
            g_Sequence++;
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 0)
        {
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);
            result = nn::svc::ClearEvent(readableEvent);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_Sequence == 2 * i + 1);
            g_Sequence++;
            result = nn::svc::SignalEvent(writableEvent);
            ASSERT_RESULT_SUCCESS(result);
        }
        if (x == 1)
        {
            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &readableEvent, 1, -1);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(index == 0);
            result = nn::svc::ClearEvent(readableEvent);
            ASSERT_RESULT_SUCCESS(result);
            ASSERT_TRUE(g_Sequence == 2 * i + 2);
        }
    }
}
void TestManualEventThread(TestData* pData)
{
    AutoThreadExit autoExit;
    TestManualEvent(0, pData->readableEvent, pData->writableEvent);
}
}

/* TEST 15-1 (同じコア) 15-3 (違うコア) */
/* resetType = ResetType_Auto の時にスレッド間でシグナルの送受信が出来ることを確認*/
TEST(SignalEvent, TestAuto0)
{
    TestEventLeak leakTest;
    for (int j = 0; j < NumTest0; j++)
    {
        nn::Result result;
        nn::svc::Handle writableEvent0;
        nn::svc::Handle readableEvent0;
        result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        nn::svc::Handle writableEvent1;
        nn::svc::Handle readableEvent1;
        result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        TestData data;
        data.readableEvent = readableEvent0;
        data.writableEvent = writableEvent1;

        for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
        {
            uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
            uintptr_t pc = reinterpret_cast<uintptr_t>(TestAutoEventThread);
            nn::svc::Handle handle;

            result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&data), sp, 32, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            g_Sequence = 0;
            TestAutoEvent(1, readableEvent1, writableEvent0);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
        result = nn::svc::CloseHandle(writableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(writableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
    }
}

/* TEST 15-2 (同じコア) 15-4 (違うコア) */
/* resetType = ResetType_Manual の時にスレッド間でシグナルの送受信が出来ることを確認*/
TEST(SignalEvent, TestManual0)
{
    TestEventLeak leakTest;
    for (int j = 0; j < NumTest0; j++)
    {
        nn::Result result;
        nn::svc::Handle writableEvent0;
        nn::svc::Handle readableEvent0;
        result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        nn::svc::Handle writableEvent1;
        nn::svc::Handle readableEvent1;
        result = nn::svc::CreateEvent(&writableEvent1, &readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        TestData data;
        data.readableEvent = readableEvent0;
        data.writableEvent = writableEvent1;

        for (int32_t idealCore = 0; idealCore < NumCore; idealCore++)
        {
            uintptr_t sp = reinterpret_cast<uintptr_t>(g_Buffer + sizeof(g_Buffer));
            uintptr_t pc = reinterpret_cast<uintptr_t>(TestManualEventThread);
            nn::svc::Handle handle;

            result = nn::svc::CreateThread(&handle, pc, reinterpret_cast<uintptr_t>(&data), sp, 32, idealCore);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::StartThread(handle);
            ASSERT_RESULT_SUCCESS(result);

            g_Sequence = 0;
            TestManualEvent(1, readableEvent1, writableEvent0);

            int32_t index;
            result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
            ASSERT_RESULT_SUCCESS(result);

            result = nn::svc::CloseHandle(handle);
            ASSERT_RESULT_SUCCESS(result);
        }
        result = nn::svc::CloseHandle(writableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readableEvent0);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(writableEvent1);
        ASSERT_RESULT_SUCCESS(result);
        result = nn::svc::CloseHandle(readableEvent1);
        ASSERT_RESULT_SUCCESS(result);
    }
}

TEST(SignalEvent, TestManual1)
{
    TestEventLeak leakTest;
    nn::Result result;
    nn::svc::Handle writableEvent0;
    nn::svc::Handle readableEvent0;
    result = nn::svc::CreateEvent(&writableEvent0, &readableEvent0);
    int32_t index;

    // TEST 15-7
    result = nn::svc::SignalEvent(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &readableEvent0, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // WaitでClearされていない
    result = nn::svc::WaitSynchronization(&index, &readableEvent0, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // TEST 16-1
    // Readable Event クリア
    result = nn::svc::ClearEvent(readableEvent0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &readableEvent0, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::SignalEvent(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &readableEvent0, 1, 0);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(index == 0);

    // TEST 16-2
    // Wrirable Event クリア
    result = nn::svc::ClearEvent(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &readableEvent0, 1, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    // Wrirable Event の Wait
    result = nn::svc::WaitSynchronization(&index, &writableEvent0, 1, -1);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 15-8
    // Readable Event の Signal
    result = nn::svc::SignalEvent(readableEvent0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    result = nn::svc::CloseHandle(writableEvent0);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(readableEvent0);
    ASSERT_RESULT_SUCCESS(result);
}

