﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_DebugProcess.h"


namespace {

const int64_t SleepTime = 100 * 1000 * 1000;

void InitDebugProcess(nn::svc::Handle *pOut, TestProcess* process)
{
    SCOPED_TRACE("InitDebugProcess");

    nn::Result result;
    nn::Bit64 pid;
    nn::svc::Handle processHandle = process->GetHandle();
    result = nn::svc::GetProcessId(&pid, processHandle);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Initializing));

    result = nn::svc::DebugActiveProcess(pOut, pid);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_PreAttached));

    process->Start();

    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Breaked));

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, *pOut);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_CreateProcess);

    WaitDebugEvent(&debugInfo, *pOut);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_AttachBreak);

    result = ContinueDebugProcess(*pOut);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(CheckProcessState(processHandle, nn::svc::ProcessState_Attached));
}
} // namespace

TEST(DebugProcess, GeneralRegister)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3a00001, // mov     r0, #1
        0xe3a01001, // mov     r0, #1
        // label:
        0xe1500001, // cmp     r0, r1
        0x0afffffd, // beq     label
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc    NN_SVC_ID_EXIT_PROCESS
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debug;
    InitDebugProcess(&debug, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;
    nn::svc::DebugInfoCreateThread createThreadInfo;

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_CreateThread);
    createThreadInfo = debugInfo.info.createThread;
    ASSERT_TRUE(process.GetCodeAddress() == createThreadInfo.entryAddress);
    nn::Bit64 tid = createThreadInfo.id;

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &debug, 1, SleepTime);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    // レジスタの値を読む
    nn::svc::ThreadContext context;
    nn::svc::ContextFlag contextFlag = nn::svc::ContextFlag_General;
    result = nn::svc::GetDebugThreadContext(&context, debug, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(context.r[0] == 1);
    ASSERT_TRUE(context.r[1] == 1);

    // レジスタの値を設定する
    context.r[1] = 0;
    result = nn::svc::SetDebugThreadContext(debug, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debug, 1);

    process.Wait();
    process.Close();
}

TEST(DebugProcess, ControlRegister)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3a00001, // mov     r0, #1
        0xe3a01001, // mov     r0, #1
        0xe1500001, // cmp     r0, r1
        // label:
        0x0afffffe, // b       label
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc    NN_SVC_ID_EXIT_PROCESS
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debug;
    InitDebugProcess(&debug, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_CreateThread);

    nn::svc::DebugInfoCreateThread createThreadInfo;
    createThreadInfo = debugInfo.info.createThread;
    ASSERT_TRUE(process.GetCodeAddress() == createThreadInfo.entryAddress);
    nn::Bit64 tid = createThreadInfo.id;

    nn::svc::ThreadContext context;
    nn::svc::ContextFlag contextFlag = nn::svc::ContextFlag_Control;
    result = nn::svc::GetDebugThreadContext(&context, debug, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(context.pc == process.GetCodeAddress());
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    ASSERT_TRUE((context.pstate & (1 << 30)) == 0);
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    ASSERT_TRUE((context.cpsr & (1 << 30)) == 0);
#endif

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &debug, 1, SleepTime);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    // レジスタの値を読む
    contextFlag = nn::svc::ContextFlag_Control;
    result = nn::svc::GetDebugThreadContext(&context, debug, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(context.pc == process.GetCodeAddress() + 0xC);
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    ASSERT_TRUE((context.pstate & (1 << 30)) > 1);
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    ASSERT_TRUE((context.cpsr & (1 << 30)) > 1);
#endif

    // レジスタの値を設定する
    context.pc = process.GetCodeAddress() + 0x10;
    result = nn::svc::SetDebugThreadContext(debug, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debug, 1);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

TEST(DebugProcess, FpuRegister)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3a005fe, // mov     r0, #0x3f800000
        0xe50d0004, // str     r0, [sp, #-4]
        0xed1d0a01, // vldr    s0, [sp, #-4]
        0xed1d1a01, // vldr    s1, [sp, #-4]
        // <label>:
        0xeeb40a41, // vcmp.f32       s0, s2
        0xeef1fa10, // vmrs    APSR_nzcv, fpscr
        0x0afffffc, // beq     <label>
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc    NN_SVC_ID_EXIT_PROCESS
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debug;
    InitDebugProcess(&debug, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_CreateThread);

    nn::svc::DebugInfoCreateThread createThreadInfo;
    createThreadInfo = debugInfo.info.createThread;
    ASSERT_TRUE(process.GetCodeAddress() == createThreadInfo.entryAddress);
    nn::Bit64 tid = createThreadInfo.id;

    nn::svc::ThreadContext context;
    nn::svc::ContextFlag contextFlag;

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &debug, 1, SleepTime);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    // レジスタの値を読む
    contextFlag = nn::svc::ContextFlag_Fpu;
    result = nn::svc::GetDebugThreadContext(&context, debug, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    nn::Bit32* fpu32 = reinterpret_cast<nn::Bit32*>(&context.v[0]);
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    nn::Bit32* fpu32 = reinterpret_cast<nn::Bit32*>(&context.fpuRegisters[0]);
#endif
    ASSERT_TRUE(fpu32[0] == 0x3f800000);
    ASSERT_TRUE(fpu32[2] == 0x3f800000);

    // レジスタの値を設定する
    fpu32[0] = 0x40000000;

    result = nn::svc::SetDebugThreadContext(debug, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debug, 1);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

TEST(DebugProcess, FpuControlRegister)
{
    TestProcessLeak testProcessLeak;
    nn::Result result;
    int32_t index;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    TestProcess process(1, 0, flags, DefaultCapabilityFlagNum + 1);
    nn::svc::Handle processHandle = process.GetHandle();

    uint32_t codes[] = {
        0xe3a005fe, // mov     r0, #0x3f800000
        0xe50d0004, // str     r0, [sp, #-4]
        0xed1d0a01, // vldr    s0, [sp, #-4]
        0xed1d1a01, // vldr    s1, [sp, #-4]
        0xeeb40a41, // vcmp.f32       s0, s2
        // <label>:
        0xeef1fa10, // vmrs    APSR_nzcv, fpscr
        0x0afffffd, // beq     <label>
        (0xef000000 | NN_SVC_ID_EXIT_PROCESS), // svc    NN_SVC_ID_EXIT_PROCESS
    };

    SetExecuteCode(
            processHandle, codes, sizeof(codes),
            process.GetCodeAddress(), process.GetCodeAreaSize());

    nn::svc::Handle debug;
    InitDebugProcess(&debug, &process);

    nn::svc::DebugEventInfo debugInfo;
    nn::svc::DebugInfoException except;

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_CreateThread);

    nn::svc::DebugInfoCreateThread createThreadInfo;
    createThreadInfo = debugInfo.info.createThread;
    ASSERT_TRUE(process.GetCodeAddress() == createThreadInfo.entryAddress);
    nn::Bit64 tid = createThreadInfo.id;

    nn::svc::ThreadContext context;
    nn::svc::ContextFlag contextFlag;

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::WaitSynchronization(&index, &debug, 1, SleepTime);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultTimeout());

    result = nn::svc::BreakDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    WaitDebugEvent(&debugInfo, debug);
    ASSERT_TRUE(debugInfo.event == nn::svc::DebugEvent_Exception);
    except = debugInfo.info.exception;
    ASSERT_TRUE(except.exceptionCode == nn::svc::DebugException_DebuggerBreak);

    // レジスタの値を読む
    contextFlag = nn::svc::ContextFlag_FpuControl;
    result = nn::svc::GetDebugThreadContext(&context, debug, tid, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    nn::Bit32 fpsr = context.fpsr;
    ASSERT_TRUE((fpsr & (1 << 30)) > 0);

    // レジスタの値を設定する
    fpsr &= ~(1 << 30);
    context.fpsr = fpsr;
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    nn::Bit32 fpscr = context.fpscr;
    ASSERT_TRUE((fpscr & (1 << 30)) > 0);

    // レジスタの値を設定する
    fpscr &= ~(1 << 30);
    context.fpscr = fpscr;
#endif
    result = nn::svc::SetDebugThreadContext(debug, tid, context, contextFlag);
    ASSERT_RESULT_SUCCESS(result);

    result = ContinueDebugProcess(debug);
    ASSERT_RESULT_SUCCESS(result);

    ExitDebugProcess(debug, 1);

    result = nn::svc::WaitSynchronization(&index, &processHandle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    process.Close();
}

