﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestMemory.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

extern "C" void nnMain();

namespace {
#ifdef INVALID_POINTER_TEST
const char ConstHandle[sizeof(nn::svc::Handle)] = {0};
#endif
char g_Buffer[0x1000] __attribute__((aligned(0x1000)));
char g_TestBuffer[0x1000] __attribute__((aligned(0x1000)));

void CheckSuccessMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::Handle handle;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_None;

    size_t size = 0x1000;

    for (int i = 0; i < numState; i++)
    {
        SCOPED_TRACE("Success Memory Area");
        ASSERT_TRUE(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0));

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::CreateTransferMemory(&handle, addr, size, perm);
        ASSERT_RESULT_SUCCESS(result);

        CheckMemory(
                addr,
                array[i]->GetState(),
                perm,
                nn::svc::MemoryAttribute_Locked);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}

void CheckInvalidMemoryArea(TestMemoryInfo** array, int numState)
{
    nn::Result result;

    nn::svc::Handle handle;
    nn::svc::MemoryPermission perm = nn::svc::MemoryPermission_None;

    size_t size = 0x1000;

    for (int i = 0; i < numState; i++)
    {
        if (!(array[i]->GetSize() > 0 && ((array[i]->GetSize() & 0xfff) == 0)))
        {
            array[i]->Close();
            continue;
        }

        SCOPED_TRACE("Invalid Memory Area");

        array[i]->Initialize();
        uintptr_t addr = array[i]->GetAddress();

        result = nn::svc::CreateTransferMemory(&handle, addr, size, perm);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        array[i]->CheckDefaultState();

        array[i]->Close();
    }
}


} // namespace

TEST(CreateTransferMemory, pOutTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(g_Buffer);
    size_t size = 0x1000;
    nn::svc::MemoryInfo blockInfo;

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 126-1
    // MemoryPermission_ReadWrite の領域を受け付ける

    // ローカル変数
    nn::svc::Handle handle;

    result = nn::svc::CreateTransferMemory(
            &handle, addr, size, nn::svc::MemoryPermission_Read);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_Read);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // ヒープ変数
    TestHeap heap(HeapAlign);
    ASSERT_TRUE((heap.GetAddress() & (0xfff)) == 0);
    nn::svc::Handle* pHandle = reinterpret_cast<nn::svc::Handle*>(heap.GetAddress());
    result = nn::svc::CreateTransferMemory(
            pHandle, addr, size, nn::svc::MemoryPermission_None);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
    ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

    result = nn::svc::CloseHandle(*pHandle);
    ASSERT_RESULT_SUCCESS(result);

    GetMemoryInfo(&blockInfo, addr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_CodeData);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

#ifdef INVALID_POINTER_TEST
    // TEST 126-2
    // NULL を受け付けない
    pHandle = NULL;
    result = nn::svc::CreateTransferMemory(
            pHandle, addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 126-3
    // MemoryPermission_ReadWrite の領域を指し示していると失敗する
    pHandle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::CreateTransferMemory(
            pHandle, addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 126-4
    // MemoryPermission_Read の領域を指し示していると失敗する
    uintptr_t tmpAddr = reinterpret_cast<uintptr_t>(&ConstHandle);
    pHandle = reinterpret_cast<nn::svc::Handle*>(tmpAddr);
    result = nn::svc::CreateTransferMemory(
            pHandle, addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 126-5
    // MemoryPermission_ReadExecute の領域を指し示していると失敗する
    pHandle = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::CreateTransferMemory(
            pHandle, addr, size, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

}

TEST(CreateTransferMemory, AddrTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    uintptr_t heapAddr;
    size_t heapSize = 0x01000000; // 16MB
    TestHeap heap(heapSize);
    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    ASSERT_TRUE((heapAddr & (0xfff)) == 0);

    // TEST 126-8
    // 4KB にアライメントされている値を受け付ける
    for (uintptr_t diff = 0; diff < heapSize; diff += 0x1000)
    {
        GetMemoryInfo(&blockInfo, heapAddr + diff);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::CreateTransferMemory(
                &handle, heapAddr + diff, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr + diff);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr + diff);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }

    // TEST 126-9
    // 4KB にアライメントされていないと失敗する
    for (uintptr_t diff = 1; diff < 0x1000; diff++)
    {
        GetMemoryInfo(&blockInfo, heapAddr + diff);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);

        result = nn::svc::CreateTransferMemory(
                &handle, heapAddr + diff, 0x1000, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

        GetMemoryInfo(&blockInfo, heapAddr + diff);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
    }
}

TEST(CreateTransferMemory, SizeTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::MemoryInfo blockInfo;

    uintptr_t heapAddr;
    size_t heapSize = 0x01000000; // 16MB
    TestHeap heap(heapSize);
    heapAddr = heap.GetAddress();
    ASSERT_TRUE(heapSize == heap.GetSize());
    ASSERT_TRUE((heapAddr & (0xfff)) == 0);

    // TEST 126-8
    // 4KB にアライメントされている値を受け付ける
    for (size_t size = 0x1000; size < heapSize; size += 0x1000)
    {
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= heapAddr + size);

        result = nn::svc::CreateTransferMemory(
                &handle, heapAddr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == nn::svc::MemoryAttribute_Locked);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= heapAddr + size);

        result = nn::svc::CloseHandle(handle);
        ASSERT_RESULT_SUCCESS(result);

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= heapAddr + size);
    }

    GetMemoryInfo(&blockInfo, heapAddr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 126-10
    // 0 を受け付けない
    result = nn::svc::CreateTransferMemory(
            &handle, heapAddr, 0, nn::svc::MemoryPermission_ReadWrite);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

    GetMemoryInfo(&blockInfo, heapAddr);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
    ASSERT_TRUE(blockInfo.attribute == 0);

    // TEST 126-12
    // 4KB にアライメントされていないと失敗する
    for (size_t size = 1; size < 0x1000; size++)
    {
        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= heapAddr + size);

        result = nn::svc::CreateTransferMemory(
                &handle, heapAddr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidSize());

        GetMemoryInfo(&blockInfo, heapAddr);
        ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Normal);
        ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_ReadWrite);
        ASSERT_TRUE(blockInfo.attribute == 0);
        ASSERT_TRUE(blockInfo.baseAddress <= heapAddr);
        ASSERT_TRUE(blockInfo.baseAddress + blockInfo.size >= heapAddr + size);
    }
}

TEST(CreateTransferMemory, MemoryAreaTest)
{
    TestTransferMemoryLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    uintptr_t addr;
    size_t size = 0x1000;

    TestMemoryInfo* header;
    GenerateMemoryStateList(
            &header, reinterpret_cast<uintptr_t>(g_Buffer), sizeof(g_Buffer));

    nn::svc::MemoryState successState[] = {
        nn::svc::MemoryState_Normal,
        nn::svc::MemoryState_CodeData,
#ifdef SUPPORT_ALIAS_CODE_DATA
        nn::svc::MemoryState_AliasCodeData,
#endif
    };
    const int NumSuccessState = sizeof(successState) / sizeof(nn::svc::MemoryState);

    // 許可されたメモリ状態の領域を受け付ける
    TestMemoryInfo* successArea[NumSuccessState];
    header->GetTestListWithStates(successArea, successState, NumSuccessState);

    CheckSuccessMemoryArea(successArea, NumSuccessState);

    // 許可されていないメモリ状態の領域を受け付けない
    const int NumInvalid = NumTestMemoryState - NumSuccessState;
    TestMemoryInfo* invalidArea[NumInvalid];
    header->GetTestListExceptStates(invalidArea, NumInvalid, successState, NumSuccessState);

    CheckInvalidMemoryArea(invalidArea, NumInvalid);

    uint32_t successAttr[] = {
        0,
    };
    const int NumSuccessAttr = sizeof(successAttr) / sizeof(nn::svc::MemoryAttribute);
    TestMemoryInfo* successAttrArea[NumSuccessAttr];

    GenerateMemoryAttributeList(
            &header, reinterpret_cast<uintptr_t>(g_TestBuffer), sizeof(g_TestBuffer));
    header->GetTestListWithAttributes(successAttrArea, successAttr, NumSuccessAttr);
    CheckSuccessMemoryArea(successAttrArea, NumSuccessAttr);

    const int NumInvalidAttrs = NumTestMemoryAttribute - NumSuccessAttr;
    TestMemoryInfo* invalidAttrs[NumInvalidAttrs];
    header->GetTestListExceptAttributes(invalidAttrs, NumInvalidAttrs, successAttr, NumSuccessAttr);
    CheckInvalidMemoryArea(invalidAttrs, NumInvalidAttrs);

    // TEST 126-29
    // オーバーフローする組み合わせを指定できない
    {
        TestCodeDataMemoryState codeData;
        codeData.Initialize();

        addr = codeData.GetAddress();
        size = static_cast<size_t>(-0x1000);

        result = nn::svc::CreateTransferMemory(
                &handle, addr, size, nn::svc::MemoryPermission_ReadWrite);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

        codeData.CheckDefaultState();

        codeData.Close();
    }

    // メモリ状態が一様でないと失敗する
    {
        TestNormalMemoryState normal;
        normal.Initialize();

        addr = normal.GetAddress();
        size = 0x1000;
        ASSERT_TRUE(size * 3 <= normal.GetSize());

        {
            TestMapMemory map(g_FreeAreaBegin, addr + size, size);

            // 前方
            result = nn::svc::CreateTransferMemory(
                    &handle, addr + size, size * 2, nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());

            // 後方
            result = nn::svc::CreateTransferMemory(
                    &handle, addr, size * 2, nn::svc::MemoryPermission_Read);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidCurrentMemory());
        }

        normal.CheckDefaultState();

        normal.Close();
    }
} // NOLINT(readability/fn_size)

