﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include "util_TestProcess.h"
#include "util_TestLoader.h"
#include "util_TestIpc.h"
#include "test_ForceDebug.h"
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/ipc/svc_SessionMessage.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include <nn/svc/svc_TcbId.autogen.h>
#include <nn/svc/svc_DdId.autogen.h>
#include <nn/svc/svc_ServerId.autogen.h>
#include <nn/svc/svc_Version.h>
#include <cstring>

extern "C" void nnMain();

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
#include "test_TestTmrDevice_Jetson.h"
#endif


namespace {

} // namespace

#ifdef ENABLE_MAX_HANDLE_TEST
TEST(CreateProcess, MaxHandleTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    {
        ConsumeHandle maxHandle;

        // TEST 121-74
        // ハンドル数が上限に達していると、プロセスを作成できない
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultMaxHandle());
    }

    // TEST 121-75
    // ハンドルが解放されると、プロセスが作成できるようになる
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // ENABLE_MAX_HANDLE_TEST

#ifdef ENABLE_OUT_OF_RESOURCE_TEST
TEST(CreateProcess, OutOfMemoryTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};
    TestResourceEnv resourceEnv;

    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    // ヒープ領域を使い切る
    uintptr_t heapPtr;
    size_t heapSize = 0;
    for(int32_t i = 1;i > 0;i++)
    {
        heapSize = HeapAlign * i;
        result = nn::svc::SetHeapSize(&heapPtr, heapSize);
        if (result.IsFailure())
        {
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultOutOfMemory());
            // 念のため
            heapSize = HeapAlign * (i - 1);
            result = nn::svc::SetHeapSize(&heapPtr, heapSize);
            ASSERT_RESULT_SUCCESS(result);
            break;
        }
    }

    nn::svc::Handle* handles = reinterpret_cast<nn::svc::Handle*>(heapPtr);
    int32_t procNum = 0;
    for (;procNum < static_cast<int32_t>(heapSize / sizeof(nn::svc::Handle)); procNum++)
    {
        result = nn::svc::CreateProcess(&handles[procNum], param, flags, DefaultCapabilityFlagNum);
        if (result.IsFailure())
        {
            // TEST 121-78
            // 物理メモリが上限に達していると、プロセスを作成できない
            ASSERT_TRUE(result <= nn::svc::ResultOutOfMemory()
                    || result <= nn::svc::ResultOutOfResource());
            break;
        }
    }

    for (int32_t i = 0; i < procNum; i++)
    {
        result = nn::svc::CloseHandle(handles[i]);
        ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::svc::SetHeapSize(&heapPtr, 0);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-79
    // 物理メモリの領域に空きが出来ると、プロセスが作成できるようになる
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}
#endif // ENABLE_OUT_OF_RESOURCE_TEST

// TEST 121-80
// 異常なパラメータを指定して失敗した後に、正常なパラメータを与えてもう一度呼び出すと成功する
TEST(CreateProcess, ReCallTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    nn::svc::CreateProcessParameter param = {};

    SetDefaultParam(&param);
    param.memoryAddress = 1;

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidAddress());

    SetDefaultParam(&param);
    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::CloseHandle(handle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(CreateProcess, EnableJitTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int64_t info;
    uint32_t codes[] = {
        0xe3a00000, // mov r0, #0
        0xe5803000, // str r3, [r0] ; ここでNULL アクセス
        0xef000000 | NN_SVC_ID_EXIT_PROCESS, // ここにはこないはず
    };

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    param.flags = nn::svc::CreateProcessParameterFlag_EnableJitDebug;

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose processCloser(handle);

    SetExecuteCode(
            handle, codes, sizeof(codes), param.memoryAddress, param.memoryNumPages * 0x1000);

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = g_ProcessIdealCore;
    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);
    nn::svc::ResetSignal(handle);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &handle, 1, -1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(nn::svc::ProcessState_WaitAttach == static_cast<nn::svc::ProcessState>(info));

    nn::Bit64 pid;
    result = nn::svc::GetProcessId(&pid, handle);
    ASSERT_RESULT_SUCCESS(result);

    // TEST 121-105
    // EnableJitDebug を指定したときに、JIT Debug が可能になる
    nn::svc::Handle debugHandle;
    result = nn::svc::DebugActiveProcess(&debugHandle, pid);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose debugCloser(debugHandle);

    result = nn::svc::TerminateDebugProcess(debugHandle);
    ASSERT_RESULT_SUCCESS(result);
}

TEST(CreateProcess, DisableJit)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    int64_t info;
    uint32_t codes[] = {
        0xe3a00000, // mov r0, #0
        0xe5803000, // str r3, [r0] ; ここでNULL アクセス
        0xef000000 | NN_SVC_ID_EXIT_PROCESS, // ここにはこないはず
    };

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0x1);

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    param.flags = 0;

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + 1);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose processCloser(handle);

    SetExecuteCode(
            handle, codes, sizeof(codes), param.memoryAddress, param.memoryNumPages * 0x1000);

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = g_ProcessIdealCore;
    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);
    nn::svc::ResetSignal(handle);

    WaitProcess(handle);

    nn::Bit64 pid;
    result = nn::svc::GetProcessId(&pid, handle);
    ASSERT_RESULT_SUCCESS(result);
    result = nn::svc::GetProcessInfo(&info, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            nn::svc::ProcessState_Terminated == static_cast<nn::svc::ProcessState>(info) ||
            nn::svc::ProcessState_Terminating == static_cast<nn::svc::ProcessState>(info)
            );

    // TEST 121-106
    // EnableJitDebug を指定しないと、JIT デバッグが出来ない
    nn::svc::Handle debugHandle;
    result = nn::svc::DebugActiveProcess(&debugHandle, pid);
    ASSERT_RESULT_FAILURE(result);
}

TEST(CreateProcess, CodeAreaTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose processCloser(handle);

    nn::svc::MemoryInfo blockInfo;
    nn::svc::PageInfo pageInfo;

    // TEST 121-100
    // 作成したプロセスの memoryAddress/memoryNumPages の領域は MemoryState_Code/MemoryPermission_None になる
    result = nn::svc::QueryProcessMemory(&blockInfo, &pageInfo, handle, 0x200000);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(blockInfo.state == nn::svc::MemoryState_Code);
    ASSERT_TRUE(blockInfo.permission == nn::svc::MemoryPermission_None);
}

#if defined NN_BUILD_CONFIG_CPU_ARM_V8A
// TEST 121-116
// 64 bit フラグが指定されていると、Aarch64 の命令が利用できる
TEST(CreateProcess, 64bitOperation)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;
    const uint32_t code = 0xd4000001 | (NN_SVC_ID_EXIT_PROCESS << 5);

    nn::Bit32 flags[DefaultCapabilityFlagNum];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    param.flags = nn::svc::CreateProcessParameterFlag_64Bit;

    result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose processCloser(handle);

    SetExecuteCode(
            handle, &code, sizeof(code), param.memoryAddress, param.memoryNumPages * 0x1000);

    int32_t priority = TestLowestThreadPriority;
    int32_t coreNo = g_ProcessIdealCore;
    result = nn::svc::StartProcess(handle, priority, coreNo, DefaultStackSize);
    ASSERT_RESULT_SUCCESS(result);

    WaitProcess(handle);
}
#endif // defined NN_BUILD_CONFIG_CPU_ARM_V8A

#ifdef SUPPORT_CAPABILITY
TEST(CreateProcess, EnableDebugTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    nn::svc::Handle handle;

    nn::Bit32 flags[DefaultCapabilityFlagNum + 1];
    SetDefaultCapability(flags, DefaultCapabilityFlagNum);
    MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 1 << 0);

    nn::svc::CreateProcessParameter param = {};
    SetDefaultParam(&param);
    param.flags = 0;

    {
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose processCloser(handle);

        nn::Bit64 pid;
        result = nn::svc::GetProcessId(&pid, handle);
        ASSERT_RESULT_SUCCESS(result);

        // デバッグ許可フラグが立っていると、デバッグできる
        nn::svc::Handle debugHandle;
        result = nn::svc::DebugActiveProcess(&debugHandle, pid);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose debugCloser(debugHandle);
    }
    {
        MakeNo16Flag(&flags[DefaultCapabilityFlagNum], 0);
        result = nn::svc::CreateProcess(&handle, param, flags, DefaultCapabilityFlagNum + 1);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose processCloser(handle);

        nn::Bit64 pid;
        result = nn::svc::GetProcessId(&pid, handle);
        ASSERT_RESULT_SUCCESS(result);

        // デバッグ許可フラグが立っていないと、デバッグできない
        nn::svc::Handle debugHandle;
        result = nn::svc::DebugActiveProcess(&debugHandle, pid);
        ASSERT_RESULT_FAILURE(result);
        AutoHandleClose debugCloser(debugHandle);
    }
}

extern char BinForceDebug_begin[];
extern char BinForceDebug_end[];
TEST(CreateProcess, ForceDebugTest)
{
    TestProcessLeak leakTest;
    nn::Result result;
    uintptr_t begin = reinterpret_cast<uintptr_t>(BinForceDebug_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinForceDebug_end);

    NamedPortManager namedPort(ForceDebugPortName, 1);
    nn::svc::Handle portHandle = namedPort.GetHandle();
    int32_t index;

    // ForceDebug あり
    {
        TestLoader loader(BinForceDebug_begin, end - begin);
        uint32_t flag = 0;
        MakeNo16Flag(&flag, 1 << 1);

        loader.SetAdditionalCapability(&flag, 1);

        nn::svc::Handle handle;
        loader.SpawnProcess(&handle);
        loader.StartProcess(handle);
        AutoHandleClose processCloser(handle);

        result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle serverSession;
        result = nn::svc::AcceptSession(&serverSession, portHandle);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        bool forceDebug = true;
        ServerSendData(serverSession, &forceDebug, sizeof(forceDebug));

        WaitProcess(handle);

        int64_t procInfo;
        result = nn::svc::GetProcessInfo(&procInfo, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(procInfo) != nn::svc::ProcessState_WaitAttach);
    }
    // ForceDebug なし
    {
        TestLoader loader(BinForceDebug_begin, end - begin);
        uint32_t flag = 0;
        MakeNo16Flag(&flag, 0);

        loader.SetAdditionalCapability(&flag, 1);

        nn::svc::Handle handle;
        loader.SpawnProcess(&handle);
        loader.StartProcess(handle);
        AutoHandleClose processCloser(handle);

        result = nn::svc::WaitSynchronization(&index, &portHandle, 1, -1);
        ASSERT_RESULT_SUCCESS(result);

        nn::svc::Handle serverSession;
        result = nn::svc::AcceptSession(&serverSession, portHandle);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);

        bool forceDebug = false;
        ServerSendData(serverSession, &forceDebug, sizeof(forceDebug));

        WaitProcess(handle);

        int64_t procInfo;
        result = nn::svc::GetProcessInfo(&procInfo, handle, nn::svc::ProcessInfoType_State);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_TRUE(
                static_cast<nn::svc::ProcessState>(procInfo) != nn::svc::ProcessState_WaitAttach);
    }
}

extern char BinCapability_begin[];
extern char BinCapability_end[];
TEST(CreateProcess, CapabilityTest)
{
    nn::Result result;
    uintptr_t begin = reinterpret_cast<uintptr_t>(BinCapability_begin);
    uintptr_t end = reinterpret_cast<uintptr_t>(BinCapability_end);
    TestLoader loader(BinCapability_begin, end - begin);

    TestHeap heap(HeapAlign);
    int maxFlag = heap.GetSize() / sizeof(nn::Bit32);
    nn::Bit32 *flags = reinterpret_cast<nn::Bit32*>(heap.GetAddress());

    int32_t flagNum = 0;
    int32_t tmp = 0;

    MakeNo3Flag(&flags[flagNum++], TestLowestThreadPriority, TestHighestThreadPriority, 0, 0);
    loader.SetPriority(TestLowestThreadPriority);
    loader.SetProcessorId(0);
    maxFlag--;

    int32_t enableSvc[] = {
        NN_SVC_ID_EXIT_PROCESS,
        NN_SVC_ID_BREAK,
        NN_SVC_ID_QUERY_MEMORY,
        NN_SVC_ID_GET_THREAD_PRIORITY,
        NN_SVC_ID_GET_THREAD_ID,
        NN_SVC_ID_OUTPUT_DEBUG_STRING,
        NN_SVC_ID_CREATE_PROCESS,
        NN_SVC_ID_SET_THREAD_PRIORITY,
        NN_SVC_ID_SET_THREAD_CORE_MASK,
        NN_SVC_ID_CREATE_INTERRUPT_EVENT,
        NN_SVC_ID_CLOSE_HANDLE,
        NN_SVC_ID_CREATE_THREAD,
        NN_SVC_ID_RETURN_FROM_EXCEPTION,
        NN_SVC_ID_SET_HEAP_SIZE,
        NN_SVC_ID_EXIT_THREAD,
        NN_SVC_ID_GET_INFO,
        NN_SVC_ID_CREATE_EVENT,
        NN_SVC_ID_CREATE_SESSION,
        NN_SVC_ID_ARBITRATE_LOCK,
        NN_SVC_ID_ARBITRATE_UNLOCK,
        NN_SVC_ID_WAIT_PROCESS_WIDE_KEY_ATOMIC,
        NN_SVC_ID_SIGNAL_PROCESS_WIDE_KEY,
    };

    MakeNo4Flag(&flags[flagNum], &tmp, maxFlag, enableSvc, sizeof(enableSvc) / sizeof(int32_t));
    flagNum += tmp;
    maxFlag -= tmp;

#if defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) \
    || defined( NN_BUILD_CONFIG_HARDWARE_NX )
    uint16_t name = static_cast<uint16_t>(TmrDefaultInterruptNumber);
    MakeNo11Flag(&flags[flagNum], &tmp, maxFlag, &name, 1);
    flagNum += tmp;
    maxFlag -= tmp;
#endif

    MakeNo14Flag(&flags[flagNum], NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
    flagNum++;
    maxFlag--;

    MakeNo15Flag(&flags[flagNum], NumMaxHandle);
    flagNum++;
    maxFlag--;

    loader.SetCapability(flags, flagNum);

    nn::svc::Handle handle;
    loader.SpawnProcess(&handle);
    loader.StartProcess(handle);
    AutoHandleClose processCloser(handle);

    WaitProcess(handle);

    int64_t procInfo;
    result = nn::svc::GetProcessInfo(&procInfo, handle, nn::svc::ProcessInfoType_State);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(
            static_cast<nn::svc::ProcessState>(procInfo) != nn::svc::ProcessState_WaitAttach);
}
#endif // SUPPORT_CAPABILITY

