﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/os.h>

extern "C" void nnMain();

namespace {

const int32_t NumPriority = TestLowestThreadPriority - TestHighestThreadPriority;
const int32_t HalfPriority = NumPriority / 2 + TestHighestThreadPriority;

const int64_t SleepTime = 100 * 1000 * 1000;
char *g_Stack[2][DefaultStackSize] __attribute__((aligned(0x1000)));
volatile nn::Bit32 g_Lock;
int32_t g_Sequence;

#ifdef WAIT_LOCK_FOREVER
void DummyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
}
#endif

struct TestData
{
    nn::svc::Handle handle;
    uintptr_t addr;
};

void WaitLockThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    TestData* testData = reinterpret_cast<TestData*>(arg);

    NN_ASSERT(g_Sequence == 0);
    g_Sequence = 1;

    result = nn::svc::ArbitrateLock(testData->handle, testData->addr, 1);
    ASSERT_RESULT_SUCCESS(result);

    g_Sequence = 2;
}

void WaitMultiThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    TestData* testData = reinterpret_cast<TestData*>(arg);

    nnHandle handle = testData->handle;
    nn::Bit32 value = handle.value | nn::svc::Handle::WaitMask;
    nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(testData->addr);
    *ptr = value;

    // 低優先度
    if (g_Sequence == 0)
    {
        g_Sequence = 1;
        nnHandle handle = testData->handle;
        nn::Bit32 value = handle.value | nn::svc::Handle::WaitMask;
        nn::Bit32* ptr = reinterpret_cast<nn::Bit32*>(testData->addr);
        *ptr = value;

        result = nn::svc::ArbitrateLock(testData->handle, testData->addr, 1);
        NN_ASSERT_RESULT_SUCCESS(result);

        NN_ASSERT(g_Sequence == 3);

        // TEST 135-18
        // 待機が解除され、このシステムコールを呼び出しているスレッドの待機列上に
        // addr をキーとして待機しているスレッドが存在していない場合、
        // newValue の値を addr で指定されるアドレスに書き込まれる
        NN_ASSERT(*ptr == 1);
    }
    // 高優先度
    else if (g_Sequence == 1)
    {
        int32_t priority;

        // TEST 137-10
        // カレントスレッドの待機列上で addr をキーとして待機しているスレッドが存在する場合、
        // 待機しているスレッドの中で一番高優先度のスレッドを起床させる
        g_Sequence = 2;

        result = nn::svc::ArbitrateLock(testData->handle, testData->addr, 2);
        NN_ASSERT_RESULT_SUCCESS(result);

        NN_ASSERT(g_Sequence == 2);

        // TEST 135-19
        // 待機が解除され、このシステムコールを呼び出しているスレッドの待機列上に
        // addr をキーとして待機しているスレッドが存在している場合、
        // newValue の値と 0x40000000 のビットごとの OR を取った Bit32 値を
        // addr で指定されるアドレスに書き込まれる
        NN_ASSERT(*ptr == (2 | nn::svc::Handle::WaitMask));

        // TEST 137-11
        // 同じスレッドの同じアドレスで待っていた他のスレッドがこのスレッドの待機列に入っている
        nn::svc::SleepThread(SleepTime);

        NN_ASSERT(g_Sequence == 2);

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT(priority == HalfPriority);

        result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, HalfPriority + 2);
        NN_ASSERT_RESULT_SUCCESS(result);

        // TEST 137-12
        // カレントスレッドに対して実施されていた優先度継承の全てを起床させたスレッドで開始される
        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT(priority == HalfPriority + 1);

        g_Sequence = 3;

        result = nn::svc::ArbitrateUnlock(testData->addr);
        NN_ASSERT_RESULT_SUCCESS(result);

        result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT(priority == HalfPriority + 2);
    }
    else
    {
        NN_ASSERT(false);
    }
}

#ifdef WAIT_LOCK_FOREVER
void WaitDummyThread(uintptr_t arg)
{
    AutoThreadExit autoExit;
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    nn::svc::Handle* handle = reinterpret_cast<nn::svc::Handle*>(arg);

    g_Lock = static_cast<nnHandle>(*handle).value | nn::svc::Handle::WaitMask;

    result = nn::svc::ArbitrateLock(*handle, addr, 0);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidState());
}
#endif

} // namespace

TEST(ArbitrateLock, AddrValueTest)
{
    nn::Result result;
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    g_Lock = 0;

    TestData testData;
    testData.handle = nn::svc::Handle(nn::os::GetCurrentThread()->_handle);
    testData.addr = addr;

    nnHandle handle = testData.handle;
    nn::Bit32 value = handle.value | nn::svc::Handle::WaitMask;

    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitLockThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&testData);
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    // TEST 135-16
    // addr で指定されるアドレスに格納されている Bit32 の値が、
    // thread の値と 0x40000000 のビットごとの OR を取った Bit32 値と一致しないなら
    // 何もせずに返る
    {
        g_Sequence = 0;
        g_Lock = 0;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();
        nn::svc::SleepThread(SleepTime);
        ASSERT_TRUE(g_Sequence == 2);
        thread.Wait();
    }

    // TEST 135-17
    // addr で指定されるアドレスに格納されている Bit32 の値が、
    // thread の値と 0x40000000 のビットごとの OR を取った Bit32 値と一致すると、
    // ロックが解除されるのを待つ
    {
        g_Sequence = 0;
        g_Lock = value;
        TestThread thread(pc, param, sp, priority, idealCore);
        thread.Start();

        nn::svc::SleepThread(SleepTime);
        ASSERT_TRUE(g_Sequence == 1);

        result = nn::svc::ArbitrateUnlock(addr);
        ASSERT_RESULT_SUCCESS(result);

        thread.Wait();
        ASSERT_TRUE(g_Sequence == 2);
    }
}

// TEST 135-23
// ロックを確保しているスレッドがロック待ちをしているスレッドより優先度が低くなった場合でも、
// 優先度逆転が起こる
TEST(ArbitrateLock, ChangeLowerPriority)
{
    nn::Result result;
    TestChangePriority testPriority(HalfPriority - 1);
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    g_Lock = 0;

    TestData testData;
    testData.handle = nn::svc::Handle(nn::os::GetCurrentThread()->_handle);
    testData.addr = addr;

    nnHandle handle = testData.handle;
    nn::Bit32 value = handle.value | nn::svc::Handle::WaitMask;
    g_Lock = value;

    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitLockThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&testData); // TODO
    int32_t priority = HalfPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    g_Sequence = 0;
    TestThread thread(pc, param, sp, priority, idealCore);
    thread.Start();

    // 相手が待ち状態になるのを待つ
    while(g_Sequence == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }

    result = nn::svc::GetThreadPriority(&priority, thread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority);

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority - 1);

    result = nn::svc::SetThreadPriority(nn::svc::PSEUDO_HANDLE_CURRENT_THREAD, HalfPriority + 1);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority);

    result = nn::svc::GetThreadPriority(&priority, thread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority );

    result = nn::svc::ArbitrateUnlock(addr);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority + 1);

    thread.Wait();
}

TEST(ArbitrateLock, WaitMultiThread)
{
    nn::Result result;
    TestChangePriority testPriority(HalfPriority + 2);
    uintptr_t addr = reinterpret_cast<uintptr_t>(&g_Lock);
    g_Lock = 0;

    TestData testData;
    testData.handle = nn::svc::Handle(nn::os::GetCurrentThread()->_handle);
    testData.addr = addr;

    uintptr_t pc = reinterpret_cast<uintptr_t>(WaitMultiThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack) + sizeof(g_Stack);
    uintptr_t param = reinterpret_cast<uintptr_t>(&testData); // TODO
    int32_t priority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    g_Sequence = 0;
    TestThread lowerThread(pc, param, sp, HalfPriority + 1, idealCore);
    lowerThread.Start();

    // 相手が待ち状態になるのを待つ
    while(g_Sequence == 0)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == 1);

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority + 1);

    result = nn::svc::GetThreadPriority(&priority, lowerThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority + 1);

    TestThread higherThread(pc, param, sp, HalfPriority, idealCore);
    higherThread.Start();

    // 相手が待ち状態になるのを待つ
    while(g_Sequence == 1)
    {
        nn::svc::SleepThread(SleepTime);
    }
    ASSERT_TRUE(g_Sequence == 2);

    // TEST 135-24
    // スレッドが3つあり、それぞれ A  <- B, A <- C (ロック確保 <- 解除待ち) の関係にあるとき、
    // C の優先度がAよりも高い場合、A の優先度が C と同じ優先度になるが、
    // B の優先度は変わらない
    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority);

    result = nn::svc::GetThreadPriority(&priority, lowerThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority + 1);

    result = nn::svc::GetThreadPriority(&priority, higherThread.GetHandle());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority);

    // TEST 135-25
    // スレッドが3つあり、それぞれ A <- B, A <- C (ロック確保 <- 解除待ち) の関係にあるとき、
    // C の優先度がBよりも高い場合、A がロックを解除すると、C <- B の関係になる
    result = nn::svc::ArbitrateUnlock(addr);
    ASSERT_RESULT_SUCCESS(result);

    result = nn::svc::GetThreadPriority(&priority, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(priority == HalfPriority + 2);

    higherThread.Wait();
    lowerThread.Wait();
}

#ifdef WAIT_LOCK_FOREVER
// TEST 135-28
// ロックを保持したスレッドがロックを解放せずに終了すると、スレッドは待ち続ける
TEST(ArbitrateLock, LostLockThreadTest)
{
    nn::Result result;
    g_Lock = 0;

    uintptr_t pc = reinterpret_cast<uintptr_t>(DummyThread);
    uintptr_t sp = reinterpret_cast<uintptr_t>(g_Stack[0]) + sizeof(g_Stack[0]);
    uintptr_t param = 0;
    int32_t priority = TestLowestThreadPriority;
    int32_t idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread dummyThread(pc, param, sp, priority, idealCore);
    nn::svc::Handle dummyHandle = dummyThread.GetHandle();

    pc = reinterpret_cast<uintptr_t>(WaitDummyThread);
    sp = reinterpret_cast<uintptr_t>(g_Stack[1]) + sizeof(g_Stack[1]);
    param = reinterpret_cast<uintptr_t>(&dummyHandle);
    priority = TestLowestThreadPriority - 1;
    idealCore = nn::svc::IdealCoreUseProcessValue;

    TestThread waitThread(pc, param, sp, priority, idealCore);
    waitThread.Start();
    dummyThread.Start();

    dummyThread.Wait();
    result = nn::svc::CloseHandle(dummyHandle);
    ASSERT_RESULT_SUCCESS(result);

    // waitThread のロック待ち状態を解除する方法がない
}
#endif // WAIT_LOCK_FOREVER

