﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_Common.h"
#include <nn/svc/svc_Tcb.h>

namespace {

const nn::svc::Handle TmpHandle = nn::svc::INVALID_HANDLE_VALUE;
extern "C" void nnMain();

} // namespace

#ifdef INVALID_POINTER_TEST
TEST(AcceptSession, InvalidPointerTest)
{
    nn::Result result;
    nn::svc::Handle* handle;

    // ポートの設定
    nn::svc::Handle serverPort;
    nn::svc::Handle clientPort;
    result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose sPortCloser(serverPort);
    AutoHandleClose cPortCloser(clientPort);

    // クライアントの接続
    nn::svc::Handle clientSession;
    result = nn::svc::ConnectToPort(&clientSession, clientPort);
    ASSERT_RESULT_SUCCESS(result);
    AutoHandleClose cSessionCloser(clientSession);

    int32_t index;
    result = nn::svc::WaitSynchronization(&index, &serverPort, 1, 0);
    ASSERT_RESULT_SUCCESS(result);

#ifdef INVALID_POINTER_TEST
    // TEST 65-6
    // NULL アドレスを受け付けない
    result = nn::svc::AcceptSession(NULL, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 65-7
    // MemoryPermission_None の領域を渡すと失敗する
    handle = reinterpret_cast<nn::svc::Handle*>(g_FreeAreaBegin);
    result = nn::svc::AcceptSession(handle, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 65-8
    // MemoryPermission_Read の領域を渡すと失敗する
    {
        TestHeap heap(HeapAlign);
        uintptr_t addr = heap.GetAddress();
        {
            TestMemoryPermission perm(addr, 0x1000, nn::svc::MemoryPermission_Read);
            handle = reinterpret_cast<nn::svc::Handle*>(addr);
            result = nn::svc::AcceptSession(handle, serverPort);
            ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
        }
    }
#endif // INVALID_POINTER_TEST

#ifdef INVALID_POINTER_TEST
    // TEST 65-9
    // MemoryPermission_ReadExecute の領域を渡すと失敗する
    handle = reinterpret_cast<nn::svc::Handle*>(nnMain);
    result = nn::svc::AcceptSession(handle, serverPort);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidPointer());
#endif // INVALID_POINTER_TEST
}
#endif // INVALID_POINTER_TEST

TEST(AcceptSession, InvalidHandleTest)
{
    TestSessionLeak sessionTest;
    nn::Result result;
    nn::svc::Handle handle;

    // TEST 65-12
    // INVALID_HANDLE_VALUE を受け付けない
    result = nn::svc::AcceptSession(&handle, nn::svc::INVALID_HANDLE_VALUE);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 65-13
    // T_CurrentThread を受け付けない
    result = nn::svc::AcceptSession(&handle, nn::svc::PSEUDO_HANDLE_CURRENT_THREAD);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 65-14
    // T_CurrentProcess を受け付けない
    result = nn::svc::AcceptSession(&handle, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

    // TEST 65-15
    // Session のハンドルを受け付けない
    {
        nn::svc::Handle serverSession;
        nn::svc::Handle clientSession;
        result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sSessionCloser(serverSession);
        AutoHandleClose cSessionCloser(clientSession);

        result = nn::svc::AcceptSession(&handle, serverSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());

        result = nn::svc::AcceptSession(&handle, clientSession);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }

    {
        nn::svc::Handle serverPort;
        nn::svc::Handle clientPort;

        result = nn::svc::CreatePort(&serverPort, &clientPort, 1, false, 0);
        ASSERT_RESULT_SUCCESS(result);
        AutoHandleClose sPortCloser(serverPort);
        AutoHandleClose cPortCloser(clientPort);

        // TEST 65-16
        // Connect されていないサーバーポートを受け付けない
        result = nn::svc::AcceptSession(&handle, serverPort);
        ASSERT_RESULT_FAILURE(result);

        // TEST 65-17
        // クライアントポートのハンドルを受け付けない
        result = nn::svc::AcceptSession(&handle, clientPort);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::svc::ResultInvalidHandle());
    }
}


