﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include "../include/nn/i2c/driver/i2c.h"

#include "detail/i2c_TargetSpec.h"
#include "detail/i2c_ResourceManager.h"

namespace {

void AssertLibraryInitialized()
{
    NN_SDK_REQUIRES(nn::i2c::driver::detail::ResourceManager::GetInstance().IsInitialized(),
        "I2C Error: Library is not initialized\n");
}

} // namespace

namespace nn {
namespace i2c {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    nn::i2c::driver::detail::ResourceManager::GetInstance().Initialize();
}

void Finalize() NN_NOEXCEPT
{
    nn::i2c::driver::detail::ResourceManager::GetInstance().Finalize();
}

bool HasDevice(I2cDevice device) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    return HasDeviceForDev( detail::GetBusIdx(device), detail::GetSlaveAddress(device), detail::GetAddressingMode(device), detail::GetSpeedMode(device) );
}

bool HasDeviceForDev( int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT
{
    AssertLibraryInitialized();

    I2cSession session;
    const nn::i2c::TransactionOption InOption = static_cast<nn::i2c::TransactionOption>( nn::i2c::TransactionOption_StartCondition | nn::i2c::TransactionOption_StopCondition );
    Bit8 dummyData = 0x00; // ダミーデータを 1 Byte 受信してデバイスの存在の有無をチェックする

    OpenSessionForDev(&session, busIdx, slaveAddress, addressingMode, speedMode );
    auto result = Receive( &dummyData, session, 1, InOption );
    CloseSession( session );

    return result.IsSuccess();
}

void OpenSession(I2cSession* pOutSession, I2cDevice device) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    nn::i2c::driver::detail::ResourceManager::GetInstance().OpenSession(
        pOutSession, detail::GetBusIdx(device), detail::GetSlaveAddress(device), detail::GetAddressingMode(device), detail::GetSpeedMode(device) );
}

void OpenSessionForDev(I2cSession* pOutSession, int busIdx, Bit16 slaveAddress, AddressingMode addressingMode, SpeedMode speedMode ) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_SDK_REQUIRES(busIdx < nn::i2c::driver::detail::MaxBuses);

    nn::i2c::driver::detail::ResourceManager::GetInstance().OpenSession(
        pOutSession, busIdx, slaveAddress, addressingMode, speedMode );
}

Result Send( const I2cSession& session, const void* pInData, size_t dataBytes, TransactionOption inOption ) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    NN_SDK_REQUIRES_NOT_NULL(pInData);
    NN_ABORT_UNLESS(dataBytes > 0);

    return nn::i2c::driver::detail::ResourceManager::GetInstance().GetSession(session._sessionId).ExecuteTransaction(
        nullptr, pInData, dataBytes, inOption, nn::i2c::driver::detail::Command_Send );
}

Result Receive( void* pOutData, const I2cSession& session, size_t dataBytes, TransactionOption inOption ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    NN_ABORT_UNLESS(dataBytes > 0);

    AssertLibraryInitialized();
    return nn::i2c::driver::detail::ResourceManager::GetInstance().GetSession(session._sessionId).ExecuteTransaction(
        pOutData, nullptr, dataBytes, inOption, nn::i2c::driver::detail::Command_Receive );
}

Result ExecuteCommandList( void* pReceiveBuffer, size_t receiveBufferSize, const I2cSession& session, const void* pCommandList, size_t commandListLength ) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    // TODO: 実装
    return ResultSuccess();
}

void CloseSession( const I2cSession& session ) NN_NOEXCEPT
{
    AssertLibraryInitialized();
    nn::i2c::driver::detail::ResourceManager::GetInstance().CloseSession( session );
}

} // driver
} // i2c
} // nn
