﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
--- To launch the test process, which will start a thread:
testCpu_Use.nsp --gtest_filter=CpuUseTestFixture.StartThread --cpu_core=3 --busy_milliseconds=60000 --thread_priority=19 --thread_duty_cycle=30
--- To launch the test as a separate process, which will signal the thread to complete
testCpu_Use.nsp --gtest_filter=CpuUseTestFixture.DestroyThread
*/


#include "test_CpuUse_Setup.h"
#include <nnt/nntest.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/fs.h>
#include <nn/init.h>


// initialize our heap
const size_t HeapSize = 2 * 1024 * 1024;
const size_t BlockSize = 2 * 1024 * 1024;

// thread stack
NN_OS_ALIGNAS_THREAD_STACK char threadStack1[STACK_SIZE];


// the default nninitStartup() function automatically allocates the maximum usable memory heap when the program is started
// use our own for memory management
extern "C" void nninitStartup()
{
    // allocate the memory heap
    NN_LOG("SetMemoryHeapSize initialization\n");
    nn::Result result = nn::os::SetMemoryHeapSize(HeapSize);
    ASSERT_TRUE(result.IsSuccess());

    // use a block of the allocated memory heap
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, BlockSize);
    ASSERT_TRUE(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), BlockSize);
}


// thread function
void ThreadFunc(void *args)
{
    // max time to let thread run before yield
    const int thread_yield_ms = 10;
    ThreadArg *testArg = (ThreadArg *)args;
    nn::TimeSpan stopTime = 0;
    nn::TimeSpan yieldTime = 0;
    bool fTerminate = false;
    nn::Result result;
    int i = 0;
    int off_ms = testArg->busy_wait_ms * (100 - testArg->thread_duty_cycle_percent) / testArg->thread_duty_cycle_percent;

    while (!fTerminate)
    {
        // thread on cycle
        stopTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(testArg->busy_wait_ms);
        yieldTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(thread_yield_ms);
        while (nn::os::GetSystemTick().ToTimeSpan() < stopTime)
        {
            // busy wait loop
            i++;
            i--;

            // since threads are run to completion on NX, a call to YieldThread
            // is necessary in order to allow other threads with the same priority to run
            // yield every duration of thread_yield_ms
            if (nn::os::GetSystemTick().ToTimeSpan() >= yieldTime)
            {
                nn::os::YieldThread();
                yieldTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromMilliSeconds(thread_yield_ms);
            }
        }

        // check the signal file to see if this thread should terminate
        fTerminate = CpuUseTestFixture::IsSignalFileExists();

        if (!fTerminate)
        {
            int slices_off_ms = off_ms;

            // sleep during the thread off cycle in time slices of thread_yield_ms duration
            while (slices_off_ms >= thread_yield_ms)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(thread_yield_ms));
                slices_off_ms = slices_off_ms - thread_yield_ms;

                // since threads are run to completion on NX, a call to YieldThread
                // is necessary in order to allow other threads with the same priority to run
                nn::os::YieldThread();
            }

            // check for any left over time slice to sleep in ms
            if (slices_off_ms > 0)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(slices_off_ms));
            }

            // check the signal file to see if this thread should terminate
            fTerminate = CpuUseTestFixture::IsSignalFileExists();
        }
    }

    // delete the signal file
    CpuUseTestFixture::DeleteSignalFile();
}


/*
--- To launch the test process, which will start a thread:
testCpu_Use.nsp --gtest_filter=CpuUseTestFixture.StartThread --cpu_core=3 --busy_milliseconds=60000 --thread_priority=19 --thread_duty_cycle=30
*/
TEST_F(CpuUseTestFixture, StartThread)
{
    nn::Result result;

    DeleteSignalFile();

    // create a thread to run on the specific cpu core
    result = nn::os::CreateThread(
                        &threadForUse,
                        ThreadFunc,
                        (void *) &threadArgs,
                        threadStack1,
                        STACK_SIZE,
                        threadArgs.thread_priority,
                        threadArgs.cpu_core);
    ASSERT_TRUE(result.IsSuccess());

    // start the thread
    NN_LOG("Thread started : %i\n", &threadForUse);
    nn::os::StartThread(&threadForUse);
    nn::os::WaitThread(&threadForUse);

    NN_LOG("Thread destroyed : %i\n", &threadForUse);
    nn::os::DestroyThread(&threadForUse);
}


/*
--- To launch the test as a separate process, which will signal the thread to complete
testCpu_Use.nsp --gtest_filter=CpuUseTestFixture.DestroyThread
*/
TEST_F(CpuUseTestFixture, DestroyThread)
{
    nn::Result result;

    // create the file to signal end
    ASSERT_TRUE(CreateSignalFile());
}
