﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <map>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/crypto.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Result.h>
#include <nn/spl/smc/spl_SecureMonitorApi.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/factory/settings_DeviceKey.h>
#include <nn/settings/factory/settings_GameCard.h>
#include <nn/settings/factory/settings_Ssl.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

namespace
{
    nn::Bit8 g_Work[16 * 1024 * 1024 ];

    const nn::Bit8 Zero[16] = { 0 };

    const nn::Bit8 Key[nn::spl::AesKeySize] = {0x63,0xde,0x8b,0xba,0xb0,0xf9,0x1d,0x60,0xc2,0x1c,0x6d,0xf6,0x11,0x64,0xea,0x28};
/*
    void Dump(const void* p, size_t size)
    {
        auto p8 = reinterpret_cast<const nn::Bit8*>(p);
        for( int i = 0; i < size; ++i )
        {
            if( (i % 32) == 0 )
            {
                NN_LOG("%03x| ", i);
            }
            NN_LOG("%02x ", p8[i]);
            if( (i % 32) == 31 )
            {
                NN_LOG("\n");
            }
        }
        if( (size % 32) != 0 )
        {
            NN_LOG("\n");
        }
    }
*/

    void LoadKey(int slotIndex)
    {
        nn::spl::AccessKey ak;

        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::GenerateAesKek(
                &ak,
                Zero,   sizeof(Zero),
                0,
                0 ));

        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::LoadAesKey(
                slotIndex,
                ak,
                Zero,   sizeof(Zero) ));
    }

    void RunMonobitTest(const uint8_t* pData, size_t size) NN_NOEXCEPT
    {
        auto count = 0;
        for (auto i = 0; i < size; ++i)
        {
            count += ::nn::util::popcount(pData[i]);
        }

        NN_LOG("# of 1: %d\n", count);
        EXPECT_TRUE(9725 < count && count < 10275);
    }

    void GetHistogram(std::map<int, int> *pOut, const uint8_t* pData, size_t size) NN_NOEXCEPT
    {
        for (auto i = 0; i < 2; ++i)
        {
            pOut[i].clear();
        }

        int previousBit = -1;
        int current = 0;

        for (auto i = 0; i < size; ++i)
        {
            for (auto j = 0; j < 8; ++j)
            {
                int bit = (pData[i] >> j) & 1;

                if (bit == previousBit)
                {
                    ++current;
                }
                else
                {
                    if (previousBit != -1)
                    {
                        ++pOut[previousBit][current];
                    }
                    current = 1;
                    previousBit = bit;
                }
            }
        }
    }

    void RunRunsTest(const uint8_t* pData, size_t size) NN_NOEXCEPT
    {
        std::map<int, int> histogram[2];
        GetHistogram(histogram, pData, size);

        const auto lengthMax = 6;

        struct Range
        {
            int min;
            int max;
        };
        const Range expectedCounts[] =
            {
                { 2315, 2685 },
                { 1114, 1386 },
                {  527,  723 },
                {  240,  384 },
                {  103,  209 },
                {  103,  209 },
            };
        NN_STATIC_ASSERT(sizeof(expectedCounts) / sizeof(expectedCounts[0]) == lengthMax);

        for (auto i = 0; i < 2; ++i)
        {
            int counts[lengthMax] = {};

            for (auto& e : histogram[i])
            {
                auto length = e.first;
                auto count = e.second;
                if (length >= lengthMax)
                {
                    counts[lengthMax - 1] += count;
                }
                else
                {
                    counts[length - 1] += count;
                }
            }

            for (auto j = 0; j < lengthMax; ++j)
            {
                if (j == lengthMax - 1)
                {
                    NN_LOG("# of %d run whose length >=", i);
                }
                else
                {
                    NN_LOG("# of %d run whose length ==", i);
                }
                NN_LOG(" %d: %d\n", j + 1, counts[j]);
                EXPECT_TRUE(expectedCounts[j].min <= counts[j] && counts[j] <= expectedCounts[j].max);
            }
        }
    }

    void RunLongRunsTest(const uint8_t* pData, size_t size) NN_NOEXCEPT
    {
        std::map<int, int> histogram[2];
        GetHistogram(histogram, pData, size);

        auto longRunLength = 0;

        for (auto i = 0; i < 2; ++i)
        {
            for (auto& e : histogram[i])
            {
                auto length = e.first;
                longRunLength = std::max(longRunLength, length);
            }
        }

        NN_LOG("longrun: %d\n", longRunLength);
        EXPECT_TRUE(longRunLength < 26);
    }

    void RunPokerTest(const uint8_t* pData, size_t size) NN_NOEXCEPT
    {
        int frequency[16] = {};
        for (auto i = 0; i < size; ++i)
        {
            for (auto j = 0; j < 8; j += 4)
            {
                int halfByte = (pData[i] >> j) & 0xF;
                ++frequency[halfByte];
            }
        }

        int sum = 0;
        for (auto i = 0; i < sizeof(frequency) / sizeof(frequency[0]); ++i)
        {
            sum += frequency[i] * frequency[i];
        }
        auto criteria = static_cast<double>(sum) * 16 / 5000 - 5000;
        NN_LOG("criteria: %f\n", criteria);
        EXPECT_TRUE(2.16 < criteria && criteria < 46.17);
    }

    template <int ModulusSize, int PublicExponentSize, typename Function>
    void RsaOaepSha256Test(
        const uint8_t (&modulus)[ModulusSize],
        const uint8_t (&publicExponent)[PublicExponentSize],
        Function decrypt) NN_NOEXCEPT
    {
        uint8_t input[] =
        {
            0xDF,0x37,0x76,0xFE,0x7D,0xB6,0x47,0x94,0xE7,0xCE,0x26,0xE2,0x9E,0x8E,0xA3,0xA5,
            0x39,0xB0,0x17,0xE5,0xA4,0x09,0x6E,0xBD,0x45,0x0B,0xF1,0xD6,0x14,0x89,0x49,0x3A,
            0x5F,0x22,0xEE,0xB4,0x73,0xD9,0x23,0x58,0xBE,0x9D,0x6E,0x7B,0x6D,0xDE,0xCA,0x9F,
            0x7D,0x46,0x22,0x07,0x56,0x5F,0xA6,0x25,0x21,0xE4,0x5D,0x7C,0xAE,0xC1,0x75,0xD7,
            0x84,0xC3,0x05,0xAB,0x1C,0x40,0xD0,0x12,0x7E,0x39,0xCA,0xE2,0x04,0xE2,0xF2,0x16,
            0xF2,0x4F,0xDE,0xFB,0x4C,0x83,0x7D,0xC4,0x46,0xB5,0x0F,0xDD,0x6E,0xAD,0xCD,0x82,
            0x3E,0x0D,0xD5,0x52,0xBA,0x36,0x77,0x87,0x2E,0x38,0x77,0x73,0x9D,0x59,0x5C,0x6D,
            0xAB,0xC4,0x1F,0x56,0xF5,0x22,0x69,0x0D,0xB3,0x26,0x81,0xC9,0x89,0x4F,0xDD,0x35,
            0x7C,0x1F,0xE4,0x57,0x13,0x84,0x8A,0xDC,0xAF,0xEC,0x99,0x14,0x02,0xE0,0x2D,0x91,
            0x98,0x33,0xBF,0x3E,0x28,0x56,0x70,0x27,0xF2,0x2C,0x90,0x20,0xFD,0xCB,0x1D,0xF6,
            0x27,0xD3,0x1B,0x51,0x5A,0xB3,0xDF,0xCE,0x80,0xDE,0xC6,0x85,0xB2,0xCB,0x38,0xA0,
            0x74,0x1F,0xD0,0x6F,0xEC,0xB4,0x8B,0xB4,0x82,0xAE,0x24,0xEA,0x62,0x32,
        };
        uint8_t output[256];
        uint8_t labelDigest[::nn::crypto::Sha256Generator::HashSize] = {};
        // テストのためにシード固定
        uint8_t seed[::nn::crypto::Sha256Generator::HashSize] = {};

        ::nn::crypto::Rsa2048OaepSha256Encryptor encryptor;
        EXPECT_TRUE(encryptor.Initialize(modulus, sizeof(modulus), publicExponent, sizeof(publicExponent)));
        encryptor.SetLabelDigest(labelDigest, sizeof(labelDigest));
        EXPECT_TRUE(encryptor.Encrypt(output, sizeof(output), input, sizeof(input), seed, sizeof(seed)));

        size_t resultSize;
        NNT_EXPECT_RESULT_SUCCESS(
            decrypt(
                &resultSize,
                output, sizeof(output),
                output, sizeof(output),
                modulus, sizeof(modulus),
                labelDigest, sizeof(labelDigest)));

        EXPECT_EQ(resultSize, sizeof(input));
        EXPECT_EQ(::std::memcmp(output, input, sizeof(input)), 0);
    }

    template <int ModulusSize, int PublicExponentSize, typename Function>
    void RsaTest(
        const uint8_t (&modulus)[ModulusSize],
        const uint8_t (&publicExponent)[PublicExponentSize],
        Function decrypt) NN_NOEXCEPT
    {
        uint8_t input[] =
        {
            0x00,0x37,0x76,0xFE,0x7D,0xB6,0x47,0x94,0xE7,0xCE,0x26,0xE2,0x9E,0x8E,0xA3,0xA5,
            0x39,0xB0,0x17,0xE5,0xA4,0x09,0x6E,0xBD,0x45,0x0B,0xF1,0xD6,0x14,0x89,0x49,0x3A,
            0x5F,0x22,0xEE,0xB4,0x73,0xD9,0x23,0x58,0xBE,0x9D,0x6E,0x7B,0x6D,0xDE,0xCA,0x9F,
            0x7D,0x46,0x22,0x07,0x56,0x5F,0xA6,0x25,0x21,0xE4,0x5D,0x7C,0xAE,0xC1,0x75,0xD7,
            0x84,0xC3,0x05,0xAB,0x1C,0x40,0xD0,0x12,0x7E,0x39,0xCA,0xE2,0x04,0xE2,0xF2,0x16,
            0xF2,0x4F,0xDE,0xFB,0x4C,0x83,0x7D,0xC4,0x46,0xB5,0x0F,0xDD,0x6E,0xAD,0xCD,0x82,
            0x3E,0x0D,0xD5,0x52,0xBA,0x36,0x77,0x87,0x2E,0x38,0x77,0x73,0x9D,0x59,0x5C,0x6D,
            0xAB,0xC4,0x1F,0x56,0xF5,0x22,0x69,0x0D,0xB3,0x26,0x81,0xC9,0x89,0x4F,0xDD,0x35,
            0x7C,0x1F,0xE4,0x57,0x13,0x84,0x8A,0xDC,0xAF,0xEC,0x99,0x14,0x02,0xE0,0x2D,0x91,
            0x98,0x33,0xBF,0x3E,0x28,0x56,0x70,0x27,0xF2,0x2C,0x90,0x20,0xFD,0xCB,0x1D,0xF6,
            0x27,0xD3,0x1B,0x51,0x5A,0xB3,0xDF,0xCE,0x80,0xDE,0xC6,0x85,0xB2,0xCB,0x38,0xA0,
            0x74,0x1F,0xD0,0x6F,0xEC,0xB4,0x8B,0xB4,0x82,0xAE,0x24,0xEA,0x62,0x32,0x73,0xE5,
            0x09,0x63,0xDB,0xB6,0x9A,0xF4,0x2A,0x0D,0x25,0x9D,0x03,0x66,0xBF,0xAB,0x0C,0x47,
            0x12,0xDD,0x2E,0xDC,0xA9,0xA0,0x75,0x3B,0x91,0x15,0x2D,0xF0,0x7C,0xD3,0x5A,0xCC,
            0x6C,0x89,0xDF,0x03,0x20,0xA0,0x31,0xAB,0x6A,0x1F,0x5E,0xC6,0xAE,0x36,0xC8,0x58,
            0xDD,0xAD,0xFB,0x70,0xF1,0x2E,0x07,0x65,0xDD,0x7F,0xDA,0x68,0x7D,0xC6,0x7A,0x02,
        };
        uint8_t output[256];

        ::nn::crypto::RsaCalculator<sizeof(modulus), sizeof(publicExponent)> encryptor;
        EXPECT_TRUE(encryptor.Initialize(modulus, sizeof(modulus), publicExponent, sizeof(publicExponent)));
        EXPECT_TRUE(encryptor.ModExp(output, sizeof(output), input, sizeof(input)));

        NNT_EXPECT_RESULT_SUCCESS(
            decrypt(
                output, sizeof(output),
                output, sizeof(output),
                modulus, sizeof(modulus)));

        EXPECT_EQ(::std::memcmp(output, input, sizeof(output)), 0);
    }
}

TEST(SplBasicTest, MultiInitialization)
{
    ::nn::spl::Initialize();
    ::nn::spl::Initialize();
    ::nn::spl::Finalize();
    ::nn::spl::Finalize();
}

TEST(SplBasicTest, GetHardwareType)
{
    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto hardwareType = ::nn::spl::GetHardwareType();
    EXPECT_TRUE(
        hardwareType == ::nn::spl::HardwareType_Icosa
        || hardwareType == ::nn::spl::HardwareType_Copper
        || hardwareType == ::nn::spl::HardwareType_Hoag
        || hardwareType == ::nn::spl::HardwareType_Iowa);

    switch (hardwareType)
    {
    case ::nn::spl::HardwareType_Icosa:
        NN_LOG("HardwareType: Icosa\n");
        break;
    case ::nn::spl::HardwareType_Copper:
        NN_LOG("HardwareType: Copper\n");
        break;
    case ::nn::spl::HardwareType_Hoag:
        NN_LOG("HardwareType: Hoag\n");
        break;
    case ::nn::spl::HardwareType_Iowa:
        NN_LOG("HardwareType: Iowa\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

TEST(SplBasicTest, GetHardwareState)
{
    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto hardwareState = ::nn::spl::GetHardwareState();
    EXPECT_TRUE(
        hardwareState == ::nn::spl::HardwareState_Development
        || hardwareState == ::nn::spl::HardwareState_End);
    auto isDevelopment = ::nn::spl::IsDevelopment();

    switch (hardwareState)
    {
    case ::nn::spl::HardwareState_Development:
        NN_LOG("HardwareState: Development\n");
        EXPECT_TRUE(isDevelopment == true);
        break;
    case ::nn::spl::HardwareState_End:
        NN_LOG("HardwareState: End\n");
        EXPECT_TRUE(isDevelopment == false);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

TEST(SplBasicTest, GetSocType)
{
    std::map<::nn::spl::SocType, const char*> entries;
#define MAKE_ENTRY(name) std::make_pair(::nn::spl::SocType_ ## name, #name)
    entries.insert(MAKE_ENTRY(Erista));
    entries.insert(MAKE_ENTRY(Mariko));
#undef MAKE_ENTRY

    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto socType = ::nn::spl::GetSocType();
    const auto it = entries.find(socType);
    EXPECT_TRUE(it != entries.end());

    NN_LOG("SocType: %s\n", it->second);
}

TEST(SplBasicTest, GetDramId)
{
    struct Entry
    {
        const char* pName;
        ::nn::spl::HardwareType hardwareType;
    };
    std::map<::nn::spl::DramId, Entry> entries;
#define MAKE_ENTRY(hardware, name) \
    std::make_pair<::nn::spl::DramId, Entry>( \
        ::nn::spl::DramId_ ## hardware ## name, \
        { #hardware #name, ::nn::spl::HardwareType_ ## hardware } \
    )
    entries.insert(MAKE_ENTRY(Icosa, Samsung2g));
    entries.insert(MAKE_ENTRY(Icosa, Hynix));
    entries.insert(MAKE_ENTRY(Icosa, Micron));
    entries.insert(MAKE_ENTRY(Icosa, Samsung3g));
    entries.insert(MAKE_ENTRY(Copper, Samsung2g));
    entries.insert(MAKE_ENTRY(Copper, Hynix));
    entries.insert(MAKE_ENTRY(Copper, Micron));
    entries.insert(MAKE_ENTRY(Iowa, Samsung2g));
    entries.insert(MAKE_ENTRY(Iowa, X3Samsung2g));
    entries.insert(MAKE_ENTRY(Iowa, X3Samsung4g));
    entries.insert(MAKE_ENTRY(Iowa, X3Hynix));
    entries.insert(MAKE_ENTRY(Iowa, X3Micron));
    entries.insert(MAKE_ENTRY(Hoag, Samsung2g));
    entries.insert(MAKE_ENTRY(Hoag, Samsung4g));
    entries.insert(MAKE_ENTRY(Hoag, Hynix));
    entries.insert(MAKE_ENTRY(Hoag, Micron));
#undef MAKE_ENTRY

    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto dramId = ::nn::spl::GetDramId();
    const auto it = entries.find(dramId);
    EXPECT_TRUE(it != entries.end());
    const auto e = it->second;

    auto hardwareType = ::nn::spl::GetHardwareType();
    EXPECT_EQ(e.hardwareType, hardwareType);

    NN_LOG("DramId: %s\n", e.pName);
}

TEST(SplBasicTest, GetDeviceIdLow)
{
    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto deviceIdLow = ::nn::spl::GetDeviceIdLow();
    NN_LOG("DeviceIdLow: %016llx\n", deviceIdLow);
}

TEST(SplBasicTest, GetRetailInteractiveDisplayState)
{
    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto questState = ::nn::spl::GetRetailInteractiveDisplayState();
    EXPECT_TRUE(
        questState == ::nn::spl::RetailInteractiveDisplayState_Disabled
        || questState == ::nn::spl::RetailInteractiveDisplayState_Enabled);

    switch (questState)
    {
    case ::nn::spl::RetailInteractiveDisplayState_Disabled:
        NN_LOG("RetailInteractiveDisplayState: Disabled\n");
        break;
    case ::nn::spl::RetailInteractiveDisplayState_Enabled:
        NN_LOG("RetailInteractiveDisplayState: Enabled\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

TEST(SplBasicTest, GetRegulator)
{
    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    auto regulator = ::nn::spl::GetRegulator();
    EXPECT_TRUE(
        regulator == ::nn::spl::Regulator_Max77621
        || regulator == ::nn::spl::Regulator_Max77812PhaseConfiguration31
        || regulator == ::nn::spl::Regulator_Max77812PhaseConfiguration211);

    switch (regulator)
    {
    case ::nn::spl::Regulator_Max77621:
        NN_LOG("Regulator: Max77621\n");
        break;
    case ::nn::spl::Regulator_Max77812PhaseConfiguration31:
        NN_LOG("Regulator: Max77812PhaseConfiguration31\n");
        break;
    case ::nn::spl::Regulator_Max77812PhaseConfiguration211:
        NN_LOG("Regulator: Max77812PhaseConfiguration211\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto hardwareType = ::nn::spl::GetHardwareType();

    if (hardwareType == ::nn::spl::HardwareType_Iowa)
    {
        EXPECT_TRUE(
            regulator == ::nn::spl::Regulator_Max77812PhaseConfiguration31
            || regulator == ::nn::spl::Regulator_Max77812PhaseConfiguration211);
    }
    else
    {
        EXPECT_TRUE(regulator == ::nn::spl::Regulator_Max77621);
    }
}

TEST(SplBasicTest, ModularExponentiate)
{
    uint8_t modulus[] =
    {
        0xd0,0xb8,0x9c,0x8b,0x00,0x77,0xe6,0xbd,0x8e,0x1c,0xd6,0x24,0xc4,0x0f,0x98,0x00,
        0x59,0x62,0xde,0xc2,0x87,0x0f,0xc4,0x27,0x3d,0x62,0x5f,0xcc,0x6a,0x33,0x2e,0x49,
        0xe1,0x75,0x1e,0x8d,0x02,0xf2,0xa8,0x90,0x47,0xf2,0x64,0xd3,0xbc,0x2b,0x34,0x79,
        0xd0,0x22,0xb8,0x6c,0x62,0x7a,0xaa,0xa9,0x6a,0x1f,0x47,0xbf,0x95,0xdb,0x90,0x88,
        0xc3,0x02,0x8d,0x2c,0x52,0x99,0xa4,0x14,0x34,0xc8,0x78,0xb3,0x56,0xee,0xbb,0xa7,
        0x93,0xd3,0x97,0xea,0x8c,0xd0,0x64,0x5c,0x31,0xfb,0xca,0x06,0x5f,0x33,0x88,0x54,
        0x62,0xbe,0x95,0x4a,0x89,0x8a,0x36,0x2e,0xdb,0x6a,0xd3,0x4b,0xe5,0xe3,0x5e,0xce,
        0x78,0x65,0x9a,0xd2,0x5a,0x6a,0xfc,0x7c,0x7e,0xc4,0x3f,0xe9,0x89,0x7b,0x61,0x06,
        0x4f,0x26,0x00,0xa0,0x20,0x58,0x0e,0x91,0x5e,0x72,0x00,0x01,0x9b,0x17,0x65,0xff,
        0xf7,0x9b,0x41,0x58,0xc0,0x5d,0x78,0xa2,0x43,0x86,0xcc,0x9d,0xa3,0xa7,0x83,0x3d,
        0xbc,0x4c,0x03,0xa1,0x95,0x40,0x82,0x7d,0x57,0x17,0xe0,0x6a,0x13,0xc0,0x79,0x13,
        0x81,0xd6,0xeb,0x12,0x19,0x43,0xf0,0x66,0x66,0x50,0xaa,0xe5,0x02,0x68,0x76,0x3e,
        0x69,0x63,0x3a,0x74,0x7d,0xc3,0xaa,0xed,0xe2,0x9d,0x4d,0x24,0xec,0x08,0xaa,0x0c,
        0xc6,0xa4,0xef,0xb2,0x11,0x96,0xa7,0xbc,0x60,0xb7,0xc5,0x5c,0xc4,0xab,0xe9,0x89,
        0xbf,0x36,0x80,0x7a,0x79,0xeb,0xc8,0xad,0xd9,0xd4,0x2e,0xea,0xf5,0x59,0x8a,0x72,
        0x7e,0xff,0x03,0x97,0x37,0x40,0x02,0xf0,0x30,0x0b,0x77,0x5d,0x78,0x5e,0x7b,0xcf,
    };
    uint8_t publicExponent[] =
    {
        0x01,0x00,0x01,
    };
    uint8_t privateExponent[] =
    {
        0xad,0x73,0x1d,0x34,0x2e,0x59,0x75,0xb8,0xc4,0xa6,0x96,0x16,0xcb,0x8f,0xa1,0xe3,
        0x78,0x64,0xbf,0x75,0xac,0xc3,0x9f,0x60,0x74,0x94,0x91,0xc3,0xcf,0xe5,0x64,0x6c,
        0x25,0x1e,0x2d,0x1b,0x45,0xfd,0xdd,0xb2,0xb8,0xf7,0x68,0x6e,0xc4,0x8a,0xaa,0xfb,
        0x27,0xca,0x12,0xb4,0x8f,0x4a,0x1d,0x14,0xd8,0x89,0xd2,0x18,0x16,0x32,0xf6,0x73,
        0x90,0x8b,0x48,0x42,0x33,0x0a,0x56,0xde,0x2b,0xd7,0x04,0xe7,0x14,0x31,0x24,0x64,
        0xe8,0x66,0x08,0x72,0xf4,0x3e,0x21,0x88,0xb4,0x48,0xe7,0xe4,0x4f,0xad,0x05,0x78,
        0x0a,0xef,0x12,0x25,0x75,0x13,0x14,0x5f,0xf0,0x1b,0x60,0x28,0x4c,0x8c,0x9c,0xb6,
        0xbc,0x3a,0xc3,0x64,0x57,0x9b,0x5c,0xe3,0x20,0x7d,0x7a,0x17,0x5b,0x16,0xda,0xd2,
        0xe0,0x2c,0x6a,0x7b,0x53,0x6f,0xba,0xd2,0x68,0x14,0xc1,0x13,0x95,0x02,0x7c,0xda,
        0xe5,0xf1,0xef,0xad,0xb7,0x25,0x1e,0xb0,0x72,0x7e,0x35,0x68,0xa2,0x53,0x9d,0xda,
        0xf3,0xf4,0x07,0x2a,0x6f,0x26,0xb1,0x13,0x7d,0xec,0x20,0x59,0xe6,0x4b,0xce,0xd5,
        0xcc,0xaa,0x90,0xa6,0x26,0x21,0x48,0x48,0x5b,0x35,0x99,0x9d,0x1b,0x55,0xc2,0x62,
        0x61,0x1c,0xcd,0x27,0x84,0x81,0x32,0x14,0x9e,0xd1,0x22,0x2c,0x93,0xc8,0x00,0xfc,
        0x40,0x4a,0x06,0x15,0xaa,0x32,0x30,0x9b,0xa9,0xcf,0xde,0xd1,0xe5,0x4b,0x1f,0xb7,
        0xc3,0x53,0xde,0x46,0xdf,0xeb,0x32,0xe2,0xe0,0x81,0x40,0x1a,0x59,0x65,0xa9,0x65,
        0xc5,0x90,0x95,0xd6,0x52,0x21,0x19,0xe1,0xfa,0xfc,0x08,0xf6,0xcb,0x7e,0xd4,0x31,
    };

    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    RsaTest(
        modulus,
        publicExponent,
        [&](
            void* pResultBuffer, size_t resultBufferSize,
            const void* pBase, size_t baseSize,
            const void* pModulus, size_t modulusSize) NN_NOEXCEPT
        {
            return ::nn::spl::ModularExponentiate(
                pResultBuffer, resultBufferSize,
                pBase, baseSize,
                privateExponent, sizeof(privateExponent),
                pModulus, modulusSize);
        }
    );
}

TEST(SplBasicTest, DecryptAndStoreGcKey)
{
    ::nn::settings::factory::GameCardKey data;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetGameCardKey(&data));

    ::nn::settings::factory::GameCardCertificate cert;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetGameCardCertificate(&cert));

    uint8_t modulus[256];
    uint8_t publicExponent[3];
    ::std::memcpy(modulus, &cert.data[0x130], sizeof(modulus));
    ::std::memcpy(publicExponent, &cert.data[0x230], sizeof(publicExponent));

    ::nn::spl::InitializeForFs();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::DecryptAndStoreGcKey(
            data.data,
            data.size));

    RsaOaepSha256Test(
        modulus,
        publicExponent,
        ::nn::spl::DecryptGcMessage);
}

TEST(SplBasicTest, SslClientCertKey)
{
    ::nn::settings::factory::SslKey data;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetSslKey(&data));

    ::nn::settings::factory::SslCertificate cert;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetSslCertificate(&cert));

    uint8_t modulus[256];
    uint8_t publicExponent[3];
    ::std::memcpy(modulus, &cert.data[0x158], sizeof(modulus));
    ::std::memcpy(publicExponent, &cert.data[0x25A], sizeof(publicExponent));

    uint32_t generation;
    ::std::memcpy(&generation, &data.data[data.size - sizeof(generation)], sizeof(generation));
    bool isNewVersion = (generation > 0);
    NN_LOG("Generation: %d\n", generation);

    ::nn::spl::InitializeForSsl();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    ::std::function<::nn::Result(
        void* pResultBuffer, size_t resultBufferSize,
        const void* pBase, size_t baseSize,
        const void* pModulus, size_t modulusSize
    )> modularExponentiate;
    uint8_t privateExponent[256];

    if (isNewVersion)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::DecryptAndStoreSslClientCertKey(
                data.data, data.size));
        modularExponentiate = ::nn::spl::ModularExponentiateWithSslClientCertKey;
    }
    else
    {
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::ExtractSslClientCertKey(
                privateExponent, sizeof(privateExponent),
                data.data, data.size));
        modularExponentiate = [&](
            void* pResultBuffer, size_t resultBufferSize,
            const void* pBase, size_t baseSize,
            const void* pModulus, size_t modulusSize) NN_NOEXCEPT
        {
            nn::crypto::RsaCalculator<sizeof(modulus), sizeof(privateExponent)> privateKey;
            privateKey.Initialize(pModulus, modulusSize, privateExponent, sizeof(privateExponent));
            privateKey.ModExp(pResultBuffer, resultBufferSize, pBase, baseSize);
            return ::nn::ResultSuccess();
        };
    }

    RsaTest(
        modulus,
        publicExponent,
        modularExponentiate);
}

TEST(SplBasicTest, ExtractDrmDeviceCertEccKey)
{
    ::nn::settings::factory::EccB233DeviceKey data;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetEciDeviceKey(&data));

    ::nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    uint8_t result[32] = {};
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::ExtractDrmDeviceCertEccKey(
            result,
            sizeof(result),
            data.data,
            data.size));
}

TEST(SplBasicTest, DecryptAndStoreDrmDeviceCertKey)
{
    ::nn::settings::factory::Rsa2048DeviceKey data;
    auto result = ::nn::settings::factory::GetEciDeviceKey(&data);

    if (result.IsFailure())
    {
        NN_LOG("Skip\n");
        return;
    }

    ::nn::settings::factory::Rsa2048DeviceCertificate cert;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetEciDeviceCertificate(&cert));

    uint8_t modulus[256];
    uint8_t publicExponent[4];
    ::std::memcpy(modulus, &cert.data[0x108], sizeof(modulus));
    ::std::memcpy(publicExponent, &cert.data[0x208], sizeof(publicExponent));

    ::nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::DecryptAndStoreDrmDeviceCertKey(
            data.data, data.size));

    RsaTest(
        modulus,
        publicExponent,
        ::nn::spl::ModularExponentiateWithDrmDeviceCertKey);
}

TEST(SplBasicTest, ComputeCtr)
{
    const size_t testSize = 3 * 1024 * 1024;

    void* pSrc = &g_Work[0];
    void* pDst = &g_Work[testSize];
    void* pExp = &g_Work[testSize];

    std::memset(pSrc, 0, testSize);
    std::memset(pDst, 0, testSize);

    nn::crypto::Aes128CtrEncryptor ctr;

    ctr.Initialize(Key, sizeof(Key), Zero, sizeof(Zero));
    ctr.Update(pExp, testSize, pSrc, testSize);

    ::nn::spl::InitializeForCrypto();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    int slotIndex;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::AllocateAesKeySlot(
            &slotIndex));

    LoadKey(slotIndex);

    for( int srcSize = 0; srcSize < 40; ++srcSize )
    {
        for( int dstSize = srcSize; dstSize < 40; ++dstSize )
        {
            const auto r = ::nn::spl::ComputeCtr(pDst, dstSize, slotIndex, pSrc, srcSize, Zero, sizeof(Zero));

            if( (srcSize % 16) != 0 )
            {
                NNT_EXPECT_RESULT_FAILURE(nn::spl::ResultInvalidBufferSize, r);
            }
            else
            {
                NNT_EXPECT_RESULT_SUCCESS(r);
            }
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::ComputeCtr(
            pDst,       testSize,
            slotIndex,
            pSrc,       testSize,
            Zero,       sizeof(Zero) ));

    EXPECT_EQ(::std::memcmp(pDst, pExp, testSize), 0);
}

TEST(SplBasicTest, ComputeCmac)
{
    void* pSrc = &g_Work[0];
    std::memset(g_Work, 0, sizeof(g_Work));

    ::nn::spl::InitializeForCrypto();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    int slotIndex;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::AllocateAesKeySlot(
            &slotIndex));

    LoadKey(slotIndex);

    for( size_t size = 0; size < 600; ++size )
    {
        nn::Bit8 challenge[nn::spl::AesBlockSize];
        nn::Bit8 expected[nn::crypto::Aes128CmacGenerator::MacSize];

        auto r = nn::spl::ComputeCmac(challenge, sizeof(challenge), slotIndex, pSrc, size);

        NNT_EXPECT_RESULT_SUCCESS(r);

        nn::crypto::GenerateAes128Cmac(expected, sizeof(expected), pSrc, size, Key, sizeof(Key));

        EXPECT_EQ(::std::memcmp(challenge, expected, sizeof(expected)), 0);
    }
}

TEST(SplBasicTest, LoadEsDeviceKey)
{
    ::nn::settings::factory::Rsa2048DeviceKey data;
    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::settings::factory::GetEticketDeviceKey(&data));

    ::nn::spl::InitializeForEs();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    NNT_EXPECT_RESULT_SUCCESS(
        ::nn::spl::LoadEsDeviceKey(
            data.data, data.size));
}

// FIPS 140-2
TEST(SplBasicTest, StatisticalRandomNumberGeneratorTest)
{
    uint8_t data[20000 / 8];

    ::nn::spl::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    for (auto i = 0; i < sizeof(data); i += sizeof(data) / 10)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            ::nn::spl::GenerateRandomBytes(
                &data[i], sizeof(data) / 10));
    }

    RunMonobitTest(data, sizeof(data));
    RunRunsTest(data, sizeof(data));
    RunLongRunsTest(data, sizeof(data));
    RunPokerTest(data, sizeof(data));
}

TEST(SplBasicTest, StatisticalRawRandomNumberGeneratorTest)
{
    uint8_t data[20000 / 8];

    const size_t bufferSize = sizeof(nn::Bit64) * 7;
    for (auto i = 0; i < sizeof(data); i += bufferSize)
    {
        EXPECT_EQ(
            nn::spl::smc::GenerateRandomBytes(&data[i], std::min(sizeof(data) - i, bufferSize)),
            nn::spl::smc::SmcResult_Success);
    }

    RunMonobitTest(data, sizeof(data));
    RunRunsTest(data, sizeof(data));
    RunLongRunsTest(data, sizeof(data));
    RunPokerTest(data, sizeof(data));
}
