﻿using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Reactive.Bindings;
using System;
using System.Diagnostics;
using System.Reactive.Linq;
using System.Text;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Reactive.Concurrency;
using ToolVersionUpdater.Properties;
using System.IO;
using System.Collections.Generic;
using System.Runtime.Serialization;

namespace ToolVersionUpdater
{
    public enum UIType
    {
        FilePath,
        DirectoryPath,
    }

    [DataContract]
    public class ToolInfo
    {
        [DataMember(Order = 0)]
        public string ToolPath { get; set; } = string.Empty;

        [DataMember(Order = 1)]
        public string AssemblyInfoPath { get; set; } = string.Empty;

        [DataMember(Order = 2)]
        public string[] SourcePaths { get; set; } = null;
    }

    [DataContract]
    public class ToolVersionCheckInfo
    {
        [DataMember(Order = 0)]
        public ToolInfo[] TargetToolInfos { get; set; }
    }

    public class MainViewModel : ObservableObject
    {
        private ObservableList<IParam> versionCheckOptions = new ObservableList<IParam>();
        private ObservableList<IParam> versionUpdateOptions = new ObservableList<IParam>();
        private ObservableList<IParam> globalOptions = new ObservableList<IParam>();
        private ObservableList<IParam> debugSettings = new ObservableList<IParam>();
        private ToolVersionCheckInfo toolInfo = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MainViewModel()
        {
            // デバッグ設定
            {
                var sdkReposPathParam = new ParamString()
                {
                    ID = "sdk_repos_path",
                    Name = "SDK リポジトリパス",
                    IsReadOnly = true,
                };
                sdkReposPathParam.Value.Value = !string.IsNullOrEmpty(Settings.Default.SdkReposPath) ?
                    Settings.Default.SdkReposPath : FindSigloRootFolder();
                this.debugSettings.Add(sdkReposPathParam);
            }

            {
                var gitPathParam = new ParamString()
                {
                    ID = "git_path",
                    Name = "git.exe パス",
                    UIType = UIType.FilePath.ToString(),
                    UIParam = "git.exe|git.exe",
                };
                gitPathParam.Value.Value = !string.IsNullOrEmpty (Settings.Default.GitPath) ?
                    Settings.Default.GitPath : "C:/Program Files/Git/bin/git.exe";
                this.versionCheckOptions.Add(gitPathParam);

                var oldSdkRootParam = new ParamString()
                {
                    ID = "old_sdk_root_path",
                    Name = "前回のリリースパッケージパス",
                    UIType = UIType.DirectoryPath.ToString(),
                };
                oldSdkRootParam.Value.Value = !string.IsNullOrEmpty(Settings.Default.OldSdkPackagePath) ?
                    Settings.Default.OldSdkPackagePath : string.Empty;
                this.versionCheckOptions.Add(oldSdkRootParam);

                var oldReleaseBranchParam = new ParamString()
                {
                    ID = "old_sdk_branch",
                    Name = "前回のリリースブランチ",
                };
                oldReleaseBranchParam.Value.Value = "origin/release/sdk/v1.0";
                oldReleaseBranchParam.Value.Value = !string.IsNullOrEmpty(Settings.Default.OldSdkReleaseBranchName) ?
                    Settings.Default.OldSdkReleaseBranchName : "origin/release/sdk/v1.0";
                this.versionCheckOptions.Add(oldReleaseBranchParam);
            }

            {
                var targetToolDefineParam = new ParamString()
                {
                    ID = "target_tool_define_path",
                    Name = "対象ツール定義パス",
                    UIType = UIType.FilePath.ToString(),
                    UIParam = "対象ツール定義|*.json",
                };
                string defaultTargetToolDefinePath = System.IO.Path.Combine(
                    FindSdkRepositoryRootPath(), "Tests/Sources/Tools/ToolVersionChecker/3dTargetToolDefine.json");
                targetToolDefineParam.Value.Value =
                    !string.IsNullOrEmpty(Settings.Default.TargetToolDefinePath) && System.IO.File.Exists(Settings.Default.TargetToolDefinePath) ?
                    Settings.Default.TargetToolDefinePath : defaultTargetToolDefinePath;
                targetToolDefineParam.Value.PropertyChanged += TargetToolDefine_PropertyChanged;
                this.globalOptions.Add(targetToolDefineParam);
            }

            CreateToolVersionUpdateOptions();

            this.BuildToolVersionCheckerCommand.ObserveOn(Scheduler.Default).Subscribe(_ =>
            {
                BuildByNact("Tests/Sources/Tools/ToolVersionChecker");
            });

            this.BuildNintendoWareGraphicsCommand.ObserveOn(Scheduler.Default).Subscribe(_ =>
            {
                BuildByNact("Programs/NintendoWare/Sources/Tools/Graphics");
            });

            this.Check3dToolsVersionCommand.ObserveOn(Scheduler.Default).Subscribe(_ =>
            {
                string sdkReposPath = FindSdkRepositoryRootPath();
                if (!System.IO.Directory.Exists(sdkReposPath))
                {
                    WriteErrorLine($"SDK ルート \"{sdkReposPath}\" が見つかりません。");
                    return;
                }

                string toolVersionCheckerPath = System.IO.Path.Combine(
                    sdkReposPath,
                    "Tests/Outputs/AnyCPU/Tools/ToolVersionChecker/Release/ToolVersionChecker.exe");
                if (!System.IO.File.Exists(toolVersionCheckerPath))
                {
                    string toolVersionCheckerSourcePath = System.IO.Path.Combine(
                        sdkReposPath,
                        "Tests/Sources/Tools/ToolVersionChecker");
                    WriteErrorLine($"\"{toolVersionCheckerPath}\" が見つかりません。\"{toolVersionCheckerSourcePath}\" をビルドして下さい。");
                    return;
                }

                string targetToolDefinitionFile = FindTargetToolDefinePath();
                if (!System.IO.File.Exists(targetToolDefinitionFile))
                {
                    WriteErrorLine($"対象ツール定義 \"{targetToolDefinitionFile}\" が見つかりません。");
                    return;
                }

                WriteLogLine("チェック開始");
                string gitPath = (versionCheckOptions.First(param => param.ID == "git_path") as ParamString).Value.Value;
                string oldSdkRootPath = (versionCheckOptions.First(param => param.ID == "old_sdk_root_path") as ParamString).Value.Value;
                string oldSdkBranchName = (versionCheckOptions.First(param => param.ID == "old_sdk_branch") as ParamString).Value.Value;
                StringBuilder args = new StringBuilder();
                args.Append($" --repository-root \"{sdkReposPath}\"");
                args.Append($" --target-tool-definition-file \"{targetToolDefinitionFile}\"");
                args.Append($" --git-path \"{gitPath}\"");
                args.Append($" --old-sdk-root \"{oldSdkRootPath}\"");
                args.Append($" --old-sdk-branch \"{oldSdkBranchName}\"");
                ExecuteProcess(toolVersionCheckerPath, args.ToString());
            });

            this.UpdateMinorVersionCommand.ObserveOn(Scheduler.Default).Subscribe(_ =>
            {
                UpdateToolVersionOfSelectedTools(0, 1);
            });

            this.UpdateMajorVersionCommand.ObserveOn(Scheduler.Default).Subscribe(_ =>
            {
                UpdateToolVersionOfSelectedTools(1, 0);
            });
        }

        public void SaveSettings()
        {
            Settings.Default.GitPath = (versionCheckOptions.First(param => param.ID == "git_path") as ParamString).Value.Value;
            Settings.Default.OldSdkPackagePath = (versionCheckOptions.First(param => param.ID == "old_sdk_root_path") as ParamString).Value.Value;
            Settings.Default.OldSdkReleaseBranchName = (versionCheckOptions.First(param => param.ID == "old_sdk_branch") as ParamString).Value.Value;
            Settings.Default.TargetToolDefinePath = (globalOptions.First(param => param.ID == "target_tool_define_path") as ParamString).Value.Value;
            Settings.Default.SdkReposPath = FindSdkRepositoryRootPath();
            Settings.Default.Save();
        }

        public ReactiveProperty<string> LogText { get; } = new ReactiveProperty<string>();

        public ObservableList<IParam> VersionCheckOptions
        {
            get
            {
                return this.versionCheckOptions;
            }
        }

        public ObservableList<IParam> GlobalOptions
        {
            get
            {
                return this.globalOptions;
            }
        }

        public ObservableList<IParam> VersionUpdateOptions
        {
            get
            {
                return this.versionUpdateOptions;
            }
        }

        public ObservableList<IParam> DebugSettings
        {
            get
            {
                return this.debugSettings;
            }
        }

        public ReactiveCommand BuildToolVersionCheckerCommand { get; } = new ReactiveCommand();

        public ReactiveCommand BuildNintendoWareGraphicsCommand { get; } = new ReactiveCommand();

        public ReactiveCommand Check3dToolsVersionCommand { get; } = new ReactiveCommand();

        public ReactiveCommand UpdateMinorVersionCommand { get; } = new ReactiveCommand();

        public ReactiveCommand UpdateMajorVersionCommand { get; } = new ReactiveCommand();

        private bool ExecuteProcess(string cmd, string args = "", bool useShellExecute = false)
        {
            WriteLogLine($"{cmd} {args}");
            try
            {
                int result = ExecuteProcessImpl(cmd, args, useShellExecute);
                if (result != 0)
                {
                    WriteErrorLine($"{cmd} {args} の実行に失敗しました。");
                    return false;
                }
            }
            catch (Exception e)
            {
                WriteErrorLine($"{cmd} {args} の実行に失敗しました。\n{e.Message}");
                return false;
            }

            return true;
        }

        private void TargetToolDefine_PropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            CreateToolVersionUpdateOptions();
        }

        private string FindTargetToolDefinePath()
        {
            return (this.globalOptions.First(x => x.ID == "target_tool_define_path") as ParamString).Value.Value;
        }

        private void CreateToolVersionUpdateOptions()
        {
            string targetToolDefinitionFile = FindTargetToolDefinePath();
            if (!System.IO.File.Exists(targetToolDefinitionFile))
            {
                WriteErrorLine($"{targetToolDefinitionFile} が見つかりませんでした。");
                return;
            }

            try
            {
                toolInfo = DeserializeTargetToolInfos(targetToolDefinitionFile);
            }
            catch (Exception exception)
            {
                WriteErrorLine($"{targetToolDefinitionFile} のデシリアライズに失敗しました。\n{exception.ToString()}");
                return;
            }

            this.versionUpdateOptions.Clear();
            foreach (var info in this.toolInfo.TargetToolInfos)
            {
                // とりあえず、一番最初に定義されたソースコードにアセンブリ情報があるとみなす
                string mainSourceCodePath = System.IO.Path.Combine(FindSdkRepositoryRootPath(), info.SourcePaths.First());
                if (!System.IO.Directory.Exists(mainSourceCodePath))
                {
                    WriteErrorLine($"{mainSourceCodePath} が見つかりませんでした。");
                    continue;
                }

                string[] assemblyInfoPaths = System.IO.Directory.GetFiles(
                    mainSourceCodePath, "*AssemblyInfo.cs", System.IO.SearchOption.AllDirectories);
                if (assemblyInfoPaths.Length == 0)
                {
                    continue;
                }

                info.AssemblyInfoPath = assemblyInfoPaths.First();
                string toolName = System.IO.Path.GetFileNameWithoutExtension(info.ToolPath);
                this.versionUpdateOptions.Add(new ParamBool()
                {
                    ID = info.ToolPath,
                    Name = toolName,
                });
            }
        }

        private void WriteLog(string message)
        {
            this.LogText.Value += message;
            Console.WriteLine(message);
        }

        private void WriteLogLine(string message)
        {
            WriteLog(message + "\n");
        }

        private void WriteErrorLine(string message)
        {
            WriteLogLine("エラー: " + message);
        }

        [Conditional("DEBUG")]
        private void WriteDebugLogLine(string message)
        {
            WriteLogLine($"デバッグ: {message}\n");
        }

        private int ExecuteProcessImpl(string cmd, string args = "", bool useShellExecute = false)
        {
            string stdout, stderr;
            Utility.ExecuteCommand(out stdout, out stderr, cmd, args, Encoding.Default);

            var info = new ProcessStartInfo(cmd, args) { UseShellExecute = useShellExecute };
            if (!info.UseShellExecute)
            {
                info.RedirectStandardError = true;
                info.RedirectStandardOutput = true;
                info.CreateNoWindow = true;
            }

            var process = new Process { StartInfo = info };
            process.OutputDataReceived += (s, e) => LogText.Value += (e.Data + Environment.NewLine);
            process.ErrorDataReceived += (s, e) => LogText.Value += (e.Data + Environment.NewLine);
            process.Start();

            if (!info.UseShellExecute)
            {
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
            }

            process.WaitForExit();

            return process.ExitCode;
        }

        private string FindSdkRepositoryRootPath()
        {
            return (debugSettings.First(param => param.ID == "sdk_repos_path") as ParamString).Value.Value;
        }

        private static ToolVersionCheckInfo DeserializeTargetToolInfos(string path)
        {
            StreamReader reader = new StreamReader(path);
            string jsonString = reader.ReadToEnd();
            reader.Close();

            return JsonSerializerHelper.Deserialize<ToolVersionCheckInfo>(jsonString);
        }

        private ToolInfo FindToolInfo(string toolPath)
        {
            foreach (var info in this.toolInfo.TargetToolInfos)
            {
                if (info.ToolPath == toolPath)
                {
                    return info;
                }
            }

            return null;
        }

        private string FindSigloRootFolder()
        {
            string[] pathElems = System.Reflection.Assembly.GetExecutingAssembly().Location.Split(new char[] { '/', '\\' });
            for (int endIndex = pathElems.Length - 1; endIndex > 0; --endIndex)
            {
                string currentFolder = string.Join("/", pathElems, 0, endIndex);
                string sigloRootMarkPath = System.IO.Path.Combine(currentFolder, "SigloRootMark");
                if (System.IO.File.Exists(sigloRootMarkPath))
                {
                    return currentFolder;
                }
            }

            return string.Empty;
        }

        private void UpdateToolVersionOfSelectedTools(int majorVerIncrement, int minorVerIncrement)
        {
            StringBuilder args = new StringBuilder();
            string sdkReposPath = FindSdkRepositoryRootPath();
            if (!System.IO.Directory.Exists(sdkReposPath))
            {
                WriteErrorLine($"SDK ルート \"{sdkReposPath}\" が見つかりません。");
                return;
            }

            foreach (IParam option in this.VersionUpdateOptions)
            {
                ParamBool boolOption = option as ParamBool;
                if (boolOption != null && !boolOption.Value.Value)
                {
                    // チェックされていない
                    continue;
                }

                ToolInfo info = FindToolInfo(option.ID);
                UpdateToolVersion(System.IO.Path.Combine(sdkReposPath, info.AssemblyInfoPath), majorVerIncrement, minorVerIncrement, 0);
            }
        }

        private void UpdateToolVersion(string assemblyInfoPath, int majorVerIncrement, int minorVerIncrement, int microVerIncrement)
        {
            StreamReader streamReader = new StreamReader(assemblyInfoPath);
            string assemblyInfoText = streamReader.ReadToEnd();
            streamReader.Close();

            string[] lines = assemblyInfoText.Split(new string[] { "\r\n", "\n", "\r" }, StringSplitOptions.RemoveEmptyEntries);
            string oldVersion = string.Empty;
            string newVersion = string.Empty;
            foreach (string line in lines)
            {
                string code = line;
                int commentOutIndex = line.IndexOf("//");
                if (commentOutIndex >= 0)
                {
                    code = line.Substring(0, commentOutIndex);
                }

                int index = code.IndexOf("AssemblyInformationalVersion");
                if (index < 0)
                {
                    continue;
                }

                if (line.Contains("AssemblyInformationalVersion"))
                {
                    string versionAsString = line.Replace("[assembly: AssemblyInformationalVersion(\"", string.Empty);
                    versionAsString = versionAsString.Replace("\")]", string.Empty);
                    string[] splitedVersionStrings = versionAsString.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries);
                    int majorVersion = int.Parse(splitedVersionStrings[0]);
                    int minorVersion = int.Parse(splitedVersionStrings[1]);
                    int microVersion = int.Parse(splitedVersionStrings[2]);
                    oldVersion = $"{majorVersion}.{minorVersion}.{microVersion}";

                    if (majorVerIncrement > 0)
                    {
                        minorVersion = 0;
                    }

                    if (minorVerIncrement > 0)
                    {
                        microVersion = 0;
                    }

                    newVersion = $"{majorVersion + majorVerIncrement}.{minorVersion + minorVerIncrement}.{microVersion + microVerIncrement}";
                }
            }

            if (string.IsNullOrEmpty(oldVersion))
            {
                return;
            }

            StreamWriter streamWriter = new StreamWriter(assemblyInfoPath, false, Encoding.UTF8);
            WriteLogLine($"{assemblyInfoPath}\n  Replace {oldVersion} to {newVersion}");
            string writeText = assemblyInfoText.Replace(oldVersion, newVersion);
            streamWriter.Write(writeText);
            streamWriter.Close();
        }

        private void BuildByNact(string targetFolderPathFromReposRoot)
        {
            string sdkReposPath = FindSdkRepositoryRootPath();
            if (!System.IO.Directory.Exists(sdkReposPath))
            {
                WriteErrorLine($"SDK ルート \"{sdkReposPath}\" が見つかりません。");
                return;
            }

            string nactPath = System.IO.Path.Combine(
                sdkReposPath,
                "Integrate/CommandLineTools/nact.exe");
            string nwGraphicsToolsPath = System.IO.Path.Combine(
                sdkReposPath,
                targetFolderPathFromReposRoot);
            StringBuilder args = new StringBuilder();
            args.Append($"-m --continue_on_error -j 4 --print-eval-trace -d \"{nwGraphicsToolsPath}\"");
            ExecuteProcess(nactPath, args.ToString());
        }
    }
}
