﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Microsoft.VSSDK.Tools.VsIdeTesting;
using Nintendo.NintendoSdkVsExtension.Base;
using Nintendo.NintendoSdkVsExtension.Logic;
using Nintendo.NintendoSdkVsExtension.Models;
using Nintendo.NintendoSdkVsExtension.Shell;
using Nintendo.NintendoSdkVsExtension.VcAccessors;
using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml.Linq;

namespace Nintendo.NintendoSdkVsExtension.UnitTest.LogicTest
{
    [TestClass]
    [DeploymentItem(InIdeTestConstants.VsVersionSpecificTestDataSourceDirectory + @"OldProject.Generic", @"Data\OldProject.Generic")]
    [DeploymentItem(InIdeTestConstants.VsVersionSpecificTestDataSourceDirectory + @"DuplicatedOrMissingProjectGroup", @"Data\DuplicatedOrMissingProjectGroup")]
    public class SdkConfigurationManagerOperationTest
    {
        private static readonly string OldProjectDir = @"Data\OldProject.Generic";
        private static readonly string OldProjectSolutionName = @"OldProject.sln";
        private static readonly string DuplicatedOrMissingProjectGroupDir = @"Data\DuplicatedOrMissingProjectGroup";
        private static readonly string DuplicatedOrMissingProjectGroupSolutionName = @"DuplicatedOrMissingProjectGroup.sln";

        public TestContext TestContext { get; set; }

        [TestMethod]
        [HostType("VS IDE")]
        public void ShowsConfirmUpgradingDialogForOldProject()
        {
            ExecuteWithCopiedSolutionOpen(OldProjectDir, OldProjectSolutionName, (string solutionDirectory) =>
            {
                var mock = new UIMock()
                {
                    QueryConfirmUpgradingValue = false,
                };
                var sut = new SdkConfigurationManagerOperation(VsIdeTestHostContext.ServiceProvider, mock);

                sut.Execute();

                Assert.IsTrue(mock.IsQueryConfirmUpgradingCalled);
                Assert.IsTrue(!mock.IsEditSdkConfigurationCalled);
                Assert.IsTrue(!mock.IsWarnFailedToApplyConfigurationCalled);
            });
        }

        [TestMethod]
        [HostType("VS IDE")]
        public void CanApplyConfigurationToDuplicatedProjectGroupElements()
        {
            ExecuteWithCopiedSolutionOpen(DuplicatedOrMissingProjectGroupDir, DuplicatedOrMissingProjectGroupSolutionName, (string solutionDirectory) =>
            {
                var mock = new UIMock()
                {
                    QueryConfirmUpgradingValue = false,
                    EditSdkConfigurationImpl = (sol) =>
                    {
                        var proj = sol.VcProjects.Single();
                        // Release|x64 の ProjectGroupElement Label="Configuration" が複数存在する。
                        // そのようなプロジェクトに対して設定を反映できることを確認する。
                        var conf = proj.ProjectConfigurations.Single(x => x.Configuration == "Release" && x.Platform == "x64");
                        EnableNintendoSdk(conf);
                        return true;
                    },
                };
                var sut = new SdkConfigurationManagerOperation(VsIdeTestHostContext.ServiceProvider, mock);

                sut.Execute();

                Assert.IsTrue(!mock.IsQueryConfirmUpgradingCalled);
                Assert.IsTrue(mock.IsEditSdkConfigurationCalled);
                Assert.IsTrue(!mock.IsWarnFailedToApplyConfigurationCalled);
                {
                    var vcProject = VsSolutionUtil.EnumerateVcxproj(VsIdeTestHostContext.ServiceProvider)
                        .Select(x => new VcProjectAccessor(x))
                        .Single();
                    var conf = vcProject.Configurations
                        .Single(x => x.ConfigurationPair == new ConfigurationPair("Release", "x64"));
                    Assert.AreEqual(conf.GetEvaluatedPropertyValue("NintendoSdkSpec"), "Generic");
                    Assert.AreEqual(conf.GetEvaluatedPropertyValue("NintendoSdkBuildType"), "Release");
                }
            });
        }

        [TestMethod]
        [HostType("VS IDE")]
        public void WarnOutMissingProjectGroupElements()
        {
            ExecuteWithCopiedSolutionOpen(DuplicatedOrMissingProjectGroupDir, DuplicatedOrMissingProjectGroupSolutionName, (string solutionDirectory) =>
            {
                var mock = new UIMock()
                {
                    QueryConfirmUpgradingValue = false,
                    EditSdkConfigurationImpl = (sol) =>
                    {
                        var proj = sol.VcProjects.Single();
                        // Debug|x64 の ProjectGroupElement Label="Configuration" が存在しない。
                        // そのようなプロジェクトに対して警告メッセージが出力されることを確認する。
                        var conf = proj.ProjectConfigurations.Single(x => x.Configuration == "Debug" && x.Platform == "x64");
                        EnableNintendoSdk(conf);
                        return true;
                    },
                };
                var sut = new SdkConfigurationManagerOperation(VsIdeTestHostContext.ServiceProvider, mock);

                sut.Execute();

                Assert.IsTrue(!mock.IsQueryConfirmUpgradingCalled);
                Assert.IsTrue(mock.IsEditSdkConfigurationCalled);
                Assert.IsTrue(mock.IsWarnFailedToApplyConfigurationCalled);
                CollectionAssert.AreEquivalent(
                    mock.FailedToApplyConfigurations.Select(x => x.ConfigurationPair).ToArray(),
                    new[] { new ConfigurationPair("Debug", "x64") });
            });
        }

        // テスト用ソリューションをコピーして開き、テストコードを実行する
        private void ExecuteWithCopiedSolutionOpen(string solutionDir, string solutionName, Action<string> a)
        {
            using (var td = new TemporaryTestDirectory(TestContext))
            {
                try
                {
                    TestUtil.CopyDirectory(solutionDir, td.FullName);
                    VsTestUtil.EnsureOpenSolution(Path.Combine(td.FullName, solutionName));

                    a(td.FullName);
                }
                finally
                {
                    VsTestUtil.ForceCloseSolution();
                }
            }
        }

        private void EnableNintendoSdk(SdkVcProjectConfiguration conf)
        {
            conf.IsSdkEnabled = true;
            conf.Spec = "Generic";
            conf.SdkRoot = new SdkRoot(SdkRootKind.DirectProperty, TestUtil.GetSdkRoot(TestContext));
            conf.BuildType = SdkBuildType.Release;
        }

        private class UIMock : SdkConfigurationManagerOperation.ISdkConfigurationManagerUI
        {
            public bool IsQueryConfirmUpgradingCalled { get; private set; }
            public bool QueryConfirmUpgradingValue { get; set; } = true;

            public bool IsEditSdkConfigurationCalled { get; private set; }
            public Func<SdkSolution, bool> EditSdkConfigurationImpl { get; set; }

            public bool IsWarnFailedToApplyConfigurationCalled { get; private set; }
            public IEnumerable<FailedToApplyConfiguration> FailedToApplyConfigurations { get; private set; }

            public UIMock()
            {
            }

            public bool QueryConfirmUpgrading()
            {
                IsQueryConfirmUpgradingCalled = true;
                return QueryConfirmUpgradingValue;
            }

            public bool EditSdkConfiguration(SdkSolution sol)
            {
                IsEditSdkConfigurationCalled = true;
                return EditSdkConfigurationImpl?.Invoke(sol) ?? true;
            }

            public void WarnFailedToApplyConfiguration(IEnumerable<FailedToApplyConfiguration> failedToApplyConfigurations)
            {
                IsWarnFailedToApplyConfigurationCalled = true;
                FailedToApplyConfigurations = failedToApplyConfigurations;
            }
        }
    }
}
