# NintendoSdkVsExtension のテストを実施します
# VS2013, VS2015 用のプロジェクトファイルを更新したのち、 VS2013 VS2015 VS2017 それぞれの IDE 内でテストを実行します

$ErrorActionPreference = "Stop"

$scriptPath = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [IO.Path]::GetDirectoryName($ScriptPath)
$sigloRoot = (Join-Path $scriptDirectoryPath ../../../.. | Resolve-Path).Path
$moduleRoot = "${sigloRoot}\Integrate\Scripts\Modules\"
$outputRoot = "${sigloRoot}\Tests\Outputs\AnyCPU\Tools\NintendoSdkVsExtension"

Import-Module "${moduleRoot}\Path"

# IDE のパスを取得
function GetVsIDEPath()
{
    param(
        [string] $version
    )

    Get-SoftwareRegistryValue "Microsoft\VisualStudio\${version}.0\" "InstallDir"
}

function Get-Vs15Path
{
    $locatorPath = Resolve-Path (Join-Path (Get-NintendoSdkRootPath) "Integrate\CommandLineTools\LocateVsInstallation.exe")
    $vsPath = & $locatorPath --full 2> $null
    if ($LastExitCode)
    {
        throw "Visual Studio 15 not found."
    }

    $vsPath
}

# MSTest のパスを取得
function GetMsTestPath()
{
    param(
        [string] $version
    )

    if ($version -eq "15")
    {
        Join-Path (Get-Vs15Path) "Common7\IDE\MSTest.exe"
    }
    else
    {
        $installDir = GetVsIDEPath $version

        Join-Path $installDir "MSTest.exe"
    }
}

# VSIXInstaller のパスを取得
function GetVsixInstallerPath()
{
    Join-Path (Get-Vs15Path) "Common7\IDE\VSIXInstaller.exe"
}

#
# プロジェクトファイルを更新
#
& ${scriptDirectoryPath}\Update-Projects.ps1

$testSucceeded = $true

# MSBuild.exe のパス: VSSDK のタスクがネイティブ 32-bit コードを含んでいるため、 32-bit の MSBuild を使う必要がある
$msbuild = Join-Path (Get-Vs15Path) "MSBuild\15.0\Bin\MSBuild.exe"
$vsixInstaller = GetVsixInstallerPath

#
# NintendoSdkVsExtension をアンインストールしておく
# インストールされていると、インストールされたアセンブリが使用される可能性があるため
#
$p = Start-Process -Wait -PassThru -FilePath $vsixInstaller -ArgumentList ("/u:NintendoSdkVsExtension.Nintendo.07f3506c-f4a5-4c70-8a81-93e25dcbc050", "/q")
if ($p.ExitCode -ne 0 -and $p.ExitCode -ne 1002 -and $p.ExitCode -ne 2003) # 1002, 2003: 「拡張がインストールされていない」エラーを無視する
{
    Write-Host "VSIX uninstall failure" -ForegroundColor "Red"
    exit 1
}
Write-Host "NintendoSDKVsExtension.vsix uninstalled" -ForegroundColor "Green"

#
# MSBuild および MSTest 実行
#

& $msbuild /nologo /v:Quiet /p:Configuration=Release /p:Platform=AnyCPU "NintendoSdkVsExtension.UnitTest\NintendoSdkVsExtension.UnitTest.csproj"
 
$mstest = GetMsTestPath "15"
& $mstest /nologo /detail:errormessage /testcontainer:${outputRoot}\NintendoSdkVsExtension.UnitTest\Release\Nintendo.NintendoSdkVsExtension.UnitTest.dll
$testSucceeded = $testSucceeded -and ($LastExitCode -eq 0) 

foreach ($ver in ("14", "15"))
{
    $mstest = GetMsTestPath $ver
    & $msbuild /nologo /v:Quiet /p:Configuration=Release.${ver} /p:Platform=AnyCPU "NintendoSdkVsExtension.UnitTest.InIDE\NintendoSdkVsExtension.UnitTest.InIDE.${ver}.csproj"
    & $mstest /nologo /detail:errormessage /testcontainer:${outputRoot}\NintendoSdkVsExtension.UnitTest.InIDE\Release.${ver}\Nintendo.NintendoSdkVsExtension.UnitTest.InIDE.${ver}.dll
    $testSucceeded = $testSucceeded -and ($LastExitCode -eq 0) 
}

if ($testSucceeded)
{
    Write-Host "Test SUCCEEDED!" -ForegroundColor "Green"
    exit 0
}
else
{
    Write-Host "******************************" -ForegroundColor "Red"
    Write-Host "Test FAILED!" -ForegroundColor "Red"
    Write-Host "******************************" -ForegroundColor "Red"
    Write-Host "If Visual Studio failed to initialize correctly, try manually launching VS2015 and VS2017 once."
    exit 1
}