﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Log.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/util/util_FormatString.h>

#include "../TestApplicationSimple/TestAppSimple_FsUtilities.h"

#include "TestAppSmallSimple_CommonUtil.h"

namespace {
    const char* PostfixHashFile = ":/HashVal.txt";
} // namespace

bool CommonUtil::IsExistExpectedHashFile(const std::string& inTargetMountName) NN_NOEXCEPT
{
    const auto hashFilePath = inTargetMountName + PostfixHashFile;
    return fsutil::IsExistPath(hashFilePath.c_str());
}

nn::Result CommonUtil::ReadExpectedHashValue(char outVal[], const std::string& inTargetMountName) NN_NOEXCEPT
{
    nn::Result result;
    fsutil::File file;

    const auto hashFilePath = inTargetMountName + PostfixHashFile;
    result = file.Open(hashFilePath.c_str(), nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[Error] CommonUtil::ReadExpectedHashValue() : OpenFile() Fialed : Path = Contents:/HashVal.txt\n");
        return result;
    }

    auto fileSize = file.GetSize();
    result = file.GetLastResult();
    if (result.IsFailure() || fileSize <= 0 || fileSize <= (HASH_STRING_SIZE))
    {
        NN_LOG("[Error] CommonUtil::ReadExpectedHashValue() : GetFileSize() Failed : fileSize = %lld\n", fileSize);
        return result;
    }

    std::unique_ptr<char[]> buf(new char[HASH_STRING_SIZE + 1]);
    buf[HASH_STRING_SIZE] = '\0';

    size_t outSize = 0;
    auto readSize = HASH_STRING_SIZE;
    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    result = file.Read(&outSize, 3, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        NN_LOG("[Error] CommonUtil::ReadExpectedHashValue() : ReadFile() Failed : readSize = %d\n", readSize);
        return result;
    }

    //NN_LOG("buf = \"%s\"\n", buf);
    memcpy(outVal, buf.get(), outSize);
    outVal[HASH_STRING_SIZE] = '\0';

    return result;
}

nn::Result CommonUtil::GetFileDataHashValue(char outVal[], const std::string& inTargetFilePath) NN_NOEXCEPT
{
    nn::Result result;
    fsutil::File file;

    result = file.Open(inTargetFilePath.c_str(), nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[Error] CommonUtil::GetFileDataHashValue() : OpenFile() Fialed : Path = Contents:/DummyData.dat\n");
        return result;
    }

    auto fileSize = file.GetSize();
    result = file.GetLastResult();
    if (result.IsFailure() || fileSize < 0)
    {
        NN_LOG("[Error] CommonUtil::GetFileDataHashValue() : GetFileSize() Failed : fileSize = %lld\n", fileSize);
        return result;
    }

    const size_t workBufSize = 256 * 1024;
    std::unique_ptr<uint8_t[]> workBuf(new uint8_t[workBufSize]);

    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();

    auto restSize = fileSize;
    size_t outSize = 0;
    int64_t offset = 0;
    size_t loopCount = 0;
    do
    {
        // ファイル読み込み
        auto readSize = (restSize < workBufSize) ? static_cast<size_t>(restSize) : workBufSize;
        result = file.Read(&outSize, offset, workBuf.get(), readSize);
        if (result.IsFailure())
        {
            // ファイル読み込み処理で失敗
            break;
        }
        restSize -= outSize;
        offset += outSize;

        // プログレス表示用ログ出力
        // (ROMサイズが大きい場合、計算に時間がかかるので念のため計算中であることを出力する)
        if (loopCount > 320)
        {
            NN_LOG("  ...  Hash Calculating : %d %%\n", static_cast<int>((offset * 100) / fileSize));
            loopCount = 0;
        }

        // ハッシュの更新
        if (outSize > 0)
        {
            sha1.Update(workBuf.get(), outSize);
        }

        ++loopCount;
    } while (restSize > 0 && outSize > 0);

    file.Close();

    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗していたら抜ける
        NN_LOG("[Error] CommonUtil::GetFileDataHashValue() : ReadFile() Failed\n");
        return result;
    }

    // ハッシュ値(SHA1のチェックサム)を計算
    uint8_t hashVal[sha1.HashSize] = { 0 };
    sha1.GetHash(hashVal, sha1.HashSize);

    // ハッシュ値160bit(20byte)を16進文字列表記した40文字に加工する。
    for (int i = 0; i < sha1.HashSize; ++i)
    {
        nn::util::SNPrintf(outVal + (2 * i), 3, "%02X", hashVal[i]);
    }
    // 念のためヌル文字を設定しておく
    outVal[HASH_STRING_SIZE] = '\0';

    return result;
}
