﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include "TestAppSimple_Utility.h"
#include "TestAppSimple_Scene.h"
#include "TestAppSimple_HidController.h"

struct NextActionPair
{
    const NextAction Action;
    const char* Label;
};

// ページ順に登録する
constexpr NextActionPair ActionPairList[] = {
    { NextAction::MetaInfo, "MetaInfo" },
    { NextAction::RomHash,  "RomHash" },
    { NextAction::Account, "Account" },
    { NextAction::SaveData, "SaveData" },
    { NextAction::ECommerce, "E-Commerce" },
    { NextAction::EcOcsi, "EcOcsi" },
    { NextAction::EcCatalog, "EcCatalog" },
    { NextAction::MultiProgram, "MultiProgram" },
    { NextAction::AbortDebug, "AbortDebug" },
    { NextAction::Mii, "Mii" },
    { NextAction::AocInfo,  "AocInfo" },
    { NextAction::HtmlDoc,  "HtmlDocument" },
    { NextAction::Bcat, "BCAT" },
    { NextAction::Network, "Network" },
    { NextAction::Album, "Album" },
    { NextAction::OperationEnv, "OperationEnv" },
    // HeadlineページでHeadline自体を非表示にするため末尾に固定
    { NextAction::Headline, "Headline" },
};

constexpr auto PageCountMax = sizeof(ActionPairList) / sizeof(NextActionPair);
NN_STATIC_ASSERT(PageCountMax > 1);
NN_STATIC_ASSERT(ActionPairList[PageCountMax - 1].Action == NextAction::Headline);

constexpr auto ScreenWidth = 1280.0f;
//constexpr auto ScreenHeight = 720.0f;

constexpr nn::util::Unorm8x4 White     = { { 255, 255, 255, 255 } };
constexpr nn::util::Unorm8x4 Red       = { { 255,   0,   0, 255 } };
constexpr nn::util::Unorm8x4 Green     = { {   0, 255,   0, 255 } };
constexpr nn::util::Unorm8x4 Yellow    = { { 255, 255,   0, 255 } };
constexpr nn::util::Unorm8x4 LinkBlue  = { {   0, 102, 255, 255 } };
constexpr nn::util::Unorm8x4 Aqua      = { {   0, 255, 255, 255 } };
constexpr nn::util::Unorm8x4 Orange    = { { 255,  69,   0, 255 } };
constexpr nn::util::Unorm8x4 Coral     = { { 255, 127,  80, 255 } };
constexpr nn::util::Unorm8x4 Gray      = { { 127, 127, 127, 255 } };
constexpr nn::util::Unorm8x4 RightGray = { { 211, 211, 211, 211 } };

class WarningView
{
public:
    WarningView() NN_NOEXCEPT;
    ~WarningView() NN_NOEXCEPT {};

private:
    struct MesageParam
    {
        std::string text = "";
        nn::util::Unorm8x4 color = White;
    };

    std::string m_Title;
    nn::util::Unorm8x4 m_TitleColor;
    float m_TitleScale;
    float m_TitlePosX;
    float m_TitlePosY;

    std::vector<MesageParam> m_Messages;
    float m_MsgScale;
    float m_MsgPosX;
    float m_MsgPosY;

    bool m_IsChanged;
    bool m_IsTitleCentering;
    bool m_IsMessageCentering;

public:
    void SetTitle(const std::string& title) NN_NOEXCEPT;
    void SetTitleColor(const nn::util::Unorm8x4& color) NN_NOEXCEPT;
    void SetTitleScale(float scale) NN_NOEXCEPT;
    void SetTitlePos(float x, float y) NN_NOEXCEPT;

    void ClearMessage() NN_NOEXCEPT;
    size_t PushMessage(const std::string& text, const nn::util::Unorm8x4& color) NN_NOEXCEPT;
    void UpdateMessage(size_t index, const std::string& text) NN_NOEXCEPT;
    void SetMessageScale(float scale) NN_NOEXCEPT;
    void SetMessagePos(float x, float y) NN_NOEXCEPT;

    bool IsChanged() NN_NOEXCEPT;
    void SetChanged(bool isChanged) NN_NOEXCEPT;

    bool IsTitleCentering() NN_NOEXCEPT;
    void SetTitleCentering(bool isCentering) NN_NOEXCEPT;
    bool IsMessageCentering() NN_NOEXCEPT;
    void SetMessageCentering(bool isCentering) NN_NOEXCEPT;

    void CenteringTitle(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT;
    void CenteringMessage(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT;
    void DrawView(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT;
};

class SceneCommon : public Scene
{
    NN_DISALLOW_COPY(SceneCommon);

public:
    SceneCommon() NN_NOEXCEPT;
    virtual ~SceneCommon() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Setup(nn::gfx::util::DebugFontTextWriter* writer, const nn::hid::TouchScreenState<1>& inTouchState, NextAction ownAction) NN_NOEXCEPT NN_OVERRIDE;
    virtual void ResetState() NN_NOEXCEPT NN_OVERRIDE;

    virtual void HandleNPad() NN_NOEXCEPT NN_OVERRIDE;
    virtual void HandleTouchScreen(const nn::hid::TouchScreenState<1>& state) NN_NOEXCEPT NN_OVERRIDE;
    virtual NextAction Process() NN_NOEXCEPT NN_OVERRIDE;
    virtual void DrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT NN_OVERRIDE;

    virtual void SetLaunchStorageInfo(LaunchStorageInfo&& inStorageInfo) NN_NOEXCEPT NN_OVERRIDE;

    virtual void EnterScene() NN_NOEXCEPT NN_OVERRIDE;
    virtual void LeaveScene() NN_NOEXCEPT NN_OVERRIDE;

protected:
    virtual void InternalSetup() NN_NOEXCEPT = 0;
    virtual void InternalHandleNPad() NN_NOEXCEPT = 0;
    virtual void InternalHandleTouchScreen() NN_NOEXCEPT = 0;
    virtual void InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT = 0;
    virtual void InternalProcess() NN_NOEXCEPT = 0;

protected:
    // 他画面にまたがって影響するパラメータのため static としておく
    static nn::hid::TouchState m_PreviousTouch;

    struct Position
    {
        float x;
        float y;

        Position() : x(0.0f), y(0.0f) {}
        Position(float inX, float inY) : x(inX), y(inY) {}
    };

    struct TouchRange
    {
        Position beginPos;
        Position endPos;
        bool isSetting;
        bool isTouchable;
        std::string labelStr;
        void* dataPtr;

        TouchRange() : isSetting(false), isTouchable(true), dataPtr(nullptr) {}

        bool IsInRange(float inX, float inY)
        {
            if (isTouchable == true && isSetting == true &&
                beginPos.x <= inX && inX <= endPos.x &&
                beginPos.y <= inY && inY <= endPos.y)
            {
                return true;
            }
            return false;
        }
        bool IsInRange(int32_t inX, int32_t inY)
        {
            return this->IsInRange(static_cast<float>(inX), static_cast<float>(inY));
        }
        bool IsInRange(nn::hid::TouchState touchState)
        {
            return this->IsInRange(static_cast<float>(touchState.x), static_cast<float>(touchState.y));
        }
    };

    struct RangePackage
    {
        Position pos;
        std::string labelStr;
        TouchRange range;
    };

    struct PageRange
    {
        RangePackage package;
        NextAction action;
    };

    struct ApplicationInfo
    {
        std::string id;
        // (SIGLO-78007) ProgramId を別に保持する形とする (設定されていない場合は ApplicationId と同値)
        std::string programId;
        VersionHolder ver;
    };

    // 他画面と共通して保持しておいた方が良い情報のため static 変数としておく
    static ApplicationInfo m_ApplicationInfo;

protected:
    virtual void WriteTouchRange(nn::gfx::util::DebugFontTextWriter* writer, const Position& inPos, const char* inStr, TouchRange* inoutRange, bool isAddBracket = false) NN_NOEXCEPT;
    virtual void WriteTouchRange(nn::gfx::util::DebugFontTextWriter* writer, RangePackage* inPackage, bool isAddBracket = false) NN_NOEXCEPT;

    void InternalExitNPadHandleProcess() NN_NOEXCEPT;
    void InternalExitHandleProcess() NN_NOEXCEPT;

    void SetDefaultScale(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT;
    void DrowDebugTouchRange(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT;

    void SetPageInfo(nn::gfx::util::DebugFontTextWriter* writer, NextAction inSetAction) NN_NOEXCEPT;

    void DrawWarningMessage(nn::gfx::util::DebugFontTextWriter* writer, WarningView* warningView) NN_NOEXCEPT;

    void SetJumpPage(NextAction targetPage) NN_NOEXCEPT;

private:
    NextAction m_OwnAction;
    NextAction m_NextAction;
    bool m_IsTouch;
    bool m_IsCheckingExit;

    RangePackage m_Exit;
    RangePackage m_Headline;

    PageRange m_LeftPage;
    PageRange m_RightPage;

    // 全ての画面で共通に表示するため static 化
    static LaunchStorageInfo m_LaunchStorageInfo;

    WarningView m_ExitWarningView;

protected:
    // TODO:SaveDataSceneで使用、カーソル操作する他のページにも適用
    const nn::hid::NpadButtonSet CursorUp = nn::hid::NpadButton::Up::Mask | nn::hid::NpadButton::StickLUp::Mask;
    const nn::hid::NpadButtonSet CursorDown = nn::hid::NpadButton::Down::Mask | nn::hid::NpadButton::StickLDown::Mask;
    const nn::hid::NpadButtonSet CursorRight = nn::hid::NpadButton::Right::Mask | nn::hid::NpadButton::StickLRight::Mask;
    const nn::hid::NpadButtonSet CursorLeft = nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::StickLLeft::Mask;

    SceneCommon::RangePackage m_ConfirmOkRange;
    SceneCommon::RangePackage m_ConfirmCancelRange;
};
