﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>

#include "TestAppSimple_FsUtilities.h"

#include "TestAppSimple_RomHashScene.h"

#include "TestAppSimple_MetaInfo.h"

namespace {
    const char* const CalcHashFilePath = "Contents:/DummyData.dat";

    const float LineSize = 33.0f;
}

RomHashScene::RomHashScene() NN_NOEXCEPT
    : m_State(State_None),
      m_IsGetHashVal(false), m_IsHashMatch(false), m_IsExistDummyFile(false),
      m_Hash(CalcHashFilePath), m_CalcCount(0), m_Progress(0)
{
}

void RomHashScene::InternalSetup() NN_NOEXCEPT
{
    m_FileSize = 0;

    m_IsExistDummyFile = fsutil::IsExistPath(CalcHashFilePath);
    if (m_IsExistDummyFile == true)
    {
        fsutil::File file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(file.Open(CalcHashFilePath, nn::fs::OpenMode_Read));

        m_FileSize = file.GetSize();

        // 期待ハッシュ値を読み込んでおく (リビルドしないための措置)
        ReadHashValueFileData();
    }
}

void RomHashScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) && m_IsExistDummyFile == true)
        {
            m_State = State_CalcHash;
        }
    }
}

void RomHashScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (m_CalculateRange.IsInRange(m_PreviousTouch) && m_IsExistDummyFile == true)
        {
            m_State = State_CalcHash;
        }
    }
}

void RomHashScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_CalcHash)
    {
        // ハッシュ計算処理の開始
        if (m_Hash.StartSha1ProcessAsync() < 0)
        {
            // 計算処理のスレッド生成失敗
            char failMsg[] = "Get Failed";
            nn::util::Strlcpy(m_HashVal, failMsg, Hash::Sha1MaxStrSize);
            m_IsGetHashVal = true;
            m_State = State_CalcEnd;
        }
        else
        {
            // スレッド生成成功
            m_State = State_CalcProcessing;
        }
    }
    else if (m_State == State_CalcProcessing)
    {
        if (m_Hash.IsProcessing() == false)
        {
            // 計算処理の終了を検知した場合は、計算結果を取得する
            if (m_Hash.GetSha1Data(m_HashVal, (sizeof(m_HashVal) / sizeof(m_HashVal[0]))) == 0)
            {
                m_HashVal[Hash::Sha1MaxStrSize] = '\0';
            }
            else
            {
                char failMsg[] = "Get Failed";
                nn::util::Strlcpy(m_HashVal, failMsg, Hash::Sha1MaxStrSize);
            }
            m_IsGetHashVal = true;

            // 期待値と比較しておく
            if (nn::util::Strncmp(ExpectedHash, m_HashVal, Hash::Sha1MaxStrSize) == 0)
            {
                m_IsHashMatch = true;
            }
            m_State = State_CalcEnd;
            // (念のための後始末として)スレッドの終了処理を呼び出しておく
            m_Hash.StopSha1ProcessAsync();
        }
        else
        {
            // 計算中はプログレス値を取得する
            m_Progress = m_Hash.GetProgressPercent();
        }
    }
}

void RomHashScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[RomHash]");

    int ypos = 1;
    writer->SetCursor(70.0f, 80.0f + (LineSize * ypos));
    ++ypos;
    {
        writer->Print("FileSize : ");
        writer->Print("%lld [Byte]", m_FileSize);
        writer->SetCursor(70.0f, 80.0f + (LineSize * ypos));
        ++ypos;
    }

    if (m_IsExistDummyFile == false)
    {
        writer->SetTextColor(Red);
        writer->SetCursor(80.0f, 80.0f + (LineSize * ypos));
        writer->Print("DummyFile is Not Exist.");
        // 以降の処理は書き出さない・・
        return;
    }

    this->DrawHashInfo(writer);

    if (m_State == State_None && m_IsGetHashVal == false)
    {
        Position CalculatePos(100.0f, 560.0f);
        static const char* CalculateStr = "A: Calculate SHA1";
        this->WriteTouchRange(writer, CalculatePos, CalculateStr, &m_CalculateRange);
    }
}

void RomHashScene::DrawHashInfo(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->Print("Expected SHA1: \t");
    if (m_IsHashMatch == true)
    {
        writer->SetTextColor(Green);
    }
    writer->Print(ExpectedHash);
    if (m_IsHashMatch == true)
    {
        writer->SetTextColor(White);
    }

    writer->SetCursor(70.0f, writer->GetCursorY() + LineSize);

    writer->Print("Get SHA1: \t\t");
    if (m_State == State_CalcProcessing)
    {
        // ハッシュの計算中であることを表示
        writer->SetScale(1.25f, 1.25f);
        writer->SetTextColor(Yellow);
        writer->Print("Calculating ");

        // ちょっとした計算中アニメーション表現
        {
            if (m_CalcCount > 100)
            {
                m_CalcCount = 0;
            }

            ++m_CalcCount;

            if (m_CalcCount < 25)
            {
                writer->Print(".   ");
            }
            else if (m_CalcCount < 50)
            {
                writer->Print("..  ");
            }
            else if (m_CalcCount < 75)
            {
                writer->Print("... ");
            }
            else
            {
                writer->Print("....");
            }
        }

        // プログレス(パーセント)表示
        writer->Print(" [ %3d % ]", m_Progress);
    }
    else if (m_State == State_CalcEnd && m_IsGetHashVal == true)
    {
        if (m_IsHashMatch == true)
        {
            writer->SetTextColor(Green);
        }
        else
        {
            writer->SetTextColor(Red);
        }
        writer->Print(m_HashVal);

        writer->SetTextColor(White);
    }
}
