﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>

// nn::swkbd 系モジュールは Generic では扱わない
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/util/util_ScopeExit.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/oe/oe_SubProgramJump.private.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_MultiProgramScene.h"
#include "TestAppSimple_AccountManager.h"

namespace {
    // ひとまず 1MB 程度としておく
    const size_t SwkbdApplicationHeapSize = 1 * 1024 * 1024;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // 各項目ごとのソフトウェアキーボードの起動引数を受け設定し入力値を値渡しresult値を返す
    nn::Result GetStringBySwkbd(std::string* pOutStr, nn::swkbd::ShowKeyboardArg showKeyboardArg) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutStr);

        ApplicationHeap applicationHeap(SwkbdApplicationHeapSize);

        // 共有メモリ用バッファの割り当て
        const auto inHeapSize = nn::swkbd::GetRequiredWorkBufferSize(false);
        auto swkbdWorkBuffer = applicationHeap.Allocate(inHeapSize, nn::os::MemoryPageSize);
        NN_UTIL_SCOPE_EXIT
        {
            applicationHeap.Deallocate(swkbdWorkBuffer);
        };

        showKeyboardArg.workBuf = swkbdWorkBuffer;
        showKeyboardArg.workBufSize = inHeapSize;

        // 終了パラメータの設定
        const auto outHeapSize = nn::swkbd::GetRequiredStringBufferSize();
        nn::swkbd::String outputString;
        outputString.ptr = applicationHeap.Allocate(outHeapSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(outputString.ptr);
        NN_UTIL_SCOPE_EXIT
        {
            applicationHeap.Deallocate(outputString.ptr);
        };
        outputString.bufSize = outHeapSize;
        auto result = nn::swkbd::ShowKeyboard(&outputString, showKeyboardArg);

        if (result.IsSuccess())
        {
            // 入力結果文字列を受け取る
            *pOutStr = reinterpret_cast<char*>(outputString.ptr);
        }

        // result値を返す
        return result;
    }
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
};

MultiProgramScene::MultiProgramScene() NN_NOEXCEPT
    : m_State(State_None), m_PreviousProgramIndexNum(0), m_InputIndexNumber("0")
{
}

void MultiProgramScene::InternalSetup() NN_NOEXCEPT
{
    m_LastJumpFunctionResult = nn::ResultSuccess();

    {
        m_ChangeIndexNumber.pos.x = 600.0f;
        m_ChangeIndexNumber.pos.y = 120.0f;
        m_ChangeIndexNumber.labelStr = "Y: Change";
    }
    {
        m_JumpToIndexNumber.pos.x = 70.0f;
        m_JumpToIndexNumber.pos.y = 480.0f;
        m_JumpToIndexNumber.labelStr = "A: Jump to SubProgram";
    }
    {
        m_InputLaunchParamRange.pos.x = 600.0f;
        m_InputLaunchParamRange.pos.y = 240.0f;
        m_InputLaunchParamRange.labelStr = "X: Input";
    }

    m_WarningView.PushMessage("Do you want to jump to program of index \" ", White);
    m_IndexNumMsgIdx = m_WarningView.PushMessage(m_InputIndexNumber.c_str(), Coral);
    m_WarningView.PushMessage("\" ?", White);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    m_PreviousProgramIndexNum = nn::oe::GetPreviousProgramIndex();
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void MultiProgramScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_JumpReady;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            m_State = State_ChangeIndexNumber;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            m_State = State_InputLaunchParam;
        }
    }
    else if (m_State == State_JumpReady)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_LaunchJumpFunction;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_None;
        }
    }
}

void MultiProgramScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (m_JumpToIndexNumber.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_JumpReady;
        }
        else if (m_ChangeIndexNumber.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_ChangeIndexNumber;
        }
        else if (m_InputLaunchParamRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_InputLaunchParam;
        }
    }
    else if (m_State == State_JumpReady)
    {
        if (m_ConfirmOkRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_LaunchJumpFunction;
        }
        else if (m_ConfirmCancelRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_None;
        }
    }
}

void MultiProgramScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_ChangeIndexNumber)
    {
        m_InputIndexNumber = this->GetInputIndexNumber();
        m_WarningView.UpdateMessage(m_IndexNumMsgIdx, m_InputIndexNumber);

        m_State = State_None;
    }
    else if (m_State == State_InputLaunchParam)
    {
        m_InputLaunchParam = this->GetInputLaunchParam();
        m_State = State_None;
    }
    else if (m_State == State_LaunchJumpFunction)
    {
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // (SIGLO-79976) ジャンプする直前に PushOpenUsers を呼ぶ
        auto& am = AccountManager::GetInstance();
        am.PushOpenUsers();
        // ジャンプ処理は Generic 版では実行しない
        auto indexNum = Convert::ToUInt8(m_InputIndexNumber);
        auto launchParam = m_InputLaunchParam.c_str();
        m_LastJumpFunctionResult = nn::oe::ExecuteProgram(indexNum, launchParam, m_InputLaunchParam.size());
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        m_State = State_None;
    }
}

std::string MultiProgramScene::GetInputIndexNumber() NN_NOEXCEPT
{
    // ソフトウェアキーボードでインデックス番号を入力してもらう
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::swkbd::ShowKeyboardArg showKeyboardArg;
    nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

    auto& config = showKeyboardArg.keyboardConfig;
    // 数字入力用のキーボードを表示させる(数字のみ入力可)
    config.keyboardMode = nn::swkbd::KeyboardMode_Numeric;
    // 最大桁数は3桁まで (256までを想定)
    config.textMaxLength = 3;
    // 入力プレビューは 1 行
    config.inputFormMode = nn::swkbd::InputFormMode_OneLine;
    // 返り値は UTF-8 文字列で返してもらう
    config.isUseUtf8 = true;

    // ヘッダー文字列の設定
    const char* guideString = "Please Input ProgramIndex Number.";
    nn::swkbd::SetHeaderTextUtf8(&showKeyboardArg.keyboardConfig, guideString);

    // ソフトウェアキーボードから文字列を受け取る
    std::string inputStr = "";
    auto result = GetStringBySwkbd(&inputStr, showKeyboardArg);

    if (result.IsFailure())
    {
        // エラーResultの場合は元の値を返す
        return m_InputIndexNumber;
    }

    if (inputStr == "")
    {
        // 空白が返された場合は元の数値のまま
        return m_InputIndexNumber;
    }

    if (Convert::ToUInt8(inputStr) == 0)
    {
        // 0 の入力または UInt8 の範囲(0～255)に収まらない数値の場合は "0" を返す
        return "0";
    }

    // 入力文字列を返す
    return inputStr;
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 版では単純にインクリメントした値を返す
    auto num = Convert::ToUInt32(m_InputIndexNumber);
    ++num;
    return Convert::ToString(num);
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

std::string MultiProgramScene::GetInputLaunchParam() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::swkbd::ShowKeyboardArg showKeyboardArg;
    nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

    auto& config = showKeyboardArg.keyboardConfig;
    // フルキーボード(ラテン文字入力優先)モード
    config.keyboardMode = nn::swkbd::KeyboardMode_FullLatin;
    // プレビュー欄は複数行の見た目になります
    config.inputFormMode = nn::swkbd::InputFormMode_MultiLine;
    // 返り値は UTF-8 文字列で返してもらう
    config.isUseUtf8 = true;

    // ガイド文字列の設定
    const char* guideString = "Please Input Launch Parameter.";
    nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig, guideString);

    // ソフトウェアキーボードから文字列を受け取る
    std::string inputStr = "";
    auto result = GetStringBySwkbd(&inputStr, showKeyboardArg);
    if (result.IsFailure())
    {
        // エラーResultの場合は元の値を返す
        return m_InputLaunchParam;
    }

    return inputStr;
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // Generic 版では単純に適当な文字列を返す
    return "Generic can not use this function. This parameter is provisional data.";
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
}

void MultiProgramScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[MultiProgram]");

    if (m_State == State_JumpReady)
    {
        // 警告画面の描画
        this->DrawWarningMessage(writer, &m_WarningView);

        // 以降の処理は書き出さない
        return;
    }

    writer->SetScale(1.8f, 1.8f);

    writer->SetCursor(60.0f, 120.0f);
    writer->Print("Index Number (to Jump) :  %s", m_InputIndexNumber.c_str());

    this->WriteTouchRange(writer, &m_ChangeIndexNumber, true);

    writer->SetCursor(60.0f, m_InputLaunchParamRange.pos.y);
    writer->Print("Attach Launch Param : %d [Byte]", m_InputLaunchParam.size());

    this->WriteTouchRange(writer, &m_InputLaunchParamRange, true);

    this->SetDefaultScale(writer);
    writer->SetCursor(60.0f, 170.0f);
    writer->Print("Previous ProgramIndex : %d", m_PreviousProgramIndexNum);

    if (m_InputLaunchParam.size() > 0)
    {
        //入力済みであれば項目下部に12バイトまで表示する
        const size_t summarySize = 12;
        auto summaryStr = m_InputLaunchParam.substr(0, summarySize);

        if (m_InputLaunchParam.size() > summarySize)
        {
            summaryStr += "...";
        }

        writer->SetCursor(60.0f, m_InputLaunchParamRange.pos.y + 50.0f);
        writer->Print("Input Data : %s", summaryStr.c_str());
    }

    // 目立つように少し大きめに描画しておく
    writer->SetScale(2.5f, 2.5f);
    this->WriteTouchRange(writer, &m_JumpToIndexNumber);

    this->SetDefaultScale(writer);
    writer->SetTextColor(RightGray);
    writer->SetCursor(m_JumpToIndexNumber.pos.x, m_JumpToIndexNumber.pos.y + 60.0f);
    writer->Print("LastResult : ");
    if (m_LastJumpFunctionResult.IsFailure())
    {
        // エラーであれば赤色にしておく
        writer->SetTextColor(Red);
    }
    writer->Print("0x%08x", m_LastJumpFunctionResult.GetInnerValueForDebug());
}
