﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/util/util_StringUtil.h>

#include "TestAppSimple_MetaInfoScene.h"

#include "TestAppSimple_MetaInfo.h"

void MetaInfoScene::InternalSetup() NN_NOEXCEPT
{
    // (SIGLO-52835) BCAT が有効かどうかの確認
    bool isIncludeBcatStorageSize = false;
    bool isIncludeBcatPassphrase = false;

    for (int i = 0; i < MetaInfoMaxCount; ++i)
    {
        auto& info = MetaInfoList[i];
        const size_t itemSize = info.item.size();
        const size_t valueSize = info.value.size();
        if (itemSize == 0 && valueSize == 0)
        {
            // どちらの値も空文字の場合に抜ける(終端データの検知)
            break;
        }

        DispayInfo displayInfo;
        nn::util::Strlcpy(displayInfo.item, info.item.c_str(), (MaxItemSize - 1));
        nn::util::Strlcpy(displayInfo.value, info.value.c_str(), (MaxValueSize - 1));

        // (SIGLO-52835) BCAT の設定項目の有無を確認して有効かどうかを判断する
        if (info.item == "App/BcatDeliveryCacheStorageSize")
        {
            isIncludeBcatStorageSize = true;
        }
        else if (info.item == "App/BcatPassphrase")
        {
            isIncludeBcatPassphrase = true;
        }

        //NN_LOG("item[%d] = %s\n", i, displayInfo.item);
        //NN_LOG("value[%d] = %s\n", i, displayInfo.value);

        m_DisplayList.push_back(displayInfo);
    }

    // アプリケーションの基本情報を保持しておく
    this->SetApplicationInfo();

    // (SIGLO-52835) BCAT が有効かどうかを設定しておく
    IsBcatSettingEnabled = false;
    if (isIncludeBcatStorageSize && isIncludeBcatPassphrase)
    {
        IsBcatSettingEnabled = true;
    }

    m_LastPage = (static_cast<int>(m_DisplayList.size()) / MaxItemCount) + 1;
    if (static_cast<int>(m_DisplayList.size()) % MaxItemCount == 0)
    {
        --m_LastPage;
    }

    m_CurrentPage = 1;
}

void MetaInfoScene::SetApplicationInfo() NN_NOEXCEPT
{
    std::string rvStr;
    std::string pvStr;
    std::string uvStr;

    for (auto& info : m_DisplayList)
    {
        const std::string ri = info.item;
        if (ri == "Core/ApplicationId" || ri == "App/ApplicationId")
        {
            m_ApplicationInfo.id = info.value;
        }
        else if (ri == "Core/ProgramId")
        {
            m_ApplicationInfo.programId = info.value;
        }
        else if (ri == "App/ReleaseVersion")
        {
            rvStr = info.value;
        }
        else if (ri == "App/PrivateVersion")
        {
            pvStr = info.value;
        }
        else if (ri == "App/Version")
        {
            uvStr = info.value;
        }
    }

    if (m_ApplicationInfo.programId == "")
    {
        // ProgramId が設定されていない場合、m_ApplicationInfo.id と同値を入れておく
        m_ApplicationInfo.programId = m_ApplicationInfo.id;
    }
    else if (m_ApplicationInfo.id == "")
    {
        // ApplicationId が設定されていない場合、m_ApplicationInfo.programId と同値を入れておく
        // (この場合 ProgramId は設定されているはず)
        m_ApplicationInfo.id = m_ApplicationInfo.programId;
    }

    // バージョン値を取り込む
    m_ApplicationInfo.ver.Convert(rvStr, pvStr, uvStr);
}

void MetaInfoScene::InternalHandleNPad() NN_NOEXCEPT
{
    if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
         || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
        && m_CurrentPage < m_LastPage)
    {
        ++m_CurrentPage;
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
              || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
        && m_CurrentPage > 1)
    {
        --m_CurrentPage;
    }
}

void MetaInfoScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_NextPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage < m_LastPage)
    {
        ++m_CurrentPage;
    }
    else if (m_BackPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage > 1)
    {
        --m_CurrentPage;
    }
}

void MetaInfoScene::InternalProcess() NN_NOEXCEPT
{
    // 空実装としておく
}

void MetaInfoScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(25.0f, 70.0f);
    writer->Print("[MetaInfo]");
    if (m_LastPage > 1)
    {
        writer->Print(" (%d / %d)", m_CurrentPage, m_LastPage);
    }

    if (m_DisplayList.empty() == false)
    {
        int count = 0;
        float yPointCount = 0;
        for (auto iter = m_DisplayList.begin(); iter != m_DisplayList.end(); ++iter, ++count)
        {
            if (count < ((m_CurrentPage - 1) * MaxItemCount))
            {
                continue;
            }

            if (count >= (m_CurrentPage * MaxItemCount))
            {
                break;
            }

            writer->SetCursor(50.0f, 102.0f + (30.0f * (yPointCount)));

            writer->Print(iter->item);
            writer->Print(": ");
            writer->Print(iter->value);

            ++yPointCount;
        }
    }

    if (m_LastPage > 1)
    {
        if (m_CurrentPage > 1)
        {
            Position backPagePos(50.0f, 640.0f);
            static const char* BackPageStr = "<- : BackPage";
            this->WriteTouchRange(writer, backPagePos, BackPageStr, &m_BackPageRange);
        }
        if (m_CurrentPage < m_LastPage)
        {
            Position nextPagePos(1050.0f, 640.0f);
            static const char* NextPageStr = "-> : NextPage";
            this->WriteTouchRange(writer, nextPagePos, NextPageStr, &m_NextPageRange);
        }
    }
}
