﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <sstream>

#include <nn/nn_Log.h>

// nn::web 系モジュールは Generic に対応していないため実装分岐
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nn/web/offline_Api.h>
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )

#include "TestAppSimple_FsUtilities.h"
#include "TestAppSimple_HtmlDocScene.h"

namespace {
    static const char* const HtmlDocumentFilePath = "Contents:/HtmlPath.txt";
};

HtmlDocScene::HtmlDocScene() NN_NOEXCEPT
    : m_State(State_None), m_IsExistHtmlDocFile(false), m_LastResultValue(0),
      m_CurrentSelectPos(0), m_CurrentMaxPosNum(0), m_CurrentPage(1), m_LastPage(1)
{
}

void HtmlDocScene::InternalSetup() NN_NOEXCEPT
{
    m_IsExistHtmlDocFile = fsutil::IsExistPath(HtmlDocumentFilePath);
    if (m_IsExistHtmlDocFile == true)
    {
        // ファイルが存在すれば中身を読み込んでおく
        this->ReadHtmlPath();

        m_LastPage = (static_cast<int>(m_HtmlFilePathList.size()) / MaxItemCount) + 1;
        if (static_cast<int>(m_HtmlFilePathList.size()) % MaxItemCount == 0)
        {
            --m_LastPage;
        }

        m_CurrentPage = 1;
    }

    this->ResetPathRangeList();

    m_WarningView.SetTitleCentering(false);
    m_WarningView.SetTitleColor(White);
    m_WarningView.SetTitleScale(1.75f);
    m_WarningView.SetTitlePos(80.0f, 150.0f);
    m_WarningView.SetTitle("Open HTML File?");
    m_WarningView.SetMessageCentering(false);
    m_WarningView.SetMessageScale(1.75f);
    m_WarningView.SetMessagePos(60.0f, 190.0f);
    m_WarningView.PushMessage("Path : \n  ", White);
    m_PathMsgIdx = m_WarningView.PushMessage(m_LaunchHtmlPagePath.c_str(), Orange);
}

void HtmlDocScene::ReadHtmlPath() NN_NOEXCEPT
{
    fsutil::File file;

    auto result = file.Open(HtmlDocumentFilePath, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[Error] HtmlDocScene::ReadHtmlPath() OpenFile Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return;
    }

    const auto fileSize = file.GetSize();
    if (fileSize <= 0)
    {
        NN_LOG("[Error] HtmlDocScene::ReadHtmlPath() GetSize Failed : fileSize = %lld\n", fileSize);
        return;
    }

    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    const size_t readSize = static_cast<size_t>(fileSize - 3);
    std::unique_ptr<char[]> buf(new char[readSize + 1]);
    *(buf.get() + readSize) = '\0';

    size_t outSize = 0;
    // UTF8のBOM付きファイルであることを想定(オフセットの3バイトはそのため)
    result = file.Read(&outSize, 3, buf.get(), readSize);
    if (result.IsFailure())
    {
        // ファイル読み込み処理で失敗
        NN_LOG("[Error] HtmlDocScene::InternalSetup() ReadFile Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return;
    }

    // ファイルの中身からパスリストを構築
    std::istringstream is(buf.get());
    std::string lineStr;
    while (std::getline(is, lineStr))
    {
        // 一行ずつパス文字列を取得(末尾の \r は削除)してリストに詰める
        m_HtmlFilePathList.push_back(lineStr.erase(lineStr.size() - 1));
    }
}

void HtmlDocScene::ResetPathRangeList() NN_NOEXCEPT
{
    for (auto& range : m_PathRangeList)
    {
        range.isSetting = false;
    }
    m_CurrentSelectPos = 0;
}

void HtmlDocScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State == State_None)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) && m_IsExistHtmlDocFile == true)
        {
            m_LaunchHtmlPagePath = m_PathRangeList[m_CurrentSelectPos].labelStr;
            m_WarningView.UpdateMessage(m_PathMsgIdx, m_LaunchHtmlPagePath);
            m_State = State_LaunchReady;
        }
        else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
                  || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
                 && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
            this->ResetPathRangeList();
        }
        else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
                  || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
                 && m_CurrentPage > 1)
        {
            --m_CurrentPage;
            this->ResetPathRangeList();
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
                 || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
        {
            if (m_CurrentSelectPos < m_CurrentMaxPosNum)
            {
                ++m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = 0;
            }
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
                 || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
        {
            if (m_CurrentSelectPos > 0)
            {
                --m_CurrentSelectPos;
            }
            else
            {
                m_CurrentSelectPos = m_CurrentMaxPosNum;
            }
        }
    }
    else if (m_State == State_LaunchReady)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_Launching;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            m_State = State_None;
        }
    }
    else if (m_State == State_LaunchFailed)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            m_State = State_None;
        }
    }
}

void HtmlDocScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State == State_None && m_IsExistHtmlDocFile == true)
    {
        int count = 0;
        for (auto& range : m_PathRangeList)
        {
            if (range.isSetting == false)
            {
                break;
            }

            if (range.IsInRange(m_PreviousTouch) == true)
            {
                m_State = State_LaunchReady;
                m_LaunchHtmlPagePath = range.labelStr;
                m_WarningView.UpdateMessage(m_PathMsgIdx, m_LaunchHtmlPagePath);
                m_CurrentSelectPos = count;
                break;
            }

            ++count;
        }

        if (m_NextPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage < m_LastPage)
        {
            ++m_CurrentPage;
            this->ResetPathRangeList();
        }
        else if (m_BackPageRange.IsInRange(m_PreviousTouch) == true && m_CurrentPage > 1)
        {
            --m_CurrentPage;
            this->ResetPathRangeList();
        }
    }
    else if (m_State == State_LaunchReady)
    {
        if (m_ConfirmOkRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_Launching;
        }
        else if (m_ConfirmCancelRange.range.IsInRange(m_PreviousTouch))
        {
            m_State = State_None;
        }
    }
    else if (m_State == State_LaunchFailed)
    {
        if (m_FailedOkRange.IsInRange(m_PreviousTouch))
        {
            m_State = State_None;
        }
    }
}

void HtmlDocScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Launching)
    {
        // nn::web 系モジュールは Generic に対応していないため実装分岐
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        nn::web::ShowOfflineHtmlPageArg showOfflineHtmlPageArg(m_LaunchHtmlPagePath.c_str());
        nn::web::OfflineHtmlPageReturnValue returnValue;
        auto result = nn::web::ShowOfflineHtmlPage(&returnValue, showOfflineHtmlPageArg);
        if (result.IsFailure())
        {
            NN_LOG("[Error] HtmlDocScene::Process() ShowOfflineHtmlPage Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
            m_LastResultValue = result.GetInnerValueForDebug();
            m_State = State_LaunchFailed;
        }
        else
        {
            m_State = State_None;
        }
#else // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // Generic 版ではWebアプレットは起動できないのでとりあえず失敗画面を出すようにしておく
        m_State = State_LaunchFailed;
        m_LastResultValue = 0x12345678;
#endif // defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    }
}

void HtmlDocScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[HtmlDocument]");
    if (m_LastPage > 1)
    {
        writer->Print(" (%d / %d)", m_CurrentPage, m_LastPage);
    }

    const float LineSize = 33.0f;
    int ypos = 1;

    if (m_IsExistHtmlDocFile == false || m_HtmlFilePathList.empty() == true)
    {
        writer->SetScale(2.25f, 2.25f);
        writer->SetTextColor(Coral);
        writer->SetCursor(400.0f, 280.0f + (LineSize * ypos));
        ++ypos;
        writer->Print("HTML Files are Not Exist.");
        // 以降の処理は書き出さない・・
        return;
    }

    if (m_State == State_LaunchReady)
    {
        // アプレット起動確認画面を出す
        this->DrawWarningMessage(writer, &m_WarningView);

        // 以降の処理は書き出さない・・
        return;
    }

    if (m_State == State_LaunchFailed)
    {
        // アプレット起動失敗画面を出す (ここに来ることは滅多にないと思うが・・)
        writer->SetScale(2.25f, 2.25f);
        writer->SetTextColor(Orange);
        writer->SetCursor(150.0f, 200.0f + (LineSize * ypos));
        writer->Print("Launch Failed. (Result=0x%08x)", m_LastResultValue);
        writer->SetTextColor(White);
        writer->SetCursor(150.0f, 300.0f);
        //writer->Print("Path : %s", m_LaunchHtmlPagePath.c_str());

        writer->SetScale(3.0f, 3.0f);
        Position OkPos(530.0f, 400.0f);
        static const char* OkStr = "A: OK";
        this->WriteTouchRange(writer, OkPos, OkStr, &m_FailedOkRange);

        // 以降の処理は書き出さない・・
        return;
    }

    this->DrawPathList(writer);
}

void HtmlDocScene::DrawPathList(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(28.0f, 120.0f + (60.0f * (m_CurrentSelectPos)));
    writer->Print("*");

    {
        int count = 0;
        int yPointCount = 0;
        for (auto& path : m_HtmlFilePathList)
        {
            if (count < ((m_CurrentPage - 1) * MaxItemCount))
            {
                ++count;
                continue;
            }

            if (count >= (m_CurrentPage * MaxItemCount))
            {
                break;
            }

            Position pos(50.0f, 120.0f + (60.0f * (yPointCount)));
            this->WriteTouchRange(writer, pos, path.c_str(), &m_PathRangeList[yPointCount]);

            ++count;
            ++yPointCount;
        }
        m_CurrentMaxPosNum = (yPointCount - 1);
    }

    if (m_LastPage > 1)
    {
        if (m_CurrentPage > 1)
        {
            Position backPagePos(50.0f, 640.0f);
            static const char* BackPageStr = "<- : BackPage";
            this->WriteTouchRange(writer, backPagePos, BackPageStr, &m_BackPageRange);
        }
        if (m_CurrentPage < m_LastPage)
        {
            Position nextPagePos(1050.0f, 640.0f);
            static const char* NextPageStr = "-> : NextPage";
            this->WriteTouchRange(writer, nextPagePos, NextPageStr, &m_NextPageRange);
        }
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Open HTML File");
    }
}
