﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "TestAppSimple_EcServiceConfig.h"
#include "TestAppSimple_EcServiceAccount.h"
#include "TestAppSimple_EcServiceOcsi.h"

#if defined( TESTAPP_ENABLE_EC_ACCESS )
#include <curl/curl.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/ec/ec_ConsumableServiceItemApi.h>
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )

namespace {

#if defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )
    #define NN_ASSERT_RESULT_CURLE_OK(curlCode_)    NN_ASSERT_EQUAL(CURLE_OK, (curlCode_))

    const nn::ec::ConsumableServiceItemId PurchaseItems[] =
    {
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" },
        { "eca0000" }
    };

    size_t DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext) NN_NOEXCEPT
    {
        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        auto result = context.Create(nn::ssl::Context::SslVersion_Auto);
        if (result.IsSuccess())
        {
            nn::ssl::CertStoreId clientCertStoreId;
            result = context.RegisterInternalPki(&clientCertStoreId, nn::ssl::Context::InternalPki_DeviceClientCertDefault);
        }
        return result.IsSuccess() ? 0 : 1;
    }
#endif // defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )

} // namespace <unnamed>

// 消費型サービスアイテムの権利購入
EcOcsiRightPurchaser::EcOcsiRightPurchaser() NN_NOEXCEPT
{
    m_NsaId.id = 0;
}

EcOcsiRightPurchaser::~EcOcsiRightPurchaser() NN_NOEXCEPT
{
}

void EcOcsiRightPurchaser::Prepare(const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(nsaId.id, 0);
    SetStateToPrepared();
    m_NsaId = nsaId;
}

bool EcOcsiRightPurchaser::Exeute() NN_NOEXCEPT
{
    return Purchase();
}

bool EcOcsiRightPurchaser::Purchase() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )
    // ※タイトル固定
    const char* UrlFormat = "https://sugar.ws.td1.ss.nintendo.net/v1/applications/01002ad001e32000/accounts/%016llx/rights";
    const char* JsonFormat = "{\"item_id\": \"%s\", \"country\": \"JP\"}";

    CURL* handle = curl_easy_init();
    NN_ASSERT_NOT_NULL(handle);

    auto& nsaId = m_NsaId;
    auto& items = PurchaseItems;
    auto count = static_cast<int>(NN_ARRAY_SIZE(items));
    for (int i = 0; i < count; i++)
    {
        auto& json = m_JsonBuffer;
        NN_ABORT_UNLESS(nn::util::SNPrintf(json, sizeof(json), JsonFormat, items[i].value) < sizeof(json));

        auto& url = m_UrlBuffer;
        NN_ABORT_UNLESS(nn::util::SNPrintf(url, sizeof(url), UrlFormat, nsaId.id) < sizeof(url));

        auto& header = m_HeaerBuffer;
        NN_ABORT_UNLESS(nn::util::SNPrintf(header, sizeof(header), "Content-Length:%u", std::strlen(json)) < sizeof(header));

        curl_slist* slist = nullptr;
        NN_UTIL_SCOPE_EXIT{ curl_slist_free_all(slist); };
        slist = curl_slist_append(slist, "Content-Type:application/json");
        slist = curl_slist_append(slist, header);
        NN_ASSERT_NOT_NULL(slist);
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_HTTPHEADER, slist));

        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_HTTPAUTH, CURLAUTH_BASIC | CURLAUTH_ONLY));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_USERNAME, "c632ada668f04556"));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_PASSWORD, "XiOYjfyghJGfFRVs"));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_URL, url));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_POSTFIELDS, json));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_setopt(handle, CURLOPT_SSL_CTX_FUNCTION, DefaultSslCtxFunction));
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_perform(handle));

        long code;
        NN_ASSERT_RESULT_CURLE_OK(curl_easy_getinfo(handle, CURLINFO_RESPONSE_CODE, &code));
        NN_ASSERT_EQUAL(201, code);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    curl_easy_cleanup(handle);
#endif // defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )
    return true;
}

// 消費型サービスアイテムの権利照会・権利消費
EcOcsiRightConsumer::EcOcsiRightConsumer() NN_NOEXCEPT
{
    m_Request = Request_None;
    m_pUserSelector = nullptr;
    m_Uid = nn::account::InvalidUid;
    m_RightCount = 0;
}

EcOcsiRightConsumer::~EcOcsiRightConsumer() NN_NOEXCEPT
{
}

void EcOcsiRightConsumer::PrepareToInquire(UserSelector* pUserSelector) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserSelector);

    Prepare(Request_Inquire, nn::account::InvalidUid, pUserSelector);
}

void EcOcsiRightConsumer::PrepareToInquire(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);

    Prepare(Request_Inquire, uid, nullptr);
}

void EcOcsiRightConsumer::PrepareToConsume(const nn::account::Uid& uid, const EcOcsiRightList& list) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);
    NN_ASSERT(list.IsValid());

    Prepare(Request_Consume, uid, nullptr);
    int count = 0;
    for (int i = 0; i < list.count; i++)
    {
        if (list.infos[i].isSelected)
        {
            m_Rights[count++] = list.infos[i].rightData;
        }
    }
    NN_ASSERT_EQUAL(count, list.GetSelectedCount());
    m_RightCount = count;
}

void EcOcsiRightConsumer::Prepare(Request request, const nn::account::Uid& uid, UserSelector* pUserSelector) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(request, Request_None);

    SetStateToPrepared();
    m_Request = request;
    m_pUserSelector = pUserSelector;
    m_Uid = uid;
    m_RightCount = 0;
    std::memset(m_Rights, 0, sizeof(m_Rights));
    m_ErrorCode = nn::err::ErrorCode::GetInvalidErrorCode();
}

const nn::err::ErrorCode& EcOcsiRightConsumer::GetErrorCode() const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    return m_ErrorCode;
}

void EcOcsiRightConsumer::UpdateRights(EcOcsiRightList* list) const NN_NOEXCEPT
{
    NN_ASSERT(IsExecuted());
    if (m_LastResult.IsSuccess())
    {
        switch (m_Request)
        {
        case Request_Inquire:
            UpdateInquireRights(list);
            break;

        case Request_Consume:
            UpdateConsumeRights(list);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

void EcOcsiRightConsumer::UpdateInquireRights(EcOcsiRightList* list) const NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(m_Request, Request_Inquire);
    NN_ASSERT(m_LastResult.IsSuccess());

    list->Cleanup();
    for (int i = 0; i < m_RightCount; i++)
    {
        list->infos[i].rightData = m_Rights[i];
    }
    list->count = m_RightCount;
}

void EcOcsiRightConsumer::UpdateConsumeRights(EcOcsiRightList* list) const NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(m_Request, Request_Consume);
    NN_ASSERT(m_LastResult.IsSuccess());

    for (int i = 0; i < m_RightCount; i++)
    {
        auto consumed = false;
        for (int j = 0; j < list->count; j++)
        {
            // 権利の同値判定関数が定義されていないので memcmp で比較する
            // PrepareToConsume 時に単純に複製しているので一致する筈
            if (std::memcmp(&(list->infos[j].rightData), &m_Rights[i], sizeof(m_Rights[i])) == 0)
            {
                list->infos[j].isConsumed = true;
                consumed = true;
                break;
            }
        }
        NN_ASSERT(consumed);
    }
}

bool EcOcsiRightConsumer::Exeute() NN_NOEXCEPT
{
    if (m_pUserSelector != nullptr)
    {
        auto uid = m_pUserSelector->GetUid();
        NN_ASSERT_NOT_EQUAL(uid, nn::account::InvalidUid);
        m_Uid = uid;
    }
    switch (m_Request)
    {
    case Request_Inquire:
        m_LastResult = Inquire();
        break;

    case Request_Consume:
        m_LastResult = Consume();
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
    return m_LastResult.IsSuccess();
}

// 権利照会
nn::Result EcOcsiRightConsumer::Inquire() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! TORIAEZU: スマートポインタを使ってヒープに 256 KiB 確保する
    const size_t libraryWorkMemorySize = nn::ec::RequiredWorkMemorySizeForConsumableServiceItem;
    std::unique_ptr<nn::Bit8[], decltype(&free)> libraryWorkMemory((nn::Bit8 *)aligned_alloc(nn::os::MemoryPageSize, libraryWorkMemorySize), free);

    nn::ec::AsyncGetConsumableRightDataRequest asyncRequest;
    asyncRequest.Initialize(&accessor, m_Uid, libraryWorkMemory.get(), libraryWorkMemorySize);
    NN_RESULT_DO(asyncRequest.Begin(0, EcOcsiRightCountMax));
    NN_RESULT_TRY(asyncRequest.End(&m_RightCount, m_Rights, EcOcsiRightCountMax));
        NN_RESULT_CATCH(nn::ec::ResultShowErrorCodeRequired)
        {
            m_ErrorCode = asyncRequest.GetErrorCode();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
#else // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_FUNCTION_LOCAL_STATIC(int, s_Index);
    int count = s_Index++ % EcOcsiRightCountMax;
    for (int i = 0; i < count; i++)
    {
        std::memcpy(&m_Rights[i], &i, sizeof(i));
    }
    m_RightCount = count;
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}

// 権利消費
nn::Result EcOcsiRightConsumer::Consume() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS )
    nn::ec::ShopServiceAccessor accessor;
    NN_RESULT_DO(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! TORIAEZU: スマートポインタを使ってヒープに 256 KiB 確保する
    const size_t libraryWorkMemorySize = nn::ec::RequiredWorkMemorySizeForConsumableServiceItem;
    std::unique_ptr<nn::Bit8[], decltype(&free)> libraryWorkMemory((nn::Bit8 *)aligned_alloc(nn::os::MemoryPageSize, libraryWorkMemorySize), free);

    //! TORIAEZU: スマートポインタを使ってヒープに確保する。権利消費 API の出力を受け取るためだけに定義
    int rightDataCount = 0;
    std::unique_ptr<nn::ec::ConsumableServiceItemRightData> rightDataArray(new nn::ec::ConsumableServiceItemRightData[libraryWorkMemorySize]);

    nn::ec::AsyncConsumeRightDataRequest asyncRequest;
    asyncRequest.Initialize(&accessor, m_Uid, libraryWorkMemory.get(), libraryWorkMemorySize);
    NN_RESULT_DO(asyncRequest.Begin(m_Rights, m_RightCount));
    NN_RESULT_TRY(asyncRequest.End(&rightDataCount, rightDataArray.get(), EcOcsiRightCountMax));
        NN_RESULT_CATCH(nn::ec::ResultShowErrorCodeRequired)
        {
            m_ErrorCode = asyncRequest.GetErrorCode();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
#endif // defined( TESTAPP_ENABLE_EC_ACCESS )
    NN_RESULT_SUCCESS;
}
