﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestAppSimple_EcCatalogScene.h"
#include "TestAppSimple_EcServerAccessor.h"

namespace {

    const float IndentValue = 360.0f;

    const char* ToCstr(const std::string& value)
    {
        return value.size() > 0 ? value.c_str() : "(null or empty)";
    }

} // namespace <unnamed>

EcCatalogScene::EcCatalogScene() NN_NOEXCEPT
    : m_State(State_None)
    , m_Mode(Mode_List)
    , m_PageCount(0)
    , m_PageIndex(0)
{
}

EcCatalogScene::~EcCatalogScene() NN_NOEXCEPT
{
}

void EcCatalogScene::InternalSetup() NN_NOEXCEPT
{
    // EcService 共通
    Position line;
    SetEcServiceHeader(&line, "EcCatalog");

    // EcCatalog 固有
    m_UpdateLable.base = line;
    m_UpdateLable.prefix = "Catalog";
    m_UpdateLable.range.labelStr = "B: Update";
    line.y += NewLine; // Catalog [B:Update]
    line.y += (NewLine * LineCountMax); // ....
    line.y += NewLine; // RIGHT/LEFT: Move

    m_PrevPageLabel.base = line;
    m_PrevPageLabel.prefix = "";
    m_PrevPageLabel.range.labelStr = "<-: BackPage";
    m_NextPageLabel.base = line;
    m_NextPageLabel.base.x += 1000.0f;
    m_NextPageLabel.prefix = "";
    m_NextPageLabel.range.labelStr = "->: NextPage";
}

void EcCatalogScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // 処理中
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
    {
        m_State = State_SelectUser;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
    {
        m_State = State_Inquire;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        m_State = State_ShowShop;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickR::Mask))
    {
        m_State = State_ShowError;
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask)))
    {
        MovePageRight();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask)))
    {
        MovePageLeft();
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask)))
    {
        m_State = State_MoveItem;
    }
    else if ((HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask)))
    {
        m_State = State_MoveList;
    }
}

void EcCatalogScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    auto touch = m_PreviousTouch;
    if (m_State != State_None)
    {
        // 処理中
    }
    else if (m_AccountLabel.range.IsInRange(touch))
    {
        m_State = State_SelectUser;
    }
    else if (m_UpdateLable.range.IsInRange(touch))
    {
        m_State = State_Inquire;
    }
    else if (m_NextPageLabel.range.IsInRange(touch))
    {
        MovePageRight();
    }
    else if (m_PrevPageLabel.range.IsInRange(touch))
    {
        MovePageLeft();
    }
}

void EcCatalogScene::InternalProcess() NN_NOEXCEPT
{
    switch (m_State)
    {
    case State_None:
        break;
    case State_SelectUser:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireWithSelectUser);
        break;
    case State_Inquire:
        InquireCatalog();
        break;
    case State_MoveList:
        MoveCatalogList();
        break;
    case State_MoveItem:
        MoveCatalogItem();
        break;
    case State_ShowShop:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_ShowShopCatalogItem);
        break;
    case State_ShowError:
        EcServerAccessor::GetInstance().ShowErrorViewer();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    m_State = State_None;
}

void EcCatalogScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    Position line = BaseLine;
    DrawEcServiceHeader(&line, writer);

    // 操作
    if (EcServerAccessor::GetInstance().IsUserValid())
    {
        WriteLabel(writer, &m_UpdateLable);
    }
    line.y += NewLine;

    // 一覧
    switch (m_Mode)
    {
    case Mode_List:
        DrawCatalogCommon(&line, writer);
        DrawCatalogList(&line, writer);
        break;

    case Mode_Item:
        DrawCatalogCommon(&line, writer);
        DrawCatalogItem(&line, writer);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // 操作
    if (m_PageIndex > 0)
    {
        WriteLabel(writer, &m_PrevPageLabel);
    }
    if (m_PageCount > 0)
    {
        writer->SetCursor(600.0f, m_PrevPageLabel.base.y);
        writer->Print("%d/%d", m_PageIndex + 1, m_PageCount);
    }
    if ((m_PageIndex + 1) < m_PageCount)
    {
        WriteLabel(writer, &m_NextPageLabel);
    }
}

void EcCatalogScene::DrawCatalogCommon(Position* line, nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    auto consumable = catalog.GetSelectedConsumable();
    if (consumable)
    {
        auto language = ToCstr(catalog.list->language);
        auto count = catalog.GetConsumableCount();
        auto id = ToCstr(consumable->consumableId);

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("Consumable[%d] ", catalog.selectedConsumableIndex);
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print("Size=%d  Language=%s", count, language);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- ConsumableId:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(id);
        line->y += NewLine;
    }
    else if (EcServerAccessor::GetInstance().IsUserValid())
    {
        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("Consumable[]");
        if (catalog.list)
        {
            auto language = ToCstr(catalog.list->language);
            auto count = catalog.GetConsumableCount();

            writer->SetCursor(line->x + IndentValue, line->y);
            writer->Print("Size=%d  Language=%s", count, language);
        }
        line->y += NewLine;
    }
}

void EcCatalogScene::DrawCatalogList(Position* line, nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    auto consumable = catalog.GetSelectedConsumable();
    if (consumable)
    {
        auto name = ToCstr(consumable->name);
        auto description = ToCstr(consumable->description);
        auto disclaimer = ToCstr(consumable->disclaimer);

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- Name:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(name);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- Description:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(description);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- Disclaimer:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(disclaimer);
        line->y += NewLine;

        auto response = catalog.FindConsumableItemResponse();
        if (response)
        {
            auto language = ToCstr(response->language);
            auto itemCount = static_cast<int>(response->consumableItems.size());

            writer->SetCursor(line->x + Indent, line->y);
            if (itemCount > 0)
            {
                writer->Print("ConsumableItem[0...%d]", itemCount - 1);
            }
            else
            {
                writer->Print("ConsumableItem[]");
            }
            writer->SetCursor(line->x + IndentValue, line->y);
            writer->Print("Size=%d  Language=%s", itemCount, language);
        }
        else
        {
            writer->SetCursor(line->x + Indent, line->y);
            writer->Print("ConsumableItem[...]");
        }
        line->y += NewLine;

        line->y = m_PrevPageLabel.base.y - NewLine;
        writer->SetCursor(line->x, m_PrevPageLabel.base.y - NewLine);
        writer->Print("RIGHT/LEFT: Move   DOWN: ConsumableItem");
        line->y += NewLine;
    }
} // NOLINT(readability/fn_size)

void EcCatalogScene::DrawCatalogItem(Position* line, nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    auto consumableItem = catalog.GetSelectedConsumableItem();
    if (consumableItem)
    {
        auto response = catalog.FindConsumableItemResponse();
        NN_SDK_ASSERT_NOT_NULL(response);

        auto language = ToCstr(response->language);
        auto itemCount = static_cast<int>(response->consumableItems.size());
        auto itemId = ToCstr(consumableItem->itemId);
        auto itemName = ToCstr(consumableItem->name);

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("ConsumableItem[%d]", m_PageIndex);
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print("Size=%d  Language=%s", itemCount, language);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- ItemId:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(itemId);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- NsUid:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print("%lld", consumableItem->nsUid);
        line->y += NewLine;

        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("- Name:");
        writer->SetCursor(line->x + IndentValue, line->y);
        writer->Print(itemName);
        line->y += NewLine;

        if (consumableItem->price)
        {
            auto status = ToCstr(consumableItem->price->salesStatus);

            writer->SetCursor(line->x + Indent, line->y);
            writer->Print("- SalesStatus:");
            writer->SetCursor(line->x + IndentValue, line->y);
            writer->Print("%s", status);
            line->y += NewLine;

            if (consumableItem->price->regularPrice)
            {
                auto currency = ToCstr(consumableItem->price->regularPrice->currency);
                auto formatted = ToCstr(consumableItem->price->regularPrice->formattedValue);
                auto raw = ToCstr(consumableItem->price->regularPrice->rawValue);

                writer->SetCursor(line->x + Indent, line->y);
                writer->Print("- RegularPrice:");
                writer->SetCursor(line->x + IndentValue, line->y);
                writer->Print("%s %s  Raw %s", currency, formatted, raw);
                line->y += NewLine;
            }
            if (consumableItem->price->discountPrice)
            {
                auto currency = ToCstr(consumableItem->price->discountPrice->currency);
                auto formatted = ToCstr(consumableItem->price->discountPrice->formattedValue);
                auto raw = ToCstr(consumableItem->price->discountPrice->rawValue);
                auto start = ToCstr(consumableItem->price->discountPrice->startDatetime);
                auto end = ToCstr(consumableItem->price->discountPrice->endDatetime);

                writer->SetCursor(line->x + Indent, line->y);
                writer->Print("- DiscountPrice:");
                writer->SetCursor(line->x + IndentValue, line->y);
                writer->Print("%s %s  Raw %s", currency, formatted, raw);
                line->y += NewLine;

                writer->SetCursor(line->x + Indent, line->y);
                writer->Print("- DiscountPeriod:");
                writer->SetCursor(line->x + IndentValue, line->y);
                writer->Print("%s - %s", start, end);
                line->y += NewLine;
            }
        }

        line->y = m_PrevPageLabel.base.y - NewLine;
        writer->SetCursor(line->x, m_PrevPageLabel.base.y - NewLine);
        writer->Print("RIGHT/LEFT: Move   UP: Consumable   A: ShowShop");
    }
    else if (EcServerAccessor::GetInstance().IsUserValid())
    {
        writer->SetCursor(line->x + Indent, line->y);
        writer->Print("ConsumableItem[]");

        auto response = catalog.FindConsumableItemResponse();
        if (response)
        {
            auto language = ToCstr(response->language);
            auto itemCount = static_cast<int>(response->consumableItems.size());

            writer->SetCursor(line->x + IndentValue, line->y);
            writer->Print("Size=%d  Language=%s", itemCount, language);
        }
        line->y += NewLine;

        line->y = m_PrevPageLabel.base.y - NewLine;
        writer->SetCursor(line->x, m_PrevPageLabel.base.y - NewLine);
        writer->Print("RIGHT/LEFT: Move   UP: Consumable");
    }
    line->y += NewLine;
} // NOLINT(readability/fn_size)

void EcCatalogScene::UpdateContents() NN_NOEXCEPT
{
    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    switch (m_Mode)
    {
    case Mode_List:
        m_PageCount = catalog.GetConsumableCount();
        m_PageIndex = 0;
        EcServerAccessor::GetInstance().SelectConsumable(m_PageIndex);
        break;

    case Mode_Item:
        m_PageCount = catalog.GetConsumableItemCount();
        m_PageIndex = 0;
        EcServerAccessor::GetInstance().SelectConsumableItem(m_PageIndex);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void EcCatalogScene::InquireCatalog() NN_NOEXCEPT
{
    switch (m_Mode)
    {
    case Mode_List:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireCatalogList);
        break;

    case Mode_Item:
        EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireCatalogItem);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void EcCatalogScene::MoveCatalogList() NN_NOEXCEPT
{
    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    if (m_Mode != Mode_List)
    {
        m_Mode = Mode_List;
        m_PageCount = catalog.GetConsumableCount();
        m_PageIndex = catalog.IsConsumableSelected() ? catalog.selectedConsumableIndex : 0;
    }
}

void EcCatalogScene::MoveCatalogItem() NN_NOEXCEPT
{
    auto& catalog = EcServerAccessor::GetInstance().GetCatalog();
    if ((m_Mode != Mode_Item) && catalog.IsConsumableSelected())
    {
        if (!catalog.IsConsumableItemInquired())
        {
            EcServerAccessor::GetInstance().Execute(EcServerAccessor::Command_InquireCatalogItem);
        }
        m_Mode = Mode_Item;
        m_PageCount = catalog.GetConsumableItemCount();
        if (catalog.IsConsumableItemSelected())
        {
            m_PageIndex = catalog.selectedConsumableItemIndex;
        }
        else
        {
            EcServerAccessor::GetInstance().SelectConsumableItem(0);
            m_PageIndex = 0;
        }
    }
}

void EcCatalogScene::MovePageRight() NN_NOEXCEPT
{
    m_PageIndex = ((m_PageIndex + 1) < m_PageCount) ? (m_PageIndex + 1) : std::max((m_PageCount - 1), 0);
    switch (m_Mode)
    {
    case Mode_List:
        EcServerAccessor::GetInstance().SelectConsumable(m_PageIndex);
        break;

    case Mode_Item:
        EcServerAccessor::GetInstance().SelectConsumableItem(m_PageIndex);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void EcCatalogScene::MovePageLeft() NN_NOEXCEPT
{
    m_PageIndex = (m_PageIndex > 0) ? (m_PageIndex - 1) : 0;
    switch (m_Mode)
    {
    case Mode_List:
        EcServerAccessor::GetInstance().SelectConsumable(m_PageIndex);
        break;

    case Mode_Item:
        EcServerAccessor::GetInstance().SelectConsumableItem(m_PageIndex);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}
