﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/account.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_StateRetention.h>

#include "TestAppSimple_AccountManager.h"

// トレースログ出力を有効にしたい場合はこの define を有効にする
//#define ENABLE_TRACE_ACCOUNT_MANAGER_LOG
#if defined( ENABLE_TRACE_ACCOUNT_MANAGER_LOG )
#define TRACE_ACCOUNT_MANAGER_LOG(...)     NN_LOG(__VA_ARGS__)
#else
#define TRACE_ACCOUNT_MANAGER_LOG(...)
#endif // defined( ENABLE_TRACE_ACCOUNT_MANAGER_LOG )

AccountManager::AccountManager() NN_NOEXCEPT
{
    // 特になにもしない
}

bool AccountManager::OpenUserWithSelectorApplet(UserInfo* pOutInfo) NN_NOEXCEPT
{
    // ありえないと思うが nullptr の場合は Abortさせる
    NN_ABORT_UNLESS_NOT_NULL(pOutInfo);

    // ユーザーアカウントの選択画面の表示
    nn::account::Uid uid;
    nn::account::UserSelectionSettings setting = nn::account::DefaultUserSelectionSettings;
    setting.additionalSelect = true;
    auto result = nn::account::ShowUserSelector(&uid, setting);
    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        // キャンセルされた場合はなにもしない
        return false;
    }

    auto isOpened = this->GetOpenedUserInfo(uid, pOutInfo);
    if (isOpened)
    {
        // Open 済であれば指定できないアプレットなのであり得ないはずだが、Open 済であればその情報を取得して返す
        return true;
    }

    pOutInfo->id = uid;

    // 選択されたユーザーアカウントのニックネームの取得
    nn::account::Nickname name;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, uid));

    pOutInfo->name = name.name;

    // 選択されたユーザーを Open
    nn::account::UserHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, uid));

    pOutInfo->handle = handle;

    TRACE_ACCOUNT_MANAGER_LOG(" AccountManager::OpenUserWithSelectorApplet : push_back()\n");
    // 内部情報として Open したユーザー情報を保持しておく
    m_OpenedUserList.push_back(*pOutInfo);

    return true;
}

void AccountManager::CloseUser(const nn::account::Uid& inUid) NN_NOEXCEPT
{
    auto itr = std::begin(m_OpenedUserList);
    for (; itr != std::end(m_OpenedUserList); ++itr)
    {
        if (inUid == itr->id)
        {
            TRACE_ACCOUNT_MANAGER_LOG(" AccountManager::CloseUser : Call nn::accout::CloseUser()\n");
            // 該当する Uid のユーザーを Close する
            nn::account::CloseUser(itr->handle);
            // 内部情報も消しておく
            m_OpenedUserList.erase(itr);
            break;
        }
    }
}

bool AccountManager::GetOpenedUserInfo(const nn::account::Uid& inUid, UserInfo* pOutUserInfo) NN_NOEXCEPT
{
    for (auto&& user : m_OpenedUserList)
    {
        if (inUid == user.id)
        {
            if (pOutUserInfo != nullptr)
            {
                *pOutUserInfo = user;
            }
            return true;
        }
    }
    return false;
}

bool AccountManager::PopOpenUsers() NN_NOEXCEPT
{
    nn::account::UserHandle handleList[nn::account::UserCountMax];
    int count = 0;
    nn::account::PopOpenUsers(&count, handleList, nn::account::UserCountMax);

    TRACE_ACCOUNT_MANAGER_LOG("[Trace] AccountManager::PopOpenUsers() : count = %d\n", count);
    if (0 == count)
    {
        // ユーザー情報が取得できなければ処理を返す
        return false;
    }

    // 念のためのクリア処理
    m_OpenedUserList.clear();

    UserInfo info;
    for (int i = 0; i < count; ++i)
    {
        info.handle = handleList[i];

        // ハンドル情報から Uid 情報の取得
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&(info.id), info.handle));

        // Uid 情報からニックネームの取得
        nn::account::Nickname name;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, info.id));
        info.name = name.name;

        m_OpenedUserList.push_back(info);
    }

    return true;
}

void AccountManager::PushOpenUsers() NN_NOEXCEPT
{
    TRACE_ACCOUNT_MANAGER_LOG("[Trace] AccountManager::PushOpenUsers() : Start\n");
    nn::account::PushOpenUsers();
    TRACE_ACCOUNT_MANAGER_LOG("[Trace] AccountManager::PushOpenUsers() : End\n");
}

void AccountManager::GetOpenedUserList(std::list<UserInfo>* outOpenedUserList) NN_NOEXCEPT
{
    // ありえないと思うが nullptr の場合は Abortさせる
    NN_ABORT_UNLESS_NOT_NULL(outOpenedUserList);

    *outOpenedUserList = m_OpenedUserList;
}
