﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;
using System.Text;

namespace MakeTestApplication
{
    internal class Executor
    {
        private const string AuthoringToolDirPath = @"Tools\CommandLineTools\AuthoringTool";
        private bool isVerbose = false;
        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        /// <summary>
        /// 実行結果の情報を格納するクラス定義
        /// </summary>
        internal class ExecResult
        {
            public int ExitCode { get; set; }
            public string ErrorMessage { get; set; }

            public bool IsSuccess()
            {
                return (ExitCode == 0) ? true : false;
            }

            public bool IsFailure()
            {
                return !IsSuccess();
            }
        }

        /// <summary>
        /// AuthoringTool.exe を creatensp オプションで実行するメソッドです。(ダミーコード版)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="codeFilePath">入力コードファイルのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolCreateNsp(string outFilePath, string codeFilePath, SettingParams inSetParam)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("creatensp -o \"{0}\" --meta \"{1}\" --type \"{2}\" --program \"{3}\"",
                outFilePath, inSetParam.MetaFilePath, inSetParam.ContentType, codeFilePath);
            return InternalExecuteAuthoringTool(args, this.GetConfigKeyFilePath(inSetParam));
        }

        /// <summary>
        /// AuthoringTool.exe を creatensp オプションで実行するメソッドです。(詳細形式版)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <param name="codeDirPath">入力codeフォルダのパス</param>
        /// <param name="dataDirPath">入力dataフォルダのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolCreateNsp(
            string outFilePath, string descFilePath, string codeDirPath, string dataDirPath, SettingParams inSetParam)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("creatensp -o \"{0}\" --meta \"{1}\" --desc \"{2}\" --type \"{3}\" --program \"{4}\" \"{5}\"",
                                outFilePath, inSetParam.MetaFilePath, descFilePath, inSetParam.ContentType, codeDirPath, dataDirPath);

            if (inSetParam.KeyIndexNum != null && inSetParam.KeyIndexNum != string.Empty)
            {
                // keyindex の値はそのまま指定
                args += string.Format(" --keyindex {0}", inSetParam.KeyIndexNum);
            }

            if (inSetParam.KeyConfigFilePath != null && inSetParam.KeyConfigFilePath != string.Empty)
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var dstKeyConfigFile = Path.Combine(inSetParam.ChildWorkDirPath, "keyconfig.xml");
                File.Copy(inSetParam.KeyConfigFilePath, dstKeyConfigFile);
                // keyconfig のファイルパスを指定 (こちらは明示的な付加用の設定)
                args += string.Format(" --keyconfig \"{0}\"", dstKeyConfigFile);
            }

            // (SIGLO-49529) CreateNsp の --original オプション対応
            if (string.IsNullOrEmpty(inSetParam.OriginalApplicationForCreateNspFilePath) == false)
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var dstOriginalNspFile = Path.Combine(inSetParam.ChildWorkDirPath, CommonParams.OriginalApplicationCopyFileName);
                File.Copy(inSetParam.OriginalApplicationForCreateNspFilePath, dstOriginalNspFile);
                args += string.Format(" --original \"{0}\"", dstOriginalNspFile);
            }

            if (inSetParam.TicketFlag)
            {
                // ticket が有効であれば設定する
                args += " --ticket";
            }

            if (inSetParam.NoTicketFlag)
            {
                // no-ticket が有効であれば設定する
                args += " --no-ticket";
            }

            // (SIGLO-68043) 直接文字列指定の特殊オプション対応
            if (string.IsNullOrEmpty(inSetParam.DirectString) == false)
            {
                args += (" " + inSetParam.DirectString);
            }

            return InternalExecuteAuthoringTool(args, this.GetConfigKeyFilePath(inSetParam));
        }

        /// <summary>
        /// AuthoringTool.exe を creatensp オプションで実行するメソッドです。(SystemData作成版)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="dataDirPath">入力dataフォルダのパス</param>
        /// /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolCreateNspData(string outFilePath, string dataDirPath, SettingParams inSetParam)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("creatensp -o \"{0}\" --meta \"{1}\" --type {2}",
                                outFilePath, inSetParam.MetaFilePath, inSetParam.ContentType);

            if (dataDirPath != null && dataDirPath != string.Empty)
            {
                args += string.Format(" --data \"{0}\"", dataDirPath);
            }

            if (inSetParam.KeyIndexNum != null && inSetParam.KeyIndexNum != string.Empty)
            {
                // keyindex の値はそのまま指定
                args += string.Format(" --keyindex {0}", inSetParam.KeyIndexNum);
            }

            if (inSetParam.KeyConfigFilePath != null && inSetParam.KeyConfigFilePath != string.Empty)
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var dstKeyConfigFile = Path.Combine(inSetParam.ChildWorkDirPath, "keyconfig.xml");
                File.Copy(inSetParam.KeyConfigFilePath, dstKeyConfigFile);
                // keyconfig のファイルパスを指定 (こちらは明示的な付加用の設定)
                args += string.Format(" --keyconfig \"{0}\"", dstKeyConfigFile);
            }

            if (inSetParam.TicketFlag)
            {
                // ticket が有効であれば設定する
                args += " --ticket";
            }

            if (inSetParam.NoTicketFlag)
            {
                // no-ticket が有効であれば設定する
                args += " --no-ticket";
            }

            // (SIGLO-68043) 直接文字列指定の特殊オプション対応
            if (string.IsNullOrEmpty(inSetParam.DirectString) == false)
            {
                args += (" " + inSetParam.DirectString);
            }

            return InternalExecuteAuthoringTool(args, this.GetConfigKeyFilePath(inSetParam));
        }

        /// <summary>
        /// AuthoringTool.exe を makepatch オプションで実行するメソッドです。
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <param name="currentNspFilePath">入力currentNspファイルのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolMakePatch(string outFilePath, string descFilePath, string currentNspFilePath, SettingParams inSetParam)
        {
            // OriginalApplication は設定されているはず
            // スレッドセーフ化のため作業ディレクトリにコピーする
            var dstOriginalNspFile = Path.Combine(inSetParam.ChildWorkDirPath, CommonParams.OriginalApplicationCopyFileName);
            // (SIGLO-49529) OriginalApplicationForCreateNspFilePath が設定されていればコピー済なはず
            if (string.IsNullOrEmpty(inSetParam.OriginalApplicationForCreateNspFilePath))
            {
                File.Copy(inSetParam.OriginalApplicationFilePath, dstOriginalNspFile);
            }

            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("makepatch -o \"{0}\" --desc \"{1}\" --original \"{2}\" --current \"{3}\"",
                                        outFilePath, descFilePath, dstOriginalNspFile, currentNspFilePath);

            if (inSetParam.PreviousPatchFilePath != null && inSetParam.PreviousPatchFilePath != string.Empty)
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var dstPreviousNspFile = Path.Combine(inSetParam.ChildWorkDirPath, "previous_patch.nsp");
                File.Copy(inSetParam.PreviousPatchFilePath, dstPreviousNspFile);
                // previous のファイルパスを指定
                args += string.Format(" --previous \"{0}\"", dstPreviousNspFile);

                // previous が指定された場合はさらに --debug-delta オプションの有無を確認する
                if (inSetParam.DebugDeltaFlag)
                {
                    // debug-delta が有効であれば設定する
                    args += " --with-delta --merge-delta";
                }
            }

            // (SIGLO-73606) makepatch にも key 関連の設定オプションを適用する
            if (string.IsNullOrEmpty(inSetParam.KeyIndexNum) == false)
            {
                // keyindex の値はそのまま指定
                args += string.Format(" --keyindex {0}", inSetParam.KeyIndexNum);
            }

            // (SIGLO-73606) makepatch にも key 関連の設定オプションを適用する
            if (string.IsNullOrEmpty(inSetParam.KeyConfigFilePath) == false)
            {
                var dstKeyConfigFile = Path.Combine(inSetParam.ChildWorkDirPath, "keyconfig.xml");
                // コピー処理は current 生成時に実施済
                // keyconfig のファイルパスを指定
                args += string.Format(" --keyconfig \"{0}\"", dstKeyConfigFile);
            }

            // (SIGLO-73606) makepatch 用の直接文字列指定の特殊オプション対応
            if (string.IsNullOrEmpty(inSetParam.DirectStringForMakePatch) == false)
            {
                args += (" " + inSetParam.DirectStringForMakePatch);
            }

            return InternalExecuteAuthoringTool(args, null);
        }

        //(SIGLO-79140) マルチプログラムアプリケーション用のパッチ生成処理を特別に追加
        /// <summary>
        /// AuthoringTool.exe を makepatch オプションで実行するメソッドです。(マルチプログラムアプリケーション向け)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="descFilePath">入力descファイルのパス</param>
        /// <param name="currentNspFilePath">入力currentNspファイルのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolMakePatchForMultiProgram(string outFilePath, string descFilePath, string currentNspFilePath, SettingParams inSetParam)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = string.Format("makepatch -o \"{0}\" --desc \"{1}\" --original \"{2}\" --current \"{3}\"",
                                        outFilePath, descFilePath, inSetParam.OriginalApplicationFilePath, currentNspFilePath);

            if (string.IsNullOrEmpty(inSetParam.PreviousPatchFilePath) == false)
            {
                // previous のファイルパスを指定
                args += string.Format(" --previous \"{0}\"", inSetParam.PreviousPatchFilePath);

                // previous が指定された場合はさらに --debug-delta オプションの有無を確認する
                if (inSetParam.DebugDeltaFlag)
                {
                    // debug-delta が有効であれば設定する
                    args += " --with-delta --merge-delta";
                }
            }

            if (string.IsNullOrEmpty(inSetParam.KeyIndexNum) == false)
            {
                // keyindex の値はそのまま指定
                args += string.Format(" --keyindex {0}", inSetParam.KeyIndexNum);
            }

            if (string.IsNullOrEmpty(inSetParam.KeyConfigFilePath) == false)
            {
                // keyconfig のファイルパスを指定
                args += string.Format(" --keyconfig \"{0}\"", inSetParam.KeyConfigFilePath);
            }

            // makepatch 用の直接文字列指定の特殊オプション対応
            if (string.IsNullOrEmpty(inSetParam.DirectStringForMakePatch) == false)
            {
                args += (" " + inSetParam.DirectStringForMakePatch);
            }

            return InternalExecuteAuthoringTool(args, null);
        }

        /// <summary>
        /// AuthoringTool.exe を mergeapplication オプションで実行するメソッドです。(マルチプログラムアプリケーション作成用)
        /// </summary>
        /// <param name="outFilePath">出力nspファイルのパス</param>
        /// <param name="inInputNspFileList">入力nspファイルパラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteAuthoringToolMergeApplication(string outFilePath, List<string> inInputNspFilePathList)
        {
            // AuthoringToolに渡すオプション引数文字列を生成
            var args = new StringBuilder();
            args.AppendFormat("mergeapplication -o \"{0}\"", outFilePath);

            foreach (var inputNspPath in inInputNspFilePathList)
            {
                args.AppendFormat(" \"{0}\"", inputNspPath);
            }

            return InternalExecuteAuthoringTool(args.ToString(), null);
        }

        private ExecResult InternalExecuteAuthoringTool(string args, string inKeyPath)
        {
            if (inKeyPath != null && inKeyPath != string.Empty)
            {
                // keyconfig のファイルパスを指定 (こちらは自動付加用の設定)
                args += string.Format(" --keyconfig \"{0}\"", inKeyPath);
            }

            var result = new ExecResult();
            using (var process = new Process())
            {
                var authoringToolDirPath = CommonParams.GetSigloRoot() + AuthoringToolDirPath;
                process.StartInfo.FileName = Path.Combine(authoringToolDirPath, "AuthoringTool.exe");
                process.StartInfo.Arguments = args;

                if (isVerbose == true)
                {
                    Console.WriteLine("Execute AuthoringTool ...");
                    Console.WriteLine("  arg : {0}", process.StartInfo.Arguments);
                }

                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                //process.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
                process.Start();

                process.WaitForExit();
                result.ExitCode = process.ExitCode;
                result.ErrorMessage = process.StandardError.ReadToEnd();
            }

            return result;
        }

        /// <summary>
        /// MakeMeta.exe を実行するメソッドです。
        /// </summary>
        /// <param name="outNpdmFilePath">出力npdmファイルのパス</param>
        /// <param name="inDescFilePath">入力descファイルのパス</param>
        /// <param name="inSetParam">設定パラメータ情報</param>
        /// <returns>実行結果</returns>
        public ExecResult ExecuteMakeMeta(string outNpdmFilePath, string inDescFilePath, SettingParams inSetParam)
        {
            var result = new ExecResult();
            using (var process = new Process())
            {
                process.StartInfo.FileName = CommonParams.GetSigloRoot() + @"Tools\CommandLineTools\MakeMeta\MakeMeta.exe";
                var args = string.Format("-o \"{0}\" --desc \"{1}\" --meta \"{2}\"", outNpdmFilePath, inDescFilePath, inSetParam.MetaFilePath);

                if (inSetParam.CheckValidRangeIdFlag == false)
                {
                    args += " --no_check_programid";
                }

                process.StartInfo.Arguments = args;

                if (isVerbose == true)
                {
                    Console.WriteLine(" ExecuteMakeMeta arg : {0}", process.StartInfo.Arguments);
                }

                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                process.StartInfo.StandardErrorEncoding = System.Text.Encoding.UTF8;
                process.Start();

                process.WaitForExit();
                result.ExitCode = process.ExitCode;
                result.ErrorMessage = process.StandardError.ReadToEnd();
            }
            return result;
        }

        private string GetConfigKeyFilePath(SettingParams inSetParam)
        {
            // Keyファイルが必要ない場合は空文字を返す
            var keyPath = string.Empty;

            if (inSetParam.KeyConfigFilePath != null && inSetParam.KeyConfigFilePath != string.Empty)
            {
                // keyconfig が明示的に設定されていることになるので空文字を返して、自動付加設定を行わないようにする
                return keyPath;
            }

            // 現状、System系のコンテンツタイプにおいて keyconfig の設定が必要
            if (inSetParam.ContentType == CommonParams.ContentTypeSystemProgram ||
                inSetParam.ContentType == CommonParams.ContentTypeSystemData)
            {
                // スレッドセーフ化のため作業ディレクトリにコピーする
                var authoringToolDirPath = CommonParams.GetSigloRoot() + AuthoringToolDirPath;
                var keyFileName = "AuthoringTool.ocean.keyconfig.xml";
                var srcKeyPath = Path.Combine(authoringToolDirPath, keyFileName);
                keyPath = Path.Combine(inSetParam.ChildWorkDirPath, keyFileName);
                File.Copy(srcKeyPath, keyPath);
            }
            return keyPath;
        }
    }
}
