﻿using System;
using System.IO;
using System.Runtime.InteropServices;

namespace MakeMiniDump
{
    internal sealed class MiniDump
    {
        [Flags]
        internal enum DumpType : uint
        {
            Normal = 0x00000000,
            WithDataSegs = 0x00000001,
            WithFullMemory = 0x00000002,
            WithHandleData = 0x00000004,
            FilterMemory = 0x00000008,
            ScanMemory = 0x00000010,
            WithUnloadedModules = 0x00000020,
            WithIndirectlyReferencedMemory = 0x00000040,
            FilterModulePaths = 0x00000080,
            WithProcessThreadData = 0x00000100,
            WithPrivateReadWriteMemory = 0x00000200,
            WithoutOptionalData = 0x00000400,
            WithFullMemoryInfo = 0x00000800,
            WithThreadInfo = 0x00001000,
            WithCodeSegs = 0x00002000,
            WithoutAuxiliaryState = 0x00004000,
            WithFullAuxiliaryState = 0x00008000,
            WithPrivateWriteCopyMemory = 0x00010000,
            IgnoreInaccessibleMemory = 0x00020000,
            WithTokenInformation = 0x00040000,
            WithModuleHeaders = 0x00080000,
            FilterTriage = 0x00100000,
            ValidTypeFlags = 0x001fffff,
        }

        [Flags]
        private enum ProcessAccessRights : uint
        {
            All = 0x1f0fff, // Without QueryLimitedInformation.
            Terminate = 0x0001,
            CreateThread = 0x0002,
            SetSessionid = 0x0004,
            VmOperation = 0x0008,
            VmRead = 0x0010,
            VmWrite = 0x0020,
            DupHandle = 0x0040,
            CreateProcess = 0x0080,
            SetQuota = 0x0100,
            SetInformation = 0x0200,
            QueryInformation = 0x0400,
            SuspendResume = 0x0800,
            QueryLimitedInformation = 0x1000, // Not supported on Windows Server 2003 and Windows XP.
            Delete = 0x10000,
            ReadControl = 0x20000,
            WriteDac = 0x40000,
            WriteOwner = 0x80000,
            Synchronize = 0x100000,
        };

        [DllImport("dbghelp.dll", SetLastError = true)]
        private static extern bool MiniDumpWriteDump(
            IntPtr hProcess, uint processId, SafeHandle hFile, uint dumpType,
            IntPtr expParam, IntPtr userStreamParam, IntPtr callbackParam);

        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern IntPtr OpenProcess(
            uint dwDesiredAccess, bool bInheritHandle, uint dwProcessId);

        internal static void Write(uint processId, string filePath, DumpType dumpType)
        {
            var processHandle = OpenProcess((uint)ProcessAccessRights.All, false, processId);

            if (processHandle == IntPtr.Zero)
            {
                Marshal.ThrowExceptionForHR(Marshal.GetHRForLastWin32Error());
            }

            using (var fs = new FileStream(filePath, FileMode.Create, FileAccess.Write))
            {
                var isSuccess = MiniDumpWriteDump(
                    processHandle, processId, fs.SafeFileHandle, (uint)dumpType,
                    IntPtr.Zero, IntPtr.Zero, IntPtr.Zero);

                if (!isSuccess)
                {
                    Marshal.ThrowExceptionForHR(Marshal.GetHRForLastWin32Error());
                }
            }
        }
    }
}
