﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace DecodeStackTrace
{
    internal class SymbolTable
    {
        internal class Symbol
        {
            public long address;
            public long size;
            public string name;
            public SymbolType type;
            public string file;
            public int lineNumber;
        }

        internal enum SymbolType
        {
            GlobalAbsoluteSymbol,
            LocalAbsoluteSymbol,
            GlobalBssSymbol,
            LocalBssSymbol,
            GlobalDataSymbol,
            LocalDataSymbol,
            SourceFileNameSymbol,
            GlobalThreadLocalSymbol,
            StaticThreadLocalSymbol,
            GlobalReadOnlyDataSymbol,
            LocalReadOnlyDataSymbol,
            GlobalTextSymbol,
            LocalTextSymbol,
            UndefinedSymbol,
            GlobalTaggedWeakSymbol,
            LocalTaggedWeakSymbol,
            GlobalUnTaggedWeakSymbol,
            LocalUnTaggedWeakSymbol,
        }

        private static readonly Dictionary<char, SymbolType> SymbolTypeTable = new Dictionary<char, SymbolType>()
        {
            { 'A', SymbolType.GlobalAbsoluteSymbol },
            { 'a', SymbolType.LocalAbsoluteSymbol },
            { 'B', SymbolType.GlobalBssSymbol },
            { 'b', SymbolType.LocalBssSymbol },
            { 'D', SymbolType.GlobalDataSymbol },
            { 'd', SymbolType.LocalDataSymbol },
            { 'F', SymbolType.SourceFileNameSymbol },
            { 'L', SymbolType.GlobalThreadLocalSymbol },
            { 'l', SymbolType.StaticThreadLocalSymbol },
            { 'R', SymbolType.GlobalReadOnlyDataSymbol },
            { 'r', SymbolType.LocalReadOnlyDataSymbol },
            { 'T', SymbolType.GlobalTextSymbol },
            { 't', SymbolType.LocalTextSymbol },
            { 'U', SymbolType.UndefinedSymbol },
            { 'V', SymbolType.GlobalTaggedWeakSymbol },
            { 'v', SymbolType.LocalTaggedWeakSymbol },
            { 'W', SymbolType.GlobalUnTaggedWeakSymbol },
            { 'w', SymbolType.LocalUnTaggedWeakSymbol },
        };

        internal SymbolTable(string elfFilePath)
        {
            if (!File.Exists(elfFilePath))
            {
                throw new FileNotFoundException("The specified elf file is not found.", elfFilePath);
            }

            symbolList = new Lazy<List<Symbol>>(() => CreateSymbolList(elfFilePath));
        }

        internal Symbol QuerySymbol(long address)
        {
            address--; // 関数末尾で bl 命令が実行されると、戻りアドレスが関数外を指すため -1 して検索する。
            var begin = 0;
            var end = SymbolList.Count;
            do
            {
                var pivot = (begin + end) / 2;
                var symbol = SymbolList[pivot];
                if (address < symbol.address)
                {
                    end = pivot;
                }
                else
                {
                    if (address < symbol.address + symbol.size)
                    {
                        return symbol;
                    }
                    begin = pivot + 1;
                }
            } while (begin < end);
            throw new ArgumentOutOfRangeException();
        }

        private static List<Symbol> CreateSymbolList(string elfFilePath)
        {
            var list = EnumerateSymbols(elfFilePath).ToList();
            list.Sort((x, y) => longComparison(x.address, y.address));
            return list;
        }

        private static IEnumerable<Symbol> EnumerateSymbols(string elfFilePath)
        {
            var output = ProcessInvoker.Invoke(EnvironmentInfo.NmPath,
                $"--demangle --line-numbers --print-size {elfFilePath}");
            using (var sr = new StreamReader(new MemoryStream(Encoding.UTF8.GetBytes(output))))
            {
                while (!sr.EndOfStream)
                {
                    var line = sr.ReadLine();

                    var m = nmOutputPattern.Match(line);
                    if (!m.Success)
                    {
                        continue;
                    }

                    yield return new Symbol()
                    {
                        address = long.Parse(m.Groups["address"].Value, NumberStyles.HexNumber),
                        size = long.Parse(m.Groups["size"].Value, NumberStyles.HexNumber),
                        type = SymbolTypeTable[char.Parse(m.Groups["type"].Value)],
                        name = m.Groups["name"].Value,
                        file = m.Groups["file"].Value,
                        lineNumber = string.IsNullOrEmpty(m.Groups["lineNumber"].Value) ? -1 : int.Parse(m.Groups["lineNumber"].Value),
                    };
                }
            }
        }

        private static Comparison<long> longComparison = new Comparison<long>((x, y) =>
        {
            var diff = x - y;
            return diff >= int.MaxValue ? int.MaxValue : (diff <= int.MinValue ? int.MinValue : (int)diff);
        });

        private static readonly Regex nmOutputPattern = new Regex(@"(?<address>[0-9a-f]{8,16}) (?<size>[0-9a-f]{8,16}) (?<type>\w) (?<name>[^\t]+)(\t(?<file>.+):(?<lineNumber>\d+))?");

        private List<Symbol> SymbolList
        {
            get
            {
                return symbolList.Value;
            }
        }

        private readonly Lazy<List<Symbol>> symbolList;


    }
}
