﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PropertySheetsTest
{
    public enum ProjectType
    {
        Library,
        DynamicLibrary,
        Program,
        ProgramWithoutSdkNso,
        SystemProgram,
    }

    public class TestConfiguration
    {
        public TestConfiguration(string subRootName, ProjectType projectType, SdkEnvironment sdkEnvironment, VcxprojConfiguration config)
        {
            this.IsDevelopmentSubRoot = GetIsDevelopmentSubRoot(subRootName, sdkEnvironment);
            this.SubRootName = subRootName;
            this.SubRootRelativeDir = GetSubRootRelativeDir(subRootName, sdkEnvironment);
            this.ProjectType = projectType;
            this.Spec = config.Spec;
            this.BuildType = config.BuildType;
            this.PlatformToolset = config.PlatformToolset;
            this.Configuration = config.Configuration;
            this.Platform = config.Platform;
            this.VsVersion = config.VsVersion;
            this.BuildTarget = GetBuildTarget(Spec, Platform, PlatformToolset);
            this.TargetName = GetTargetName(this.Spec, subRootName, projectType);

            this.IsWinPlatform = Platform == "Win32" || Platform == "x64";
            this.IsNXPlatform = Platform == "NX32" || Platform == "NX64";
        }

        public bool IsDevelopmentSubRoot { get; private set; }
        public string SubRootName { get; private set; }
        public string SubRootRelativeDir { get; private set; }
        public ProjectType ProjectType { get; private set; }
        public string Spec { get; private set; }
        public string BuildType { get; private set; }
        public string PlatformToolset { get; private set; }
        public string Configuration { get; private set; }
        public string Platform { get; private set; }
        public string BuildTarget { get; private set; }
        public string TargetName { get; private set; }
        public string VsVersion { get; private set; }
        public ConfigurationPair ConfigurationPair { get { return new ConfigurationPair(this.Configuration, this.Platform); } }

        public bool IsWinPlatform { get; private set; }
        public bool IsNXPlatform { get; private set; }

        private bool GetIsDevelopmentSubRoot(string subRootName, SdkEnvironment sdkEnvironment)
        {
            switch (subRootName)
            {
                case "Alice":
                case "Eris":
                case "Chris":
                    return true;
                case "Iris":
                case "NintendoWare":
                    return sdkEnvironment == SdkEnvironment.Development;
                case "Samples":
                case "Tests":
                case "Users":
                    return false;
                default:
                    throw new ArgumentException(string.Format("{0}: 未知の subRootName です", subRootName), "subRootName");
            }
        }

        private string GetSubRootRelativeDir(string subRootName, SdkEnvironment sdkEnvironment)
        {
            switch (subRootName)
            {
                case "Alice":
                case "Eris":
                case "Chris":
                    return string.Format(@"Programs\{0}", subRootName);
                case "Iris":
                case "NintendoWare":
                    switch (sdkEnvironment)
                    {
                        case SdkEnvironment.Development:
                            return string.Format(@"Programs\{0}", subRootName);
                        case SdkEnvironment.User:
                            return string.Empty;
                        default:
                            throw new ArgumentException($"未知の SdkEnvironment です: {sdkEnvironment}", nameof(sdkEnvironment));
                    }
                case "Samples":
                case "Tests":
                case "Users":
                    return subRootName;
                default:
                    throw new ArgumentException(string.Format("{0}: 未知の subRootName です", subRootName), "subRootName");
            }
        }

        private string GetTargetName(string spec, string subRootName, ProjectType projectType)
        {
            switch (subRootName)
            {
                case "Alice":
                case "Eris":
                case "Chris":
                case "Iris":
                case "NintendoWare":
                case "Samples":
                case "Tests":
                    switch (projectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.Program:
                            return $"testBuildSystem{subRootName}";
                        case ProjectType.DynamicLibrary:
                            return $"testBuildSystem{subRootName}Dll";
                        case ProjectType.ProgramWithoutSdkNso:
                            return $"testBuildSystem{subRootName}EmptyNso";
                        case ProjectType.SystemProgram:
                            return $"testBuildSystem{subRootName}SystemProgram";
                        default:
                            throw new ArgumentException($"未知の ProjectType です: {projectType}", nameof(projectType));
                    }
                case "Users":
                    switch (projectType)
                    {
                        case ProjectType.Library:
                        case ProjectType.Program:
                            return $"testBuildSystem{subRootName}{spec}";
                        case ProjectType.DynamicLibrary:
                            return $"testBuildSystem{subRootName}{spec}Dll";
                        case ProjectType.ProgramWithoutSdkNso:
                            return $"testBuildSystem{subRootName}{spec}EmptyNso";
                        case ProjectType.SystemProgram:
                            return $"testBuildSystem{subRootName}{spec}SystemProgram";
                        default:
                            throw new ArgumentException($"未知の ProjectType です: {projectType}", nameof(projectType));
                    }
                default:
                    throw new ArgumentException(string.Format("{0}: 未知の subRootName です", subRootName), "subRootName");
            }
        }

        private string GetBuildTarget(string spec, string platform, string platformToolset)
        {
            // candidate: 命名規則通りのビルドターゲット名 <Spec>-<Platform>-<PlatformToolset>
            // * <Spec>- は、Generic の場合は省略する
            // * -<PlatformToolset> は、Win32/x64 プラットフォームの場合のみ付加する
            string candidate;
            switch (platform)
            {
                case "Win32":
                case "x64":
                    candidate = string.Format(spec == "Generic" ? "{1}-{2}" : "{0}-{1}-{2}", spec, platform, platformToolset);
                    break;
                default:
                    candidate = string.Format(spec == "Generic" ? "{1}" : "{0}-{1}", spec, platform);
                    break;
            }

            switch (candidate)
            {
                // Generic spec
                case "Win32-v110":
                    return "Win32";
                case "Cafe":
                    return "GenericOnCafe";
                // NX spec
                case "NX-Win32-v110":
                    return "NXOnWin32";
                case "NX-NX32":
                    return "NX-NXFP2-a32";
                case "NX-NX64":
                    return "NX-NXFP2-a64";
                // Cafe spec
                case "Cafe-Win32-v110":
                    return "CafeOnWin32";
                case "Cafe-Cafe":
                    return "Cafe";
                // default
                default:
                    return candidate;
            }
        }
    }
}
