﻿using Microsoft.VisualStudio.Setup.Configuration;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.Nact.Utility
{
    public class VisualStudioInstance
    {
        private readonly ISetupInstance2 m_InnerObject;
        private const string PropertyNameChannelManifestId = "channelManifestId";

        /// <summary>
        /// channelManifestId プロパティの値。
        ///
        /// 例:
        ///
        /// - VisualStudio.15.Preview/15.3.0-Preview+26510.0
        /// - VisualStudio.15.Release/15.2.0+26430.6
        /// </summary>
        public string ChannelManifestId { get; }
        public string InstallationName { get; }
        public string InstallationPath { get; }
        public string InstallationVersion { get; }
        public IEnumerable<string> Components { get; }
        public string ProductId { get; }
        public string ProductUniqueId { get; }

        public VisualStudioInstance(ISetupInstance2 instance)
        {
            this.m_InnerObject = instance;
            this.ChannelManifestId = GetProperty(instance, PropertyNameChannelManifestId) ?? string.Empty;
            this.InstallationName = instance.GetInstallationName();
            this.InstallationPath = instance.GetInstallationPath();
            this.InstallationVersion = instance.GetInstallationVersion();
            this.Components = GetComponents(instance);

            var product = instance.GetProduct();
            this.ProductId = product.GetId();
            this.ProductUniqueId = product.GetUniqueId();
        }

        /// <summary>
        /// インストールが完了しているかどうか
        /// </summary>
        public bool IsComplete => m_InnerObject.GetState() == InstanceState.Complete;

        /// <summary>
        /// このインスタンスが指定のチャンネルからインストールされたものかどうか
        /// </summary>
        /// <param name="channelId">チャンネルID (例: VisualStudio.15.Release)</param>
        public bool IsFromChannel(string channelId) => ChannelManifestId.StartsWith(channelId);

        private static string GetProperty(ISetupInstance2 instance, string name)
        {
            var props = instance.GetProperties();
            var names = props.GetNames();
            if (names.Contains(name))
            {
                return props.GetValue(name).ToString();
            }
            else
            {
                return null;
            }
        }

        private static string[] GetComponents(ISetupInstance2 instance)
        {
            return instance
                .GetPackages()
                .Where(x => x.GetType() == "Component")
                .Select(x => x.GetId())
                .ToArray();
        }
    }
}
