﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xunit;

namespace PropertySheetsTest
{
    internal static class AssertUtil
    {
        /// <summary>
        /// Enumerable.SequenceEqual(xs, ys, EqualityComparer<T>.Default) をテストします。
        /// </summary>
        public static void AssertSequenceEqual<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            AssertSequenceEqual(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// Enumerable.SequenceEqual(xs, ys, equalityComparer) をテストします。
        /// </summary>
        public static void AssertSequenceEqual<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            if (xs.Count() != ys.Count())
            {
                throw CreateAssertSequenceEqualFailedException(xs, ys, "要素数が異なります。");
            }
            if (!xs.SequenceEqual(ys, equalityComparer))
            {
                throw CreateAssertSequenceEqualFailedException(xs, ys, "要素の値が異なります。");
            }
        }

        /// <summary>
        /// Comparer<T>.Default によってシーケンスを並びかえてから Enumerable.SequenceEqual(xs, ys, EqualityComparer<T>.Default) をテストします。
        /// </summary>
        public static void AssertOrderedSequenceEqual<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            AssertOrderedSequenceEqual(xs, ys, EqualityComparer<T>.Default, Comparer<T>.Default);
        }

        /// <summary>
        /// comparer によってシーケンスを並びかえてから Enumerable.SequenceEqual(xs, ys, equalityComparer) をテストします。
        /// </summary>
        public static void AssertOrderedSequenceEqual<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer, IComparer<T> comparer)
        {
            if (xs.Count() != ys.Count())
            {
                throw CreateAssertSequenceEqualFailedException(xs, ys, "要素数が異なります。");
            }
            if (!xs.OrderBy(x => x, comparer).SequenceEqual(ys.OrderBy(y => y, comparer), equalityComparer))
            {
                throw CreateAssertSequenceEqualFailedException(xs, ys, "要素の値が異なります。");
            }
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, EqualityComparer<T>.Default) をテストします。
        /// </summary>
        public static void AssertSequenceContainsAnyElement<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            AssertSequenceContainsAnyElement(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, equalityComparer) をテストします。
        /// </summary>
        public static void AssertSequenceContainsAnyElement<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            if (!Util.SequenceContainsAnyElement(xs, ys, equalityComparer))
            {
                throw CreateAssertSequenceContainsElementsFailedException(xs, ys, equalityComparer, "AssertSequenceContainsAnyElement");
            }
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, EqualityComparer<T>.Default) をテストします。
        /// </summary>
        public static void AssertSequenceContainsElements<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            AssertSequenceContainsElements(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, equalityComparer) をテストします。
        /// </summary>
        public static void AssertSequenceContainsElements<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            if (!Util.SequenceContainsElements(xs, ys, equalityComparer))
            {
                throw CreateAssertSequenceContainsElementsFailedException(xs, ys, equalityComparer, "AssertSequenceContainsElements");
            }
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, EqualityComparer<T>.Default) をテストします。
        /// </summary>
        public static void AssertSequenceContainsElementsInOrder<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            AssertSequenceContainsElementsInOrder(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// Util.SequenceContainsElementsInOrder(xs, ys, equalityComparer) をテストします。
        /// </summary>
        public static void AssertSequenceContainsElementsInOrder<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            if (!Util.SequenceContainsElementsInOrder(xs, ys, equalityComparer))
            {
                throw CreateAssertSequenceContainsElementsFailedException(xs, ys, equalityComparer, "AssertSequenceContainsElementsInOrder");
            }
        }

        private static Exception CreateAssertSequenceEqualFailedException<T>(IEnumerable<T> xs, IEnumerable<T> ys, string msg)
        {
            return new Exception(string.Format(
                "{0}: \n" +
                "xs:\n{1}\n\n" +
                "ys:\n{2}\n\n",
                msg,
                string.Join("\n", xs.Select(x => x.ToString()).ToArray()),
                string.Join("\n", ys.Select(y => y.ToString()).ToArray())
            ));
        }

        private static Exception CreateAssertSequenceContainsElementsFailedException<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer, string msg)
        {
            return new Exception(string.Format(
                "{0}: \n" +
                "xs:\n{1}\n\n" +
                "ys:\n{2}\n\n" +
                "xs.Intersect(ys):\n{3}\n\n",
                msg,
                string.Join("\n", xs.Select(x => x.ToString()).ToArray()),
                string.Join("\n", ys.Select(y => y.ToString()).ToArray()),
                string.Join("\n", xs.Intersect(ys, equalityComparer).Select(x => x.ToString()).ToArray())
                ));
        }
    }
}
