﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\..\..\..\..\Programs\Eris\Sources\Libraries\audioctrl\server\detail\audioctrl_PlayReportController.h"

#include <nn/init.h>
#include <nnt.h>

// システムプロセス権限で動作するテストであるため、
// malloc 用のヒープをメモリヒープではなく bss から確保する。
namespace {
NN_ALIGNAS(4096) uint8_t g_MallocBuffer[1 * 1024 * 1024];
}

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

TEST(InitializeAndFinalize, Success)
{
    nn::audioctrl::server::detail::PlayReportController controller;

    controller.Initialize(nn::TimeSpan::FromSeconds(1));
    controller.Finalize();
}

TEST(Commit, Success)
{
    nn::audioctrl::server::detail::PlayReportController controller;
    nn::os::NativeHandle volumeUpdateEventHandle;
    nn::os::NativeHandle outputTargetUpdateEventHandle;
    nn::os::SystemEvent volumeUpdateEvent;
    nn::os::SystemEvent outputTargetUpdateEvent;
    const auto CommitDelayTime = nn::TimeSpan::FromMilliSeconds(100);

    controller.Initialize(CommitDelayTime);

    // CommitTimer を取得
    auto& commitTimer = controller.GetCommitTimer();

    // 初期状態ではイベントがシグナルされてないことをチェック
    EXPECT_FALSE(commitTimer.TryWait());

    // VolumeUpdateEvent を取得
    controller.AcquireAudioVolumeUpdateEvent(&volumeUpdateEventHandle);
    ::nn::os::AttachReadableHandleToSystemEvent(volumeUpdateEvent.GetBase(),
                                                volumeUpdateEventHandle,
                                                false,
                                                nn::os::EventClearMode_AutoClear);

    // OutputTargetUpdateEvent を取得
    controller.AcquireAudioOutputDeviceUpdateEvent(&outputTargetUpdateEventHandle);
    ::nn::os::AttachReadableHandleToSystemEvent(outputTargetUpdateEvent.GetBase(),
                                                outputTargetUpdateEventHandle,
                                                false,
                                                nn::os::EventClearMode_AutoClear);

    // 初期状態ではイベントがシグナルされてないことをチェック
    EXPECT_FALSE(volumeUpdateEvent.TryWait());
    EXPECT_FALSE(outputTargetUpdateEvent.TryWait());

    // 値を通知する
    nn::audioctrl::PlayReportAudioVolume speakerVolume = { 1, false };
    nn::audioctrl::PlayReportAudioVolume headphoneVolume = { 10, true };
    nn::audioctrl::PlayReportAudioVolume usbOutputDeviceVolume = { 12, true };
    const auto OutputTarget = nn::audioctrl::PlayReportAudioOutputTarget_Speaker;
    controller.NotifyOutputDeviceUpdate(OutputTarget);
    controller.NotifyAudioVolumeUpdateForVolume(nn::audioctrl::PlayReportAudioOutputTarget_Speaker, speakerVolume);
    controller.NotifyAudioVolumeUpdateForVolume(nn::audioctrl::PlayReportAudioOutputTarget_Headphone, headphoneVolume);
    controller.NotifyAudioVolumeUpdateForVolume(nn::audioctrl::PlayReportAudioOutputTarget_UsbOutputDevice, usbOutputDeviceVolume);

    // コミットタイマーがシグナル状態になることをチェック
    commitTimer.Wait();
    commitTimer.Clear();

    // コミット処理
    controller.Commit();

    // 全ての情報が通知される前はイベントがシグナルされてないことをチェック
    EXPECT_FALSE(volumeUpdateEvent.TryWait());
    EXPECT_FALSE(outputTargetUpdateEvent.TryWait());

    // 残りの値を通知する
    const auto IsHeadphonePowerLimited = false;
    controller.NotifyAudioVolumeUpdateForHeadphonePowerLimit(IsHeadphonePowerLimited);

    // コミットタイマーがシグナル状態になることをチェック
    commitTimer.Wait();
    commitTimer.Clear();

    // コミット処理
    controller.Commit();

    // 全ての情報が通知される後はイベントがシグナルされることをチェック
    EXPECT_TRUE(volumeUpdateEvent.TryWait());
    EXPECT_TRUE(outputTargetUpdateEvent.TryWait());

    // 値が正しく反映されていることをチェック
    EXPECT_EQ(OutputTarget, controller.GetAudioOutputTarget());
    nn::audioctrl::PlayReportAudioVolumeData committedVolumeData{};
    controller.GetAudioVolumeData(&committedVolumeData);
    EXPECT_EQ(committedVolumeData.speaker.volume, speakerVolume.volume);
    EXPECT_EQ(committedVolumeData.speaker.mute, speakerVolume.mute);
    EXPECT_EQ(committedVolumeData.headphone.volume, headphoneVolume.volume);
    EXPECT_EQ(committedVolumeData.headphone.mute, headphoneVolume.mute);
    EXPECT_EQ(committedVolumeData.usbOutputDevice.volume, usbOutputDeviceVolume.volume);
    EXPECT_EQ(committedVolumeData.usbOutputDevice.mute, usbOutputDeviceVolume.mute);
    EXPECT_EQ(committedVolumeData.isHeadphonePowerLimited, IsHeadphonePowerLimited);

    controller.Finalize();
}
