﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_ScopedRenderer.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_EffectCommon.h" // QF_FRACTIONAL_BIT_COUNT
#include <cmath> //pow

namespace
{
    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
    nn::mem::StandardAllocator g_EffectBufferAllocator;
    bool g_FsInitialized = false;
    int g_VoiceMax = 256;
    NN_ALIGNAS(4096) char g_WorkBuffer[16 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[24 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];
    enum FileFormat { ADPCM, PCM };
    void* g_MountRomCacheBuffer = nullptr;
    size_t g_MountRomCacheBufferSize = 0;

    void* Allocate(size_t size)
    {
        return g_Allocator.Allocate(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        g_Allocator.Free(p);
    }

    void InitializeFileSystem()
    {
        if (!g_FsInitialized)
        {
            nn::fs::SetAllocator(Allocate, Deallocate);
            g_FsInitialized = true;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&g_MountRomCacheBufferSize));
        g_MountRomCacheBuffer = Allocate(g_MountRomCacheBufferSize);
        NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("asset", g_MountRomCacheBuffer, g_MountRomCacheBufferSize));
    }

    void FinalizeFileSystem()
    {
        nn::fs::Unmount("asset");
        Deallocate(g_MountRomCacheBuffer, g_MountRomCacheBufferSize);
        g_MountRomCacheBuffer = nullptr;
        g_MountRomCacheBufferSize = 0;
    }
}

void RunAdspPerformanceTest(FileFormat format)
{
    nnt::audio::ScopedRenderer renderer(g_Allocator, g_WaveBufferAllocator, g_EffectBufferAllocator);

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 48000 / 200;
    parameter.mixBufferCount = 6;
    parameter.voiceCount = g_VoiceMax;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 1;
    parameter.performanceFrameCount = 5;

    void* workBuffer;
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    renderer.RendererSetup(&parameter, 2, 4);
    workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::MemoryPoolType effectBufferPool;
    ASSERT_TRUE(AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory)));
    ASSERT_TRUE(RequestAttachMemoryPool(&effectBufferPool));

    const int channelCount = 2;
    nn::audio::BufferMixerType mixer;
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddBufferMixer(renderer.GetConfig(), &mixer, renderer.GetFinalMix()));
    nn::audio::SetBufferMixerInputOutput(&mixer, renderer.GetSubBus(), renderer.GetMainBus(), channelCount);
    nn::audio::SetBufferMixerVolume(&mixer, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer, 1, 1.0f);

    renderer.Start();
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    ASSERT_TRUE(AcquireMemoryPool(renderer.GetConfig(), &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory)));
    ASSERT_TRUE(RequestAttachMemoryPool(&waveBufferMemoryPool));

    int currentVoiceCount = 1;
    bool reachedMax = false;
    int thresholdCycles = 5;
    InitializeFileSystem();
    if( format == ADPCM )
    {
        renderer.InitializeAdpcmVoice(g_VoiceMax);
        renderer.AddAdpcmVoice(currentVoiceCount);
        while( currentVoiceCount < 50 )
        {
            renderer.Update();
            renderer.Wait();
            for( int i = 0; i < currentVoiceCount; ++i )
            {
                if( nn::audio::GetReleasedWaveBuffer(renderer.GetAdpcmVoice(i)) )
                {
                    nn::audio::AppendWaveBuffer(renderer.GetAdpcmVoice(i), renderer.GetAdpcmWaveBuffer(i % 4));
                }
            }
            currentVoiceCount++;
            renderer.AddAdpcmVoice(currentVoiceCount);
        }
    }
    else if( format == PCM )
    {
        renderer.InitializePcmVoice(g_VoiceMax);
        renderer.AddPcmVoice(currentVoiceCount);
        while( currentVoiceCount < 50 )
        {
            renderer.Update();
            renderer.Wait();
            for( int i = 0; i < currentVoiceCount; ++i )
            {
                if( nn::audio::GetReleasedWaveBuffer(renderer.GetPcmVoice(i)) )
                {
                    nn::audio::AppendWaveBuffer(renderer.GetPcmVoice(i), renderer.GetSineWaveBuffer());
                }
            }
            currentVoiceCount++;
            renderer.AddPcmVoice(currentVoiceCount);
        }
    }
    else
    {
        NN_LOG("Unrecognized File Format");
        ASSERT_TRUE(false);
    }

    while( !reachedMax && renderer.GetTotalDumpCount() < 15 )
    {
        renderer.Wait();
        for( int i = 0; i < currentVoiceCount; ++i )
        {
            if( format == ADPCM )
            {
                if( nn::audio::GetReleasedWaveBuffer(renderer.GetAdpcmVoice(i)) )
                {
                    nn::audio::AppendWaveBuffer(renderer.GetAdpcmVoice(i), renderer.GetAdpcmWaveBuffer(i % 4));
                }
            }
            else if( format == PCM )
            {
                if( nn::audio::GetReleasedWaveBuffer(renderer.GetPcmVoice(i)) )
                {
                    nn::audio::AppendWaveBuffer(renderer.GetPcmVoice(i), renderer.GetSineWaveBuffer());
                }
            }
        }
        renderer.Update();
        renderer.Wait();
        renderer.UpdatePerformanceBuffer(10);
        if( renderer.GetTotalDumpCount() > 0 )
        {
            NN_LOG("Processing time %d \n", renderer.GetProcessingTime());
            if( renderer.GetProcessingTime() > 4000 )
            {
                thresholdCycles--;
                if( thresholdCycles < 0 )
                {
                    reachedMax = true;
                }
            }
            else
            {
                if( currentVoiceCount < g_VoiceMax )
                {
                    currentVoiceCount++;
                    NN_LOG("Voice added. Total=%d \n", currentVoiceCount - 1);
                    if( format == ADPCM )
                    {
                        renderer.AddAdpcmVoice(currentVoiceCount);
                    }
                    else if( format == PCM )
                    {
                        renderer.AddPcmVoice(currentVoiceCount);
                    }
                }
                else
                {
                    reachedMax = true;
                }
            }
            renderer.DecrementTotalDumpCount();
        }
    }

    NN_LOG("Final Voice Count = %d \n", currentVoiceCount - 1);
    int value;
    if( currentVoiceCount == g_VoiceMax )
    {
        value = -1;
    }
    else
    {
        value = currentVoiceCount - 1;
    }
    if( format == ADPCM )
    {
        NN_LOG("##teamcity[buildStatisticValue key='AdpcmVoiceMax(int)' value='%d']\n", value);
    }
    else if( format == PCM )
    {
        NN_LOG("##teamcity[buildStatisticValue key='PcmVoiceMax(int)' value='%d']\n", value);
    }

    renderer.Wait();
    renderer.Update();
    renderer.Stop(0, 0, 0);
    renderer.Update();
    g_Allocator.Free(workBuffer);
    if( format == ADPCM )
    {
        renderer.RendererAdpcmVoiceShutdown();
    }
    else if( format == PCM )
    {
        renderer.RendererPcmVoiceShutdown();
    }
    FinalizeFileSystem();
} // NOLINT(readability/fn_size)

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(AdspLoad, AdpcmVoices)
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    RunAdspPerformanceTest(ADPCM);
}

TEST(AdspLoad, PcmVoices)
{
    RunAdspPerformanceTest(PCM);
}
#endif
