﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Effect.h"

#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>

namespace {
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[1024 * 1024];

void AllocateTestBuffer(void** ppAllocatedAddress, void** ppAlignedAddress, size_t allocSize) NN_NOEXCEPT
{
    auto buf = malloc(allocSize);
    NN_ASSERT_NOT_NULL(buf);
    auto alignedBuf = nn::util::align_up(reinterpret_cast<uintptr_t>(buf), nn::audio::MemoryPoolType::AddressAlignment);
    *ppAllocatedAddress = buf;
    *ppAlignedAddress = reinterpret_cast<void*>(alignedBuf);
}
}
TEST(TestAddRemoveBufferMixer, MultiRenderer)
{
    const size_t workBufferSize = 1024 * 1024 * 2;
    void* workBuffer1 = nullptr;
    void* workBuffer2 = nullptr;
    void* alignedWorkBuffer1 = nullptr;
    void* alignedWorkBuffer2 = nullptr;

    AllocateTestBuffer(&workBuffer1, &alignedWorkBuffer1, workBufferSize);
    AllocateTestBuffer(&workBuffer2, &alignedWorkBuffer2, workBufferSize);
    nn::mem::StandardAllocator allocator1(reinterpret_cast<void*>(alignedWorkBuffer1), workBufferSize);
    nn::mem::StandardAllocator allocator2(reinterpret_cast<void*>(alignedWorkBuffer2), workBufferSize);

    nnt::audio::EffectScopedRenderer renderer1(allocator1);
    nnt::audio::EffectScopedRenderer renderer2(allocator2);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    renderer1.RendererSetup(&param);
    renderer2.RendererSetup(&param);

    nn::audio::BufferMixerType mixer1, mixer2;
    ASSERT_TRUE(nn::audio::AddBufferMixer(renderer1.GetConfig(), &mixer1, renderer1.GetSubMix()).IsSuccess());
    ASSERT_TRUE(nn::audio::AddBufferMixer(renderer2.GetConfig(), &mixer2, renderer2.GetSubMix()).IsSuccess());

    renderer1.Start();
    renderer2.Start();
    renderer1.Update();
    renderer2.Update();
    renderer1.Wait();
    renderer2.Wait();

    EXPECT_EQ(IsBufferMixerRemovable(&mixer1), false);
    EXPECT_EQ(IsBufferMixerRemovable(&mixer2), false);

    // renderer1 のみ Stop させ renderer2 への影響のないことを確認する
    renderer1.Stop();
    nn::audio::SetBufferMixerEnabled(&mixer1, false);

    const int timeoutCount = 100;
    auto count = 0;
    for (count = 0; count < 100; ++count)
    {
        if (IsBufferMixerRemovable(&mixer1) == true)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        renderer1.Update();
    }
    ASSERT_LT(count, timeoutCount);

    EXPECT_EQ(IsBufferMixerRemovable(&mixer1), true);
    EXPECT_EQ(IsBufferMixerRemovable(&mixer2), false);
    nn::audio::RemoveBufferMixer(renderer1.GetConfig(), &mixer1, renderer1.GetSubMix());

    renderer1.RendererShutdown();

    renderer2.Wait();
    renderer2.Update();

    EXPECT_EQ(IsBufferMixerRemovable(&mixer2), false);
    nn::audio::SetBufferMixerEnabled(&mixer2, false);
    for (count = 0; count < 100; ++count)
    {
        if (IsBufferMixerRemovable(&mixer2) == true)
        {
            break;
        }

        renderer2.Wait();
        renderer2.Update();
    }
    ASSERT_LT(count, timeoutCount);

    EXPECT_EQ(IsBufferMixerRemovable(&mixer2), true);
    nn::audio::RemoveBufferMixer(renderer2.GetConfig(), &mixer2, renderer2.GetSubMix());

    renderer2.RendererShutdown();

    allocator1.Finalize();
    allocator2.Finalize();

    free(workBuffer1);
    free(workBuffer2);
} // NOLINT(readability/fn_size)

TEST(TestBufferMixer, UpdateAudioRendererOnEachStep)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const int channelCountMax = 4;

    renderer.RendererSetup(&param, 1, 10);

    nn::audio::BufferMixerType bufferMixer;

    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 10; ++i) // 10 は適当な数。深い意味はない。
    {
        EXPECT_TRUE(nn::audio::AddBufferMixer(renderer.GetConfig(), &bufferMixer, renderer.GetFinalMix()).IsSuccess());

        renderer.Update();
        renderer.Wait();

        const int8_t inputChannels[]  = {0, 1, 2, 3};
        const int8_t outputChannels[] = {3, 2, 1, 0};
        const int channelCountTable[] = {1, 2, 4};
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];
        EXPECT_LE(channelCount, channelCountMax);
        nn::audio::SetBufferMixerInputOutput(&bufferMixer, inputChannels, outputChannels, channelCount);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetBufferMixerVolume(&bufferMixer, 0, 0.5f);
        renderer.Update();
        renderer.Wait();

        // 外すために無効化
        nn::audio::SetBufferMixerEnabled(&bufferMixer, false);

        // 外れるまで待機
        while(IsBufferMixerRemovable(&bufferMixer) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // BufferMixer を外す
        nn::audio::RemoveBufferMixer(renderer.GetConfig(), &bufferMixer, renderer.GetFinalMix());
    }

    renderer.RendererShutdown();
}
