﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_Common.h>

#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include "../../../../../Programs/Eris/Sources/Libraries/audio/dsp/audio_Dsp.h"
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include "../../../../../Programs/Eris/Sources/Libraries/audio/server/audio_AudioHardware.h"
#endif

namespace {
void InitializeAudioHardware(bool hasCodecIc, bool hasHda)
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::audio::InitializeAudioHardware(hasCodecIc, hasHda);
#endif
}

void FinalizeAudioHardware()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::audio::FinalizeAudioHardware();
#endif
}

} // namespace

TEST(StartAndStop, Success)
{
    const auto HasCodecIc = true;
    const auto HasHda = true;
    InitializeAudioHardware(HasCodecIc, HasHda);

    nn::audio::dsp::Start();
    nn::audio::dsp::Wait();

    nn::audio::dsp::Stop();

    FinalizeAudioHardware();
}

TEST(WaitAndSignal, Success)
{
    const auto HasCodecIc = true;
    const auto HasHda = true;
    InitializeAudioHardware(HasCodecIc, HasHda);

    nn::audio::dsp::Start();
    nn::audio::dsp::Wait();

    nn::audio::dsp::Signal();
    nn::audio::dsp::Wait();

    nn::audio::dsp::Stop();

    FinalizeAudioHardware();
}

TEST(MapAndUnmapMemory, Success)
{
    static NN_ALIGNAS(64) uint8_t buffer[1024 * 1024 * 1] = { 0 };
    const auto HasCodecIc = true;
    const auto HasHda = true;
    InitializeAudioHardware(HasCodecIc, HasHda);

    nn::audio::dsp::Start();
    nn::audio::dsp::Wait();

    nn::audio::dsp::DumpMappedPointer();

    auto dspAddr = nn::audio::dsp::MapUserPointer(buffer, sizeof(buffer));
    EXPECT_NE(dspAddr, 0u);
    nn::audio::dsp::DumpMappedPointer();

    nn::audio::dsp::UnmapUserPointer(buffer, sizeof(buffer));
    nn::audio::dsp::DumpMappedPointer();

    nn::audio::dsp::Stop();
    nn::audio::dsp::DumpMappedPointer();

    FinalizeAudioHardware();
}

TEST(SendCommandBuffer, Success)
{
    auto sessionId = 0;
    NN_ALIGNAS(64) uint8_t commandBuffer[128] = { 0 };
    const auto HasCodecIc = true;
    const auto HasHda = true;
    InitializeAudioHardware(HasCodecIc, HasHda);
    nn::audio::dsp::Start();
    nn::audio::dsp::Wait();

    auto commandBufferDspAddr = nn::audio::dsp::MapUserPointer(commandBuffer, sizeof(commandBuffer));
    nn::audio::dsp::SendCommandBuffer(sessionId, commandBufferDspAddr, sizeof(commandBuffer), 100, true, 0);
    nn::audio::dsp::Signal();

    nn::audio::dsp::Wait();
    nn::audio::dsp::Stop();
    FinalizeAudioHardware();
}

TEST(SleepWake, Success)
{
    nn::os::ThreadType thread;
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack[32768];

    const auto HasCodecIc = true;
    const auto HasHda = true;
    InitializeAudioHardware(HasCodecIc, HasHda);
    nn::audio::dsp::Start();
    std::atomic<bool> continueSleepWake(true);

    // psc handler emulation
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread, [](void* args) {
        auto pContinueSleepWake = reinterpret_cast<std::atomic_bool*>(args);
        while(*pContinueSleepWake)
        {
            nn::audio::dsp::Sleep();
            nn::audio::dsp::Wake();
        }
    }, &continueSleepWake, threadStack, sizeof(threadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    // AudioRenderSystemManager emulation
    for(int i = 0;; ++i)
    {
        nn::audio::dsp::Wait();
        // 5 ms (AudioFrame) * 200 = 1 second
        if(i == 200)
        {
            break;
        }
        nn::audio::dsp::Signal();
    }

    continueSleepWake = false;
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    nn::audio::dsp::Stop();
    FinalizeAudioHardware();
}
