﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/audio.h>

namespace {

    const int RenderRate = 32000;
    const int RenderCount = (RenderRate / 200);

    const char Title[] = "AudioVoiceTest";

    const int VoiceCount = 64;

    // 64 sine frequencies representing 64 musical pitches to be played
    const int sineFrequencies[VoiceCount] =
        { 261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349,
        440, 196, 987, 2093, 1976, 3136, 493, 2637,
        261, 329, 523, 659, 1047, 1319, 293, 392,
        587, 784, 1175, 1568, 880, 293, 1760, 2349,
        440, 196, 987, 2093, 1976, 3136, 493, 2637,
        261, 329, 523, 659, 1047, 1319, 293, 392 };

    char g_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

void PrintUsage()
{
    NN_LOG("----------------------------------------------\n");
    NN_LOG("%s Sample\n", Title);
    NN_LOG("64 Sine Voices \n");
    NN_LOG("----------------------------------------------\n");
}

void AddVoice(nn::audio::AudioRendererConfig *pConfig, nn::audio::SubMixType *pSubMix, nn::audio::WaveBuffer *pWaveBufferSine, nn::audio::VoiceType *pVoiceSine, int sineSampleRate) {

    nn::audio::AcquireVoiceSlot(pConfig, pVoiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(pConfig, pVoiceSine, pSubMix);

    // 波形を Voice に追加します。
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::SetVoicePlayState(pVoiceSine, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(pVoiceSine, pSubMix, 1.0f / 64.0f, 0, 0);
}

void ReleaseVoice(nn::audio::AudioRendererConfig *pConfig, nn::audio::VoiceType *pVoiceSine) {
    nn::audio::ReleaseVoiceSlot(pConfig, pVoiceSine);
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6;
    parameter.voiceCount = VoiceCount;
    parameter.subMixCount = 1;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2] = { 0, 1 };

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
        );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    size_t alignment = nn::os::MemoryPageSize;
    NN_LOG("%x\n",workBufferSize);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, alignment);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    nn::Result result = nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 4);

    nn::audio::SubMixType subMix;
    nn::audio::AcquireSubMix(&config, &subMix, parameter.sampleRate, 2);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::SetSubMixDestination(&config, &subMix, &finalMix);
    nn::audio::SetSubMixMixVolume(&subMix, &finalMix, 0.4f, 0, mainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix, &finalMix, 0.4f, 0, mainBus[1]);

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    bool rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Voice を用意します。
    nn::audio::VoiceType voiceSine[VoiceCount];
    nn::audio::WaveBuffer waveBufferSine[VoiceCount];
    void* dataSine[VoiceCount];

    const int sineSampleRate = 32000;
    const int sineSampleCount = 32000;

    for (int i = 0; i < VoiceCount; ++i) {
        std::size_t size = GenerateSineWave(&dataSine[i], sineSampleRate, sineFrequencies[i], sineSampleCount);

        waveBufferSine[i].buffer = dataSine[i];
        waveBufferSine[i].size = size;
        waveBufferSine[i].startSampleOffset = 0;
        waveBufferSine[i].endSampleOffset = sineSampleCount;
        waveBufferSine[i].loop = 0;
        waveBufferSine[i].isEndOfStream = false;

        AddVoice(&config, &subMix, &waveBufferSine[i], &voiceSine[i], sineSampleRate);
    }

    PrintUsage();

    int voiceCount = 64;
    // appendCount is used as a threshold for number of voices to be played
    int appendCount = 128 * 70;
    bool running = true;
    while (running)
    {
        systemEvent.Wait();
        // lower threshold (and consequently, voice count)
        appendCount--;

        for (int i = 0; i < voiceCount; ++i) {
            if (!nn::audio::IsVoiceValid(&voiceSine[i]))
            {
                continue;
            }
            if (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voiceSine[i]))
            {
                if (appendCount > i * 128)
                {
                    // If voice number is under threshold, continue to append wave buffers
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                    nn::audio::AppendWaveBuffer(&voiceSine[i], pWaveBuffer);
                }
                else if (appendCount < 0) {
                    running = false;
                }
                else {
                    ReleaseVoice(&config, &voiceSine[i]);
                    voiceCount--;
                    NN_LOG("%i Sine Voices \n", voiceCount);
                }
            }
        }
        for (int i = 0; i < voiceCount; ++i)
        {
            if (voiceCount > 0 && nn::audio::IsVoiceValid(&voiceSine[i])) {
                if (nn::audio::GetVoiceMixVolume(&voiceSine[i], &subMix, 0, 0) != (1.0f / voiceCount)) {
                    nn::audio::SetVoiceMixVolume(&voiceSine[i], &subMix, (float) 1.0f / voiceCount, 0, 0);
                }
            }
        }
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < VoiceCount; ++i) {
        if (dataSine[i])
        {
            g_WaveBufferAllocator.Free(dataSine[i]);
            dataSine[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
}  // NOLINT(readability/fn_size)
