﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../../Programs/Eris/Sources/Libraries/audio/audio_MixManager.h"

#include "testAudio_SimpleAudioRenderer.h"

namespace {

void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 16;
    pOutParameter->subMixCount = 8;
    pOutParameter->voiceCount = 24;
}

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];  // 4096 == nn::os::MemoryPageSize

}

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter)
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }
    ~ScopedAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
        m_Allocator.Finalize();
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

TEST(AcquireSubMix, SingleAcquireAndRelease)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);

    nn::audio::SubMixType subMix;
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
    nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix);

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 32000, 1));
    nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix);
}

TEST(AcquireSubMix, MultiAcquire)
{
    const int subMixCount = 3;
    nn::audio::SubMixType subMix[subMixCount];
    nn::audio::FinalMixType finalMix;
    nn::audio::AudioRendererParameter parameter;

    SetDefaultParameter(&parameter);

    parameter.subMixCount = subMixCount - 1;
    // bufferCount <= MixBufferCountMax が呼び出し事前条件なので、
    // それより少なくないと、そちらに引っかかって正しくテストできない
    parameter.mixBufferCount = nn::audio::MixBufferCountMax - 1;
    ScopedAudioRenderer sar(parameter);

    // AudioRendererParameter.subMixCount についてのテスト
    // SampleRate = 48000 hz
    for(int i = 0; i < parameter.subMixCount; ++i)
    {
        EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[i], 48000, 1));
    }

    EXPECT_FALSE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[subMixCount - 1], 48000, 1));

    for(int i = 0; i < parameter.subMixCount; ++i)
    {
       nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix[i]);
    }

    // SampleRate 32000 Hz
    for(int i = 0; i < parameter.subMixCount; ++i)
    {
        EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[i], 32000, 1));
    }

    EXPECT_FALSE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[subMixCount - 1], 32000, 1));

    for(int i = 0; i < parameter.subMixCount; ++i)
    {
        nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix[i]);
    }

    // AudioRendererParameter.mixBufferCount についてのテスト
    EXPECT_FALSE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, parameter.mixBufferCount + 1));

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, parameter.mixBufferCount));
    EXPECT_FALSE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix[0]);

    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount - 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_FALSE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AcquireSubMix, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(nullptr, &subMix, 48000, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), nullptr, 48000, 1), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000 - 1, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000 + 1, 1), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 32000 - 1, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 32000 + 1, 1), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, nn::audio::MixBufferCountMax + 1), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ReleaseSubMix, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(nullptr, &subMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(&sar.GetConfig(), nullptr), "");

    // subMix._pMixInfo = reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad);
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(&sar.GetConfig(), &subMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(ReleaseSubMix, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetSubMixDestination, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix[2];
    // nn::audio::SubMixType invalidSubMix;
    nn::audio::FinalMixType finalMix;
    // nn::audio::FinalMixType invalidFinalMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    // invalidSubMix._pMixInfo = reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad);
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
    // invalidFinalMix._pMixInfo = reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad);

    // Destination が SubMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(nullptr, &subMix[0], &subMix[1]), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), nullptr, &subMix[1]), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], static_cast<nn::audio::SubMixType*>(nullptr)), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &invalidSubMix, &subMix[1]), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &invalidSubMix), "");

    // Destination が FinalMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(nullptr, &subMix[0], &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), nullptr, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], static_cast<nn::audio::FinalMixType*>(nullptr)), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &invalidSubMix, &finalMix), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetSubMixDestination, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix[2];
    nn::audio::FinalMixType finalMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &subMix[1]);
    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &finalMix);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetSubMixSampleRate, PreCondition)
{
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixSampleRate(nullptr), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixSampleRate(&invalidSubMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetSubMixSampleRate, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));
    EXPECT_EQ(nn::audio::GetSubMixSampleRate(&subMix), 48000);

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 32000, 1));
    EXPECT_EQ(nn::audio::GetSubMixSampleRate(&subMix), 32000);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetSubMixBufferCount, PreCondition)
{
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixBufferCount(nullptr), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixBufferCount(&invalidSubMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetSubMixBufferCount, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 2));
    EXPECT_EQ(nn::audio::GetSubMixBufferCount(&subMix), 2);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetSubMixVolume, PreCondition)
{
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixVolume(nullptr), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixVolume(&invalidSubMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetSubMixVolume, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixVolume(nullptr, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixVolume(&invalidSubMix, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixVolume(&subMix, nn::audio::SubMixType::GetVolumeMin() - 1e-3f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixVolume(&subMix, nn::audio::SubMixType::GetVolumeMax() + 1e-3f), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetSubMixVolume, OutputWaveformCheck)
{
    const std::size_t size = 128 * 1024 * 1024;
    void* buffer = std::malloc(size);
    ASSERT_TRUE(buffer != nullptr);

    {
        SimpleAudioRenderer renderer(buffer, size);

        // preparation: prepare constant value short waveform
        const auto InputValue = std::numeric_limits<int16_t>::max();
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t input[nn::audio::MemoryPoolType::SizeGranularity / sizeof(int16_t)];
        for (auto i = 0; i < static_cast<int>(sizeof(input) / sizeof(input[0])); ++i)
        {
            input[i] = InputValue;
        }
        nn::audio::MemoryPoolType dataMemoryPool;
        EXPECT_TRUE(nn::audio::AcquireMemoryPool(&renderer.GetConfig(), &dataMemoryPool, input, nn::util::align_up(sizeof(input), nn::audio::MemoryPoolType::SizeGranularity)));
        EXPECT_TRUE(nn::audio::RequestAttachMemoryPool(&dataMemoryPool));

        // testcase: Setting SubMixVolume to 0.0f
        // expected: Final output is zero
        {
            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetSubMixVolume(0.0f);

            const auto LoopCount = 64;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    EXPECT_TRUE(data[i] == 0);
                }
            }
        }

        // testcase: Setting SubMixVolume to 0.5f
        // expected: max of Final output is in (InputValue - Delta, InputValue + Delta)
        {
//             const auto Delta = 32;

            renderer.AppendPcm16Waveform(input, sizeof(input));
            renderer.SetSubMixVolume(0.5f);

            int16_t max = 0;
            const auto LoopCount = 128;
            for (auto loop = 0; loop < LoopCount; ++loop)
            {
                renderer.Wait();
                renderer.Update();

                int16_t data[240 * 2];
                auto dataSize = renderer.ReadFinalMixOut(data, sizeof(data));
                for (int i = 0; i < static_cast<int>(dataSize / sizeof(int16_t)); ++i)
                {
                    max = std::max(max, data[i]);
                }
            }
//             EXPECT_TRUE(max >= std::numeric_limits<int16_t>::max() / 2 - Delta);
//             EXPECT_TRUE(max <= std::numeric_limits<int16_t>::max() / 2 + Delta);
            NN_LOG("max = %d\n", max);
        }
    }

    std::free(buffer);
}

TEST(GetAndSetSubMixVolume, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));

    float volumes[] = { 1.5f, 0.0f, nn::audio::SubMixType::GetVolumeMax(), nn::audio::SubMixType::GetVolumeMin() };
    for (auto volume : volumes)
    {
        nn::audio::SetSubMixVolume(&subMix, volume);
        EXPECT_TRUE(nn::audio::GetSubMixVolume(&subMix) == volume);
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetSubMixMixVolume, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix[2];
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    nn::audio::FinalMixType finalMix;
    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };



    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &subMix[1]);
    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[1], &finalMix);

    // Destination が SubMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(nullptr, &subMix[1], 0.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], static_cast<nn::audio::SubMixType*>(nullptr), 0.0f, 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&invalidSubMix, &subMix[1], 0.0f, 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &invalidSubMix, 0.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &subMix[1], nn::audio::SubMixType::GetVolumeMin() - 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &subMix[1], nn::audio::SubMixType::GetVolumeMax() + 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &subMix[1], 0.0f, nn::audio::GetSubMixBufferCount(&subMix[0]), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &subMix[1], 0.0f, 0, nn::audio::GetSubMixBufferCount(&subMix[1])), "");

    // Destination が FinalMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(nullptr, &finalMix, 0.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], static_cast<nn::audio::FinalMixType*>(nullptr), 0.0f, 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &invalidFinalMix, 0.0f, 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&invalidSubMix, &finalMix, 0.0f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &finalMix, nn::audio::SubMixType::GetVolumeMin() - 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &finalMix, nn::audio::SubMixType::GetVolumeMax() + 1e-3f, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &finalMix, 0.0f, nn::audio::GetSubMixBufferCount(&subMix[0]), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixMixVolume(&subMix[0], &finalMix, 0.0f, 0, nn::audio::GetFinalMixBufferCount(&finalMix)), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetSubMixMixVolume, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix[2];
    // nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };
    nn::audio::FinalMixType finalMix;
    // nn::audio::FinalMixType invalidFinalMix = { reinterpret_cast<nn::audio::MixInfo*>(0xdeadbaad) };

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &subMix[1]);
    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[1], &finalMix);

    // Destination が SubMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(nullptr, &subMix[0], 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], static_cast<nn::audio::SubMixType*>(nullptr), 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&invalidSubMix, &subMix[1], 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &invalidSubMix, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &subMix[1], nn::audio::GetSubMixBufferCount(&subMix[0]), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &subMix[1], 0, nn::audio::GetSubMixBufferCount(&subMix[1])), "");

    // Destination が FinalMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(nullptr, &finalMix, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], static_cast<nn::audio::FinalMixType*>(nullptr), 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&invalidSubMix, &finalMix, 0, 0), "");
    // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &invalidFinalMix, 0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &finalMix, nn::audio::GetSubMixBufferCount(&subMix[0]), 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetSubMixMixVolume(&subMix[0], &finalMix, 0, nn::audio::GetFinalMixBufferCount(&finalMix)), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetAndSetSubMixMixVolume, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);
    nn::audio::SubMixType subMix[2];
    nn::audio::FinalMixType finalMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 2));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 2));

    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &subMix[1]);
    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[1], &finalMix);

    nn::audio::SetSubMixMixVolume(&subMix[0], &subMix[1], 1.0f, 0, 1);
    nn::audio::SetSubMixMixVolume(&subMix[1], &finalMix, 2.0f, 0, 1);

    EXPECT_EQ(nn::audio::GetSubMixMixVolume(&subMix[0], &subMix[1], 0, 1), 1.0f);
    EXPECT_EQ(nn::audio::GetSubMixMixVolume(&subMix[1], &finalMix, 0, 1), 2.0f);
}
