﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>  // std::memcmp
#include <cstdlib>  // std::aligned_alloc
#include <memory>   // std::unique_ptr
#include <algorithm> // std::abs

#include <nnt.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>

namespace {

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[1024 * 1024];

const std::string g_AudioTestSourceFileName(g_SourceDataFolder + "/shutter.wav");

}

TEST(AddCircularBufferSink, success)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));

    nn::audio::CircularBufferSinkType circularBufferSink;
    int8_t inputs[6] = {0, 1, 2, 3, 4, 5};
    const size_t circularBufferSize = 6 * 240 * sizeof(int16_t);
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    auto buffer = allocator.Allocate(circularBufferSize, 4096);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ));
    EXPECT_NE(static_cast<int32_t>(nn::audio::GetSinkNodeId(&circularBufferSink)), 0);
    sar.Start();

    if (buffer)
    {
        allocator.Free(buffer);
        buffer = nullptr;
    }
}

TEST(AddCircularBufferSink, RegressionCheckForSIGLO_82558_BufferSize0)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nn::os::SystemEvent systemEvent;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::FinalMixType finalMix;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));

    nn::audio::CircularBufferSinkType circularBufferSink;
    int8_t inputs[6] = {0, 1, 2, 3, 4, 5};
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::MemoryPoolType memoryPool;
    nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&memoryPool);

    // 1 バイトだけ確保しておく
    auto buffer = allocator.Allocate(1, 4096);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    const size_t circularBufferSize = 0u;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ));
    EXPECT_NE(static_cast<int32_t>(nn::audio::GetSinkNodeId(&circularBufferSink)), 0);
    sar.Start();

    // NXAddon 6.0.0 時点での実績値として 500 回を設定 (100 回程度だと再現しない)
    for(int i = 0; i < 500; ++i)
    {
        sar.Update();
        systemEvent.Wait();
    }

    if (buffer)
    {
        allocator.Free(buffer);
        buffer = nullptr;
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AddCircularBufferSink, precondition)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);

    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::CircularBufferSinkType circularBufferSink;
    int8_t inputs[inputCount] = {0, 1, 2, 3, 4, 5};
    const size_t circularBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&parameter, inputCount, 1, nn::audio::SampleFormat_PcmInt16);

    nn::mem::StandardAllocator allocator;
    allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    auto buffer = allocator.Allocate(circularBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(nullptr, &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), nullptr, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, nullptr, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, nullptr, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), nullptr, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, -1, buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs) + 1, buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize - 1, nn::audio::SampleFormat_PcmInt16 ), "");

    sar.Start();

    if (buffer)
    {
        allocator.Free(buffer);
        buffer = nullptr;
    }
}

TEST(AddDeviceSink, precondition)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::DeviceSinkType deviceSink;
    int8_t inputs[inputCount] = {0, 1, 2, 3, 4, 5};

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDeviceSink(nullptr, &deviceSink, &finalMix, inputs, sizeof(inputs), "MainAudioOut"), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDeviceSink(&sar.GetConfig(), nullptr, &finalMix, inputs, sizeof(inputs), "MainAudioOut"), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, nullptr, inputs, sizeof(inputs), "MainAudioOut"), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, nullptr, sizeof(inputs), "MainAudioOut"), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputs, sizeof(inputs) + 1, "MainAudioOut"), "");
}

#endif

TEST(SetGetDownMixMatrix, Success)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::DeviceSinkType deviceSink;
    int8_t inputs[inputCount] = { 0, 1, 2, 3, 4, 5 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputs, inputCount, "MainAudioOut"));

    nn::audio::DeviceSinkType::DownMixParameter downMixParameter;
    memset(&downMixParameter, 0, sizeof(nn::audio::DeviceSinkType::DownMixParameter));
    downMixParameter.coeff[0] = 0.1f;
    downMixParameter.coeff[1] = 0.2f;
    downMixParameter.coeff[2] = 0.3f;
    downMixParameter.coeff[3] = 0.4f;
    nn::audio::SetDownMixParameter(&deviceSink, &downMixParameter);

    nn::audio::DeviceSinkType::DownMixParameter checkDownMixParameter;
    nn::audio::GetDownMixParameter(&checkDownMixParameter, &deviceSink);

    ASSERT_TRUE(memcmp(&downMixParameter, &checkDownMixParameter, sizeof(nn::audio::DeviceSinkType::DownMixParameter)) == 0);
}

TEST(SetDownMixMatrixEnabled, Success)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::DeviceSinkType deviceSink;
    int8_t inputs[inputCount] = { 0, 1, 2, 3, 4, 5 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputs, inputCount, "MainAudioOut"));

    nn::audio::DeviceSinkType::DownMixParameter checkDownMixParameter;
    nn::audio::GetDownMixParameter(&checkDownMixParameter, &deviceSink);
    for (auto i = 0; i < nn::audio::DeviceSinkType::DownMixParameter::CoeffCount; ++i)
    {
        EXPECT_EQ(checkDownMixParameter.coeff[i], 0.0f);
    }

    EXPECT_EQ(nn::audio::IsDownMixParameterEnabled(&deviceSink), false);

    nn::audio::SetDownMixParameterEnabled(&deviceSink, true);
    EXPECT_EQ(nn::audio::IsDownMixParameterEnabled(&deviceSink), true);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetGetDownMixParameter, precondition)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::DeviceSinkType deviceSink{ nullptr };
    int8_t inputs[inputCount] = { 0, 1, 2, 3, 4, 5 };

    nn::audio::DeviceSinkType::DownMixParameter downMixParameter;
    nn::audio::DeviceSinkType::DownMixParameter checkDownMixParameter;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDownMixParameter(&deviceSink, &downMixParameter), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDownMixParameter(&checkDownMixParameter, &deviceSink), "");

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputs, inputCount, "MainAudioOut"));

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDownMixParameter(nullptr, &downMixParameter), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDownMixParameter(&deviceSink, nullptr), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDownMixParameter(nullptr, &deviceSink), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetDownMixParameter(&checkDownMixParameter, nullptr), "");
}

TEST(SetDownMixMatrixEnabled, Precondition)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    const int inputCount = 6;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, inputCount));

    nn::audio::DeviceSinkType deviceSink{ nullptr };
    int8_t inputs[inputCount] = { 0, 1, 2, 3, 4, 5 };

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDownMixParameterEnabled(&deviceSink, true), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsDownMixParameterEnabled(&deviceSink), "");

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputs, inputCount, "MainAudioOut"));

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetDownMixParameterEnabled(nullptr, true), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::IsDownMixParameterEnabled(nullptr), "");
}
#endif

TEST(AddCircularBufferSink, RunSuccess)
{
    // Test conditions.
    const int channelCount = 6;
    int8_t inputs[channelCount] = {0, 1, 2, 3, 4, 5};

    // inputValue の値は テスト設計時に適当に選んだ直流成分値。深い意味はない。
    // volumes[] による操作を経て、結果が answers と等しくなるかを確認する
    const int16_t inputValue = 20;
    const int16_t srcError = 1; // inputValue の値によって決まる Resample に伴う入力信号のエラー量
    float volumes[channelCount] = { 1.0f, 0.8f, 1.5f, 0.4f, nn::audio::VoiceType::GetVolumeMax(), nn::audio::VoiceType::GetVolumeMin()};
    const int16_t answers[channelCount] = {
        static_cast<int16_t>((inputValue - srcError) * volumes[0]),
        static_cast<int16_t>((inputValue - srcError) * volumes[1]),
        static_cast<int16_t>((inputValue - srcError) * volumes[2]),
        static_cast<int16_t>((inputValue - srcError) * volumes[3]),
        static_cast<int16_t>((inputValue - srcError) * volumes[4]),
        static_cast<int16_t>((inputValue - srcError) * volumes[5]),
    };

    // test condition
    const int tolerance = 1; // 許容誤差。SRC による誤差分は正解とみなす。
    const int checkOffset = 3; // SRC による誤差が大きな最初の数は破棄する。何サンプルを読み飛ばすかの設定
    const int parameterCount = 2;

    nn::audio::AudioRendererParameter inputParameter[parameterCount] =
    {
        nnt::audio::util::GetAudioRendererParameterForDefault(),
        nnt::audio::util::GetAudioRendererParameterForDefault(),
    };
    inputParameter[1].sampleRate = 32000;
    inputParameter[1].sampleCount = 160;
    inputParameter[1].subMixCount = 10;
    inputParameter[1].mixBufferCount = 256;

    const int frameCount[parameterCount] = { 10, 5 };

    nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16;
    int sampleCount = 0;
    nn::Result result;

    for (auto paramIndex = 0; paramIndex < parameterCount; ++paramIndex)
    {
        // setup
        nn::audio::AudioRendererParameter parameter(inputParameter[paramIndex]);
        sampleCount = parameter.sampleCount;

        nn::os::SystemEvent event;
        nnt::audio::util::ScopedAudioRenderer sar(parameter, &event);
        nn::audio::MemoryPoolType memoryPool;
        nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&memoryPool);
        const int finalMixBusCount = 6;
        nn::audio::FinalMixType finalMix;
        ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, finalMixBusCount));

        // == Regression check for SIGLO-72261  ===========
        nn::audio::SubMixType subMixes[10];
        if (parameter.mixBufferCount - finalMixBusCount > 0)
        {
            for (auto& subMix : subMixes)
            {
                nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, parameter.sampleRate, 24);
                nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix, &finalMix);
            }
        }
        // == end =========================================

        nn::audio::CircularBufferSinkType circularBufferSink;
        const size_t circularBufferSize = sampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat) * frameCount[paramIndex];
        ASSERT_EQ(circularBufferSize, nn::audio::GetRequiredBufferSizeForCircularBufferSink(&parameter, channelCount, frameCount[paramIndex], sampleFormat));
        nn::mem::StandardAllocator allocator;
        allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
        auto buffer = allocator.Allocate(circularBufferSize, 4096);
        NN_ABORT_UNLESS_NOT_NULL(buffer);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ));

        sar.Start();

        // create test source voice
        nn::audio::VoiceType testVoice;
        nn::audio::WaveBuffer waveBuffer;

        size_t srcSize = sampleCount * frameCount[paramIndex] * sizeof(int16_t);
        int16_t* sourceBuffer = static_cast<int16_t*>(allocator.Allocate(srcSize, 4096));
        NN_ABORT_UNLESS_NOT_NULL(sourceBuffer);
        size_t resultDataSize = sampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
        int16_t* resultData = static_cast<int16_t*>(allocator.Allocate(resultDataSize, 4096));
        NN_ABORT_UNLESS_NOT_NULL(resultData);

        // fill buffer with normalized data
        for (int i = 0; i < static_cast<int>(srcSize / sizeof(int16_t)); ++i)
        {
            sourceBuffer[i] = inputValue;
        }

        nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &testVoice, parameter.sampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &testVoice, &finalMix);

        waveBuffer.buffer = sourceBuffer;
        waveBuffer.size = srcSize;
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(srcSize / sizeof(int16_t));
        waveBuffer.loop = 1;
        waveBuffer.isEndOfStream = false;

        for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
        {
            nn::audio::AppendWaveBuffer(&testVoice, &waveBuffer);
        }
        for (auto i = 0; i < channelCount; ++i)
        {
            nn::audio::SetVoiceMixVolume(&testVoice, &finalMix, volumes[i], 0, i);
        }
        nn::audio::SetVoiceVolume(&testVoice, 1.0f);
        nn::audio::SetVoicePlayState(&testVoice, nn::audio::VoiceType::PlayState_Play);


        for(auto checkCount = 0; checkCount < frameCount[paramIndex] * 2; ++checkCount)
        {
            event.Wait();
            sar.Update();

            memset(resultData, 0, resultDataSize);
            size_t readSize = nn::audio::ReadCircularBufferSink(&circularBufferSink, resultData, resultDataSize);
            if (readSize <= 0)
            {
                continue;
            }

            // Read Size check.
            // CircularBufferSink の読み込み結果は、ブロックインターリーブされ、Update 時にそのブロック単位で
            // WriteOffset が increment される。そのため、 resultData に十分なサイズのバッファを渡せば、
            // readSize は予測可能なブロックサイズとなる。
            ASSERT_TRUE(readSize % (channelCount * nn::audio::GetSampleByteSize(sampleFormat) * sampleCount) == 0)
                << "Read size is invalid,"
                << "readSize: " << readSize
                << "expectedSize: readSize%" << (channelCount * nn::audio::GetSampleByteSize(sampleFormat) * sampleCount) << " == 0";

            // Data verification.
            for (auto i = 0; i < channelCount; ++i )
            {
                for (auto j = checkOffset; j < sampleCount; ++j)
                {
                    if (std::abs(answers[i] - resultData[sampleCount * i + j]) > tolerance)
                    {

                        auto targetPointer = &resultData[sampleCount * i];
                        auto dumpSize = sampleCount * sizeof(int16_t);

                        static const int countPerLine = 16;
                        static const int blockSize = 128;
                        static const int lineCount = ((dumpSize + blockSize) & ~(blockSize - 1)) / countPerLine;

                        uint8_t* base = reinterpret_cast<uint8_t*>((reinterpret_cast<uintptr_t>(targetPointer) & ~(blockSize - 1)));
                        NN_LOG("=== CircularBufferSink result check | exceeds tolerance\n");
                        for (auto k = 0; k < lineCount; ++k)
                        {
                            char* baseC = reinterpret_cast<char*>(base);

                            NN_LOG("%p | %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x | %c%c%c%c %c%c%c%c %c%c%c%c %c%c%c%c\n",
                                   base,
                                   base[ 0], base[ 1], base[ 2], base[ 3],  base[ 4], base[ 5], base[ 6], base[ 7],
                                   base[ 8], base[ 9], base[10], base[11],  base[12], base[13], base[14], base[15],
                                   baseC[ 0] < 31 ? '.' : baseC[ 0] >= 127 ? '.' : baseC[ 0],
                                   baseC[ 1] < 31 ? '.' : baseC[ 1] >= 127 ? '.' : baseC[ 1],
                                   baseC[ 2] < 31 ? '.' : baseC[ 2] >= 127 ? '.' : baseC[ 2],
                                   baseC[ 3] < 31 ? '.' : baseC[ 3] >= 127 ? '.' : baseC[ 3],
                                   baseC[ 4] < 31 ? '.' : baseC[ 4] >= 127 ? '.' : baseC[ 4],
                                   baseC[ 5] < 31 ? '.' : baseC[ 5] >= 127 ? '.' : baseC[ 5],
                                   baseC[ 6] < 31 ? '.' : baseC[ 6] >= 127 ? '.' : baseC[ 6],
                                   baseC[ 7] < 31 ? '.' : baseC[ 7] >= 127 ? '.' : baseC[ 7],
                                   baseC[ 8] < 31 ? '.' : baseC[ 8] >= 127 ? '.' : baseC[ 8],
                                   baseC[ 9] < 31 ? '.' : baseC[ 9] >= 127 ? '.' : baseC[ 9],
                                   baseC[10] < 31 ? '.' : baseC[10] >= 127 ? '.' : baseC[10],
                                   baseC[11] < 31 ? '.' : baseC[11] >= 127 ? '.' : baseC[11],
                                   baseC[12] < 31 ? '.' : baseC[12] >= 127 ? '.' : baseC[12],
                                   baseC[13] < 31 ? '.' : baseC[13] >= 127 ? '.' : baseC[13],
                                   baseC[14] < 31 ? '.' : baseC[14] >= 127 ? '.' : baseC[14],
                                   baseC[15] < 31 ? '.' : baseC[15] >= 127 ? '.' : baseC[15]
                                   );

                            base += countPerLine;
                        }

                    }

                    ASSERT_NEAR(answers[i], resultData[sampleCount * i + j], tolerance)
                        << "read data broken at Index:" << (sampleCount * i + j)
                        << " parameterIndex: " << paramIndex;

                }
            }
        }

        nn::audio::SetVoicePlayState(&testVoice, nn::audio::VoiceType::PlayState_Stop);
        while (nn::audio::GetReleasedWaveBuffer(&testVoice) == nullptr)
        {
            event.Wait();
            sar.Update();
        }

        if (sourceBuffer)
        {
            allocator.Free(sourceBuffer);
            sourceBuffer = nullptr;
        }
        if (resultData)
        {
            allocator.Free(resultData);
            resultData = nullptr;
        }

    }
} // NOLINT(readability/fn_size)

TEST(DeviceAndCircularSink, Success)
{
    // Test conditions.
    const int frameCount = 10;
    const int channelCount = 6;
    int8_t inputs[channelCount] = {0, 1, 2, 3, 4, 5};
    int8_t inputsForDeviceSink[] = {0, 1};


    int sampleCount = 0;
    nn::Result result;

    nn::audio::AudioRendererParameter inputParameter[2] =
    {
        nnt::audio::util::GetAudioRendererParameterForDefault(),
        nnt::audio::util::GetAudioRendererParameterForDefault(),
    };
    inputParameter[0].sinkCount = 2;
    inputParameter[1].sinkCount = 2;
    inputParameter[1].sampleRate = 32000;
    inputParameter[1].sampleCount = 160;

    // load source data

    std::string mountPath;
    nnt::audio::util::GetMountPath(mountPath);
    nnt::audio::util::SimpleFsUtility fsUtil;
    NNT_EXPECT_RESULT_SUCCESS(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()));

    nn::mem::StandardAllocator allocator;
    allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(g_AudioTestSourceFileName);
    auto dataBuffer = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(dataBuffer);

    int sampleRate = 0;
    size_t dataSize = nnt::audio::util::LoadSourceSamples(g_AudioTestSourceFileName, dataBuffer, dataShutterSize, &sampleRate);

    for (auto paramIndex = 0; paramIndex < 2; ++paramIndex)
    {
        // setup
        nn::audio::AudioRendererParameter parameter(inputParameter[paramIndex]);
        sampleCount = parameter.sampleCount;

        nn::os::SystemEvent event;
        nnt::audio::util::ScopedAudioRenderer sar(parameter, &event);
        nn::audio::MemoryPoolType memoryPool;
        nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&memoryPool);

        nn::audio::FinalMixType finalMix;
        ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, parameter.mixBufferCount));

        // Add circularBufferSink
        nn::audio::CircularBufferSinkType circularBufferSink;
        const size_t circularBufferSize = sampleCount * channelCount * sizeof(int16_t) * frameCount;
        auto buffer = allocator.Allocate(circularBufferSize, 4096);
        NN_ABORT_UNLESS_NOT_NULL(buffer);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&sar.GetConfig(), &circularBufferSink, &finalMix, inputs, sizeof(inputs), buffer, circularBufferSize, nn::audio::SampleFormat_PcmInt16 ));

        // Add DeviceSink
        nn::audio::DeviceSinkType deviceSink;
        nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, inputsForDeviceSink, sizeof(inputsForDeviceSink), "MainAudioOut");
        EXPECT_NE(static_cast<int32_t>(nn::audio::GetSinkNodeId(&deviceSink)), 0);

        sar.Start();

        // create test source voice
        nn::audio::VoiceType testVoice;
        nn::audio::WaveBuffer waveBuffer;

        size_t resultDataSize = sampleCount * frameCount * channelCount * sizeof(int16_t);
        int16_t* resultData = static_cast<int16_t*>(allocator.Allocate(resultDataSize, 4096));
        NN_ABORT_UNLESS_NOT_NULL(resultData);

        nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &testVoice, sampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &testVoice, &finalMix);

        waveBuffer.buffer = dataBuffer;
        waveBuffer.size = dataSize;
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBuffer.loop = false;
        waveBuffer.isEndOfStream = false;
        waveBuffer.pContext = nullptr;
        waveBuffer.contextSize = 0;

        for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
        {
            nn::audio::AppendWaveBuffer(&testVoice, &waveBuffer);
        }
        for (auto i = 0; i < channelCount; ++i)
        {
            nn::audio::SetVoiceMixVolume(&testVoice, &finalMix, .5f, 0, i);
        }
        nn::audio::SetVoiceVolume(&testVoice, 1.0f);
        nn::audio::SetVoicePlayState(&testVoice, nn::audio::VoiceType::PlayState_Play);

        for(;;)
        {
            if (auto wb = nn::audio::GetReleasedWaveBuffer(&testVoice))
            {
                ASSERT_EQ(wb, &waveBuffer);
                break;
            }
            event.Wait();
            sar.Update();
        }

        if (resultData)
        {
            allocator.Free(resultData);
            resultData = nullptr;
        }
        if (buffer)
        {
            allocator.Free(buffer);
            buffer = nullptr;
        }
    }
    if (dataBuffer)
    {
        allocator.Free(dataBuffer);
        dataBuffer = nullptr;
    }
}

