﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <memory>
#include <vector>
#include <cmath>
#include <cstdlib>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>

namespace {

// Create invalid AudioRendererParmeters
std::vector<nn::audio::AudioRendererParameter> CreateInvalidAudioRendererParameters()
{
    nn::audio::AudioRendererParameter parameter;
    std::vector<nn::audio::AudioRendererParameter> parameters;

    // SampleRate
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = -1;
    parameters.push_back(parameter);
    parameter.sampleRate = 0;
    parameters.push_back(parameter);
    parameter.sampleRate = 1;
    parameters.push_back(parameter);
    parameter.sampleRate = 16000;
    parameters.push_back(parameter);
    parameter.sampleRate = 32000 + 1;
    parameters.push_back(parameter);
    parameter.sampleRate = 32000 - 1;
    parameters.push_back(parameter);
    parameter.sampleRate = 44100;
    parameters.push_back(parameter);
    parameter.sampleRate = 48000 + 1;
    parameters.push_back(parameter);
    parameter.sampleRate = 48000 - 1;
    parameters.push_back(parameter);

    // SampleCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleCount = -1;
    parameters.push_back(parameter);
    parameter.sampleCount = 0;
    parameters.push_back(parameter);
    parameter.sampleCount = 1;
    parameters.push_back(parameter);
    parameter.sampleCount = 32000 / 200 - 1;
    parameters.push_back(parameter);
    parameter.sampleCount = 32000 / 200 + 1;
    parameters.push_back(parameter);
    parameter.sampleCount = 48000 / 200 - 1;
    parameters.push_back(parameter);
    parameter.sampleCount = 48000 / 200 + 1;
    parameters.push_back(parameter);
    parameter.sampleCount = 96000 / 200;
    parameters.push_back(parameter);

    // MixBufferCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 0;
    parameters.push_back(parameter);
    parameter.mixBufferCount = nn::audio::AudioRendererParameter::MixBufferCountMax + 1;
    parameters.push_back(parameter);

    // VoiceCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.voiceCount = 0;
    parameters.push_back(parameter);
    parameter.voiceCount = nn::audio::AudioRendererParameter::VoiceCountMax + 1;
    parameters.push_back(parameter);

    // SinkCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sinkCount = 0;
    parameters.push_back(parameter);
    parameter.sinkCount = nn::audio::AudioRendererParameter::SinkCountMax + 1;
    parameters.push_back(parameter);

    // EffectCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.effectCount = -1;
    parameters.push_back(parameter);
    parameter.effectCount = nn::audio::AudioRendererParameter::EffectCountMax + 1;
    parameters.push_back(parameter);

    // SubMixCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.subMixCount = -1;
    parameters.push_back(parameter);
    parameter.subMixCount = nn::audio::AudioRendererParameter::SubMixCountMax + 1;
    parameters.push_back(parameter);

    // RenderingDevice and ExecutionMode
    const nn::audio::AudioRendererRenderingDevice RenderingDevices[] = {
        nn::audio::AudioRendererRenderingDevice_AudioCoprocessor,
        nn::audio::AudioRendererRenderingDevice_Cpu,
    };

    const nn::audio::AudioRendererExecutionMode ExecutionModes[] = {
        nn::audio::AudioRendererExecutionMode_AutoExecution,
        nn::audio::AudioRendererExecutionMode_ManualExecution,
    };

    for (const auto RenderingDevice : RenderingDevices)
    {
        for (const auto ExecutionMode : ExecutionModes)
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            const auto SupportedMethod = (RenderingDevice == nn::audio::AudioRendererRenderingDevice_Cpu);
#else
            const auto SupportedMethod = (RenderingDevice == nn::audio::AudioRendererRenderingDevice_Cpu && ExecutionMode == nn::audio::AudioRendererExecutionMode_ManualExecution)
                                    || (RenderingDevice == nn::audio::AudioRendererRenderingDevice_AudioCoprocessor && ExecutionMode == nn::audio::AudioRendererExecutionMode_AutoExecution);
#endif
            if(!SupportedMethod)
            {
                nn::audio::InitializeAudioRendererParameter(&parameter);
                parameter.renderingDevice = RenderingDevice;
                parameter.executionMode = ExecutionMode;
                parameters.push_back(parameter);
            }
        }
    }

    return parameters;
}

// Create valid AudioRendererParmeters
std::vector<nn::audio::AudioRendererParameter> CreateValidAudioRendererParameters(bool isLargeMemoryRequiredParamEnabled)
{
    nn::audio::AudioRendererParameter parameter;
    std::vector<nn::audio::AudioRendererParameter> parameters;

    // SampleRate
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 32000;
    parameter.sampleCount = parameter.sampleRate / 200;
    parameters.push_back(parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = parameter.sampleRate / 200;
    parameters.push_back(parameter);

    // MixBufferCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 1;
    parameters.push_back(parameter);
    if(isLargeMemoryRequiredParamEnabled)
    {
        parameter.mixBufferCount = nn::audio::AudioRendererParameter::MixBufferCountMax;
        parameters.push_back(parameter);
    }

    // VoiceCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.voiceCount = 1;
    parameters.push_back(parameter);

    if(isLargeMemoryRequiredParamEnabled)
    {
        // TODO: Workaround for SIGLO-14034
        parameter.voiceCount = 100;
        // parameter.voiceCount = nn::audio::AudioRendererParameter::VoiceCountMax;
        parameters.push_back(parameter);
    }

    // SinkCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sinkCount = 1;
    parameters.push_back(parameter);

    if(isLargeMemoryRequiredParamEnabled)
    {
        parameter.sinkCount = nn::audio::AudioRendererParameter::SinkCountMax;
        parameters.push_back(parameter);
    }

    // EffectCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.effectCount = 0;
    parameters.push_back(parameter);
    if(isLargeMemoryRequiredParamEnabled)
    {
        // TODO: Workaround for SIGLO-14034
        parameter.effectCount = 100;
        // parameter.effectCount = nn::audio::AudioRendererParameter::EffectCountMax;
        parameters.push_back(parameter);
    }

    // SubMixCount
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.subMixCount = 0;
    parameters.push_back(parameter);
    if(isLargeMemoryRequiredParamEnabled)
    {
        // TODO: Workaround for SIGLO-14034
        parameter.subMixCount = 100;
        // parameter.subMixCount = nn::audio::AudioRendererParameter::SubMixCountMax;
        parameters.push_back(parameter);
    }

    // RenderingDevice and ExecutionMode
    const nn::audio::AudioRendererRenderingDevice RenderingDevices[] = {
        nn::audio::AudioRendererRenderingDevice_AudioCoprocessor,
        nn::audio::AudioRendererRenderingDevice_Cpu,
    };

    const nn::audio::AudioRendererExecutionMode ExecutionModes[] = {
        nn::audio::AudioRendererExecutionMode_AutoExecution,
        nn::audio::AudioRendererExecutionMode_ManualExecution,
    };

    for (const auto RenderingDevice : RenderingDevices)
    {
        for (const auto ExecutionMode : ExecutionModes)
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            const auto SupportedMethod = (RenderingDevice == nn::audio::AudioRendererRenderingDevice_Cpu);
#else
            const auto SupportedMethod = (RenderingDevice == nn::audio::AudioRendererRenderingDevice_Cpu && ExecutionMode == nn::audio::AudioRendererExecutionMode_ManualExecution)
                                    && (RenderingDevice == nn::audio::AudioRendererRenderingDevice_AudioCoprocessor && ExecutionMode == nn::audio::AudioRendererExecutionMode_AutoExecution);
#endif
            if(SupportedMethod)
            {
                nn::audio::InitializeAudioRendererParameter(&parameter);
                parameter.renderingDevice = RenderingDevice;
                parameter.executionMode = ExecutionMode;
                parameters.push_back(parameter);
            }
        }
    }

    return parameters;
}

void SetDefaultParameter(nn::audio::AudioRendererParameter* pOutParameter)
{
    nn::audio::InitializeAudioRendererParameter(pOutParameter);
    pOutParameter->mixBufferCount = 16;
    pOutParameter->voiceCount = 24;
}

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024 * 20];  // 4096 == nn::os::MemoryPageSize

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter, nn::os::SystemEvent* pSystemEvent = nullptr, bool isAutoStartEnabled = true)
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

        if (pSystemEvent)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
        }
        else
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
        }

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        if(isAutoStartEnabled)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
        }
    }
    ~ScopedAudioRenderer()
    {
        if(nn::audio::GetAudioRendererState(m_Handle) == nn::audio::AudioRendererState_Started)
        {
            nn::audio::StopAudioRenderer(m_Handle);
        }
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
        m_Allocator.Finalize();
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

class InvalidAudioRendererParameterTest : public ::testing::TestWithParam<nn::audio::AudioRendererParameter>
{

};

INSTANTIATE_TEST_CASE_P(Precondition,
                        InvalidAudioRendererParameterTest,
                        ::testing::ValuesIn(CreateInvalidAudioRendererParameters()));

class ValidAudioRendererParameterTest : public ::testing::TestWithParam<nn::audio::AudioRendererParameter>
{

};

INSTANTIATE_TEST_CASE_P(Success,
                        ValidAudioRendererParameterTest,
                        ::testing::ValuesIn(CreateValidAudioRendererParameters(true)));

class ValidAudioRendererParameterTestForRunningRenderer : public ::testing::TestWithParam<nn::audio::AudioRendererParameter>
{

};

INSTANTIATE_TEST_CASE_P(Success,
                        ValidAudioRendererParameterTestForRunningRenderer,
                        ::testing::ValuesIn(CreateValidAudioRendererParameters(false)));

}

/**
 * @brief       InitializeAudioRendererParameter() の正常系テストです。
 */
TEST(InitializeAudioRendererParameter, Success)
{
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);

    EXPECT_EQ(parameter.sampleRate, 48000);
    EXPECT_EQ(parameter.sampleCount, 240);
    EXPECT_EQ(parameter.mixBufferCount, 1);
    EXPECT_EQ(parameter.subMixCount, 0);
    EXPECT_EQ(parameter.voiceCount, 1);
    EXPECT_EQ(parameter.sinkCount, 1);
    EXPECT_EQ(parameter.effectCount, 0);
    EXPECT_EQ(parameter.performanceFrameCount, 0);
    EXPECT_EQ(parameter.executionMode, nn::audio::AudioRendererExecutionMode_AutoExecution);
#if defined(NN_BUILD_CONFIG_OS_WIN)
    EXPECT_EQ(parameter.renderingDevice, nn::audio::AudioRendererRenderingDevice_Cpu);
#else
    EXPECT_EQ(parameter.renderingDevice, nn::audio::AudioRendererRenderingDevice_AudioCoprocessor);
#endif
}

/**
 * @brief       GetAudioRendererWorkBufferSize() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)

TEST_P(InvalidAudioRendererParameterTest, GetAudioRendererWorkBufferSize)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererWorkBufferSize(GetParam()), "");
}

#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererWorkBufferSize() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, GetAudioRendererWorkBufferSize)
{
    EXPECT_GT(nn::audio::GetAudioRendererWorkBufferSize(GetParam()), 0u);
}

/**
 * @brief       OpenAudioRenderer() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, OpenAudioRenderer)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    const auto& parameter = GetParam();

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererState(handle) == nn::audio::AudioRendererState_Stopped);
    CloseAudioRenderer(handle);

    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererState(handle) == nn::audio::AudioRendererState_Stopped);
    CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

/**
 * @brief       OpenAudioRenderer() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpenAudioRenderer, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;
    nn::os::SystemEvent systemEvent;

    // WorkBuffer is nullptr
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, parameter, nullptr, workBufferSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, nullptr, workBufferSize), "");

    // WorkBufferSize is insufficient
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize - 1), "");

    // SystemEvent is nullptr
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, nullptr, parameter, workBuffer, workBufferSize), "");

    allocator.Free(workBuffer);
}

TEST_P(InvalidAudioRendererParameterTest, OpenAudioRenderer)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    const auto& parameter = GetParam();

    size_t workBufferSize = nn::os::MemoryPageSize * 1024 ; // 4 MB
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;
    nn::os::SystemEvent systemEvent;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       CloseAudioRenderer() の正常系テストです。
 */
TEST(CloseAudioRenderer, Success)
{
    // OpenAudioRenderer の Success テストで代用します
}

/**
 * @brief       CloseAudioRenderer() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(CloseAudioRenderer, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    nn::audio::AudioRendererHandle handle;

    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::CloseAudioRenderer(handle), "");

    // nn::audio::GetAudioRendererState() == AudioRendererState_Stopped の事前条件テスト
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));
    EXPECT_DEATH_IF_SUPPORTED(CloseAudioRenderer(handle), "");
    nn::audio::StopAudioRenderer(handle);
    CloseAudioRenderer(handle);
    allocator.Free(workBuffer);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       StartAudioRenderer() の正常系テストです。
 */
TEST(StartAudioRenderer, Success)
{
    // Voice や Effect のテストでカバーします
}

/**
 * @brief       StartAudioRenderer() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(StartAudioRenderer, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    nn::audio::AudioRendererHandle handle;

    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StartAudioRenderer(handle), "");

    // nn::audio::GetAudioRendererState() == AudioRendererState_Stopped の事前条件テスト
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StartAudioRenderer(handle), "");
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    allocator.Free(workBuffer);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       StopAudioRenderer() の正常系テストです。
 */
TEST(StopAudioRenderer, Success)
{
    // Voice や Effect のテストでカバーします
}

/**
 * @brief       StopAudioRenderer() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(StopAudioRenderer, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    nn::audio::AudioRendererHandle handle;

    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StopAudioRenderer(handle), "");

    // nn::audio::GetAudioRendererState() == AudioRendererState_Stopped の事前条件テスト
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::StopAudioRenderer(handle), "");
    nn::audio::CloseAudioRenderer(handle);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererSampleRate() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, GetAudioRendererSampleRate)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererSampleRate(handle) == parameter.sampleRate);
    CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

/**
 * @brief       GetAudioRendererSampleRate() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioRendererSampleRate, PreCondition)
{
    nn::audio::AudioRendererHandle handle;
    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererSampleRate(handle), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererSampleCount() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, GetAudioRendererSampleCount)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    auto& parameter = GetParam();
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererSampleCount(handle) == parameter.sampleCount);
    CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

/**
 * @brief       GetAudioRendererSampleCount() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioRendererSampleCount, PreCondition)
{
    nn::audio::AudioRendererHandle handle;
    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererSampleCount(handle), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererMixBufferCount() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, GetAudioRendererMixBufferCount)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    auto& parameter = GetParam();
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    nn::audio::AudioRendererHandle handle;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererMixBufferCount(handle) == parameter.mixBufferCount);
    CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

/**
 * @brief       GetAudioRendererMixBufferCount() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioRendererMixBufferCount, PreCondition)
{
    nn::audio::AudioRendererHandle handle;
    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererMixBufferCount(handle), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       RequestUpdateAudioRenderer() の正常系テストです。
 */
TEST(RequestUpdateAudioRenderer, Success)
{
    // Voice や Effect のテストでカバーします
}

/**
 * @brief       RequestUpdateAudioRenderer() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(RequestUpdateAudioRenderer, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererHandle handle;
    nn::audio::AudioRendererConfig config;
    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestUpdateAudioRenderer(handle, &config), "");

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestUpdateAudioRenderer(handle, nullptr), "");
    nn::audio::CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererConfigWorkBufferSize() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, GetAudioRendererConfigWorkBufferSize)
{
    EXPECT_GT(nn::audio::GetAudioRendererConfigWorkBufferSize(GetParam()), 0u);
}

/**
 * @brief       GetAudioRendererConfigWorkBufferSize() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST_P(InvalidAudioRendererParameterTest, GetAudioRendererConfigWorkBufferSize)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererConfigWorkBufferSize(GetParam()), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       IsValidAudioRendererParameter() の異常系テストです。
 */
TEST_P(InvalidAudioRendererParameterTest, IsValidAudioRendererParameter)
{
    EXPECT_FALSE(nn::audio::IsValidAudioRendererParameter(GetParam()));
}

/**
 * @brief       IsValidAudioRendererParameter() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, IsValidAudioRendererParameter)
{
    EXPECT_TRUE(nn::audio::IsValidAudioRendererParameter(GetParam()));
}

/**
 * @brief       GetAudioRendererState() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioRendererState, PreCondition)
{
    nn::audio::AudioRendererHandle handle;
    handle._handle = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererState(handle), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioRendererState() の正常系テストです。
 */
TEST(GetAudioRendererState, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    nn::audio::AudioRendererHandle handle;

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, parameter, workBuffer, workBufferSize));
    EXPECT_TRUE(nn::audio::GetAudioRendererState(handle) == nn::audio::AudioRendererState_Stopped);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));
    EXPECT_TRUE(nn::audio::GetAudioRendererState(handle) == nn::audio::AudioRendererState_Started);
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);

    allocator.Free(workBuffer);
}

/**
 * @brief       InitializeAudioRendererConfig() の正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTest, InitializeAudioRendererConfig)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    auto& parameter = GetParam();
    size_t configSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = allocator.Allocate(configSize);
    ASSERT_NE(configBuffer, nullptr);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configSize);
    allocator.Free(configBuffer);
}

/**
 * @brief       InitializeAudioRendererConfig() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST_P(InvalidAudioRendererParameterTest, InitializeAudioRendererConfig)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    auto& parameter = GetParam();
    size_t configSize = 1024 * 1024 * 4; // 4 MB (大よそ足りるサイズ)
    void* configBuffer = allocator.Allocate(configSize);
    nn::audio::AudioRendererConfig config;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configSize), "");

    allocator.Free(configBuffer);
}

TEST(InitializeAudioRendererConfig, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    size_t configSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = allocator.Allocate(configSize);
    nn::audio::AudioRendererConfig config;

    SetDefaultParameter(&parameter);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeAudioRendererConfig(nullptr, parameter, configBuffer, configSize), "");

    SetDefaultParameter(&parameter);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeAudioRendererConfig(&config, parameter, nullptr, configSize), "");

    SetDefaultParameter(&parameter);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configSize - 1), "");

    allocator.Free(configBuffer);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       Get/SetAudioRendererRenderingTimeLimit() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetSetAudioRendererRenderingTimeLimit, PreCondition)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    ScopedAudioRenderer sar(parameter);

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 101), "");

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 100));

    nn::audio::AudioRendererHandle invalidHandle = { nullptr };
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererRenderingTimeLimit(invalidHandle), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioRendererRenderingTimeLimit(invalidHandle, 100), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(SetAudioRendererRenderingTimeLimit, OverRenderingTimeLimit)
{
    static NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t workBuffer[nn::audio::MemoryPoolType::SizeGranularity * 1024];
    nn::mem::StandardAllocator allocator(workBuffer, sizeof(workBuffer));

    const int VoiceCount = 300;

    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    parameter.voiceCount = VoiceCount;
    parameter.effectCount = 1;
    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::VoiceType voices[VoiceCount];
    nn::audio::WaveBuffer waveBuffers[VoiceCount];
    int16_t* pPcmBuffers[VoiceCount];
    nn::audio::FinalMixType finalMix;

    nn::audio::MemoryPoolType memoryPool;

    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, workBuffer, sizeof(workBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 4));

    // Playback voices
    for (int i = 0; i < VoiceCount; ++i)
    {
        auto& voice = voices[i];
        auto& waveBuffer = waveBuffers[i];
        auto& pPcmBuffer = pPcmBuffers[i];

        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);
        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        pPcmBuffer = static_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * 1024, nn::audio::MemoryPoolType::AddressAlignment));
        waveBuffer.buffer = pPcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pPcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pPcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    systemEvent.Wait();

    const int interationCount = 10;
    for (int i = 0; i < interationCount; ++i)
    {
        // Add and remove Delay effect
        {
            nn::audio::DelayType delay;
            const int channelCount = 4;
            const nn::TimeSpan delayTime = nn::TimeSpan::FromSeconds(1);
            const size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTime, parameter.sampleRate, channelCount);
            void* pDelayBuffer = allocator.Allocate(delaySize);
            ASSERT_NE(pDelayBuffer, nullptr);

            NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDelay(&sar.GetConfig(), &delay, pDelayBuffer, delaySize, &finalMix, delayTime, channelCount));
            nn::audio::SetDelayEnabled(&delay, i % 2 == 0); // 偶数番目の時は有効にする
            const int8_t input[] = { 0, 1, 2, 3 };
            const int8_t output[] = { 0, 1, 2, 3 };
            nn::audio::SetDelayInputOutput(&delay, input, output, sizeof(input) / sizeof(input[0]));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            nn::audio::SetDelayEnabled(&delay, false);
            do
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsDelayRemovable(&delay) == false);

            ASSERT_TRUE(nn::audio::RemoveDelay(&sar.GetConfig(), &delay, &finalMix) == pDelayBuffer);
            allocator.Free(pDelayBuffer);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Add and remove Reverb effect
        {
            nn::audio::ReverbType reverb;
            const int channelCount = 4;
            auto reverbSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCount);
            void* pReverbBuffer = allocator.Allocate(reverbSize);
            NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddReverb(&sar.GetConfig(), &reverb, pReverbBuffer, reverbSize, &finalMix, channelCount));
            nn::audio::SetReverbEnabled(&reverb, i % 2 == 0); // 偶数番目の時は有効にする
            const int8_t input[] = { 0, 1, 2, 3 };
            const int8_t output[] = { 0, 1, 2, 3 };
            nn::audio::SetReverbInputOutput(&reverb, input, output, sizeof(input) / sizeof(input[0]));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            nn::audio::SetReverbEnabled(&reverb, false);
            do
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsReverbRemovable(&reverb) == false);

            ASSERT_TRUE(nn::audio::RemoveReverb(&sar.GetConfig(), &reverb, &finalMix) == pReverbBuffer);
            allocator.Free(pReverbBuffer);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Add and remove I3dl2Reverb effect
        {
            nn::audio::I3dl2ReverbType i3dl2Reverb;
            const int channelCount = 2;
            auto reverbSize = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, channelCount);
            void* pReverbBuffer = allocator.Allocate(reverbSize);
            NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddI3dl2Reverb(&sar.GetConfig(), &i3dl2Reverb, pReverbBuffer, reverbSize, &finalMix, channelCount));
            nn::audio::SetI3dl2ReverbEnabled(&i3dl2Reverb, i % 2 == 0); // 偶数番目の時は有効にする
            const int8_t input[] = { 0, 1, 2, 3 };
            const int8_t output[] = { 0, 1, 2, 3 };
            nn::audio::SetI3dl2ReverbInputOutput(&i3dl2Reverb, input, output, sizeof(input) / sizeof(input[0]));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            nn::audio::SetI3dl2ReverbEnabled(&i3dl2Reverb, false);
            do
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsI3dl2ReverbRemovable(&i3dl2Reverb) == false);

            ASSERT_TRUE(nn::audio::RemoveI3dl2Reverb(&sar.GetConfig(), &i3dl2Reverb, &finalMix) == pReverbBuffer);
            allocator.Free(pReverbBuffer);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Add and remove Aux effect
        {
            nn::audio::AuxType aux;
            const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, 10, 1);
            int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
            int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
            NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(&sar.GetConfig(), &aux, &finalMix, pSendBuffer, pReturnBuffer, extBufSize));
            nn::audio::SetAuxEnabled(&aux, i % 2 == 0); // 偶数番目の時は有効にする
            const int8_t input[] = { 0, 1, 2, 3 };
            const int8_t output[] = { 0, 1, 2, 3 };
            nn::audio::SetAuxInputOutput(&aux, input, output, sizeof(input) / sizeof(input[0]));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            nn::audio::SetAuxEnabled(&aux, false);
            do
            {
                int32_t buffer[16]; // 適当なバッファサイズ
                nn::audio::ReadAuxSendBuffer(&aux, buffer, sizeof(buffer) / sizeof(buffer[0]));
                nn::audio::WriteAuxReturnBuffer(&aux, buffer, sizeof(buffer) / sizeof(buffer[0]));
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsAuxRemovable(&aux) == false);

            nn::audio::RemoveAux(&sar.GetConfig(), &aux, &finalMix);
            allocator.Free(pSendBuffer);
            allocator.Free(pReturnBuffer);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // Add and remove BufferMixer effect
        {
            nn::audio::BufferMixerType bufferMixer;
            NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddBufferMixer(&sar.GetConfig(), &bufferMixer, &finalMix));
            nn::audio::SetBufferMixerEnabled(&bufferMixer, i % 2 == 0); // 偶数番目の時は有効にする
            int8_t input[] = { 0, 1, 2, 3 };
            int8_t output[] = { 0, 1, 2, 3 };
            nn::audio::SetBufferMixerInputOutput(&bufferMixer, input, output, sizeof(input) / sizeof(input[0]));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            nn::audio::SetBufferMixerEnabled(&bufferMixer, false);
            do
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsBufferMixerRemovable(&bufferMixer) == false);

            nn::audio::RemoveBufferMixer(&sar.GetConfig(), &bufferMixer, &finalMix);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();
        }

        // MemoryPool
        {
            static NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN uint8_t tmpWorkBuffer[nn::audio::MemoryPoolType::SizeGranularity];
            nn::audio::MemoryPoolType tmpMemoryPool;
            ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &tmpMemoryPool, tmpWorkBuffer, sizeof(tmpWorkBuffer)));
            ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&tmpMemoryPool));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
            systemEvent.Wait();

            ASSERT_TRUE(nn::audio::RequestDetachMemoryPool(&tmpMemoryPool));

            do
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
                systemEvent.Wait();
            } while (nn::audio::IsMemoryPoolAttached(&tmpMemoryPool));

            nn::audio::ReleaseMemoryPool(&sar.GetConfig(), &tmpMemoryPool);
        }
    }

    nn::audio::StopAudioRenderer(sar.GetHandle());

    for (int i = 0; i < VoiceCount; ++i)
    {
        allocator.Free(pPcmBuffers[i]);
    }
} // NOLINT(readability/fn_size)

/**
 * @brief       Get/SetAudioRendererRenderingTimeLimit() の正常系テストです。
 */
TEST(GetSetAudioRendererRenderingTimeLimit, Success)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::VoiceType voice;
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = pcmBuffer;
    waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
    waveBuffer.loop = 0;
    waveBuffer.isEndOfStream = false;

    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    systemEvent.Wait();

    // default value
    EXPECT_EQ(nn::audio::GetAudioRendererRenderingTimeLimit(sar.GetHandle()), 100);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    systemEvent.Wait();

    // min value
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 0));
    EXPECT_EQ(nn::audio::GetAudioRendererRenderingTimeLimit(sar.GetHandle()), 0);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    // SetAudioRendererRenderingTimeLimit() の呼び出しタイミングによってシグナルする時としない場合があるので TimedWait() で待機
    // * サーバ側が SystemEvent をシグナル状態にしてコマンドを DSP へ送信する間に SetAudioRendererRenderingTimeLimit() を呼び出した場合は、
    //   一度もシグナル状態にならない。DSP へコマンドを送信した後に呼び出した場合は、一度だけシグナル状態になる。
    const auto audioFrame = 1000 / (parameter.sampleRate / parameter.sampleCount);
    systemEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(audioFrame * 2)); // 2 audio frames

    // max value
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 100));
    EXPECT_EQ(nn::audio::GetAudioRendererRenderingTimeLimit(sar.GetHandle()), 100);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    systemEvent.Wait();

    // median value
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioRendererRenderingTimeLimit(sar.GetHandle(), 50));
    EXPECT_EQ(nn::audio::GetAudioRendererRenderingTimeLimit(sar.GetHandle()), 50);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    systemEvent.Wait();
}


/**
 * @brief       マルチスレッドから複数のオーディオレンダラを扱う正常系テストです。
 */
TEST_P(ValidAudioRendererParameterTestForRunningRenderer, AudioRendererMultiInstanceMultiThread)
{
    struct TesterThreadInfo
    {
        nn::os::ThreadType thread;
        nn::audio::AudioRendererParameter parameter;
        void* pThreadStack;
        void* pWorkBuffer;
        size_t workBufferSize;
    };


    const auto threadFunc = [](void* args)
    {
        for(int i = 0; i < 10; ++i)
        {
            // Setup Allocator
            auto pTestThreadInfo = reinterpret_cast<TesterThreadInfo*>(args);
            nn::mem::StandardAllocator allocator;
            allocator.Initialize(pTestThreadInfo->pWorkBuffer, pTestThreadInfo->workBufferSize);

            // Setup AudioRenderer
            nn::os::SystemEvent systemEvent;
            nn::audio::AudioRendererHandle handle;
            nn::audio::AudioRendererConfig config;

            auto pParameter = reinterpret_cast<nn::audio::AudioRendererParameter*>(&pTestThreadInfo->parameter);
            size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(*pParameter);
            auto pWorkBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
            ASSERT_NE(nullptr, pWorkBuffer);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, *pParameter, pWorkBuffer, workBufferSize));

            size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(*pParameter);
            auto pConfigBuffer = allocator.Allocate(configBufferSize);
            nn::audio::InitializeAudioRendererConfig(&config, *pParameter, pConfigBuffer, configBufferSize);
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

            // Setup FinalMix
            nn::audio::FinalMixType finalMix;
            EXPECT_TRUE(nn::audio::AcquireFinalMix(&config, &finalMix, 1));

            // Setup MemoryPool for Voice
            nn::audio::MemoryPoolType memoryPool;
            NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
            ASSERT_TRUE(nn::audio::AcquireMemoryPool(&config, &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
            ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

            // Setup Voice
            nn::audio::VoiceType voice;
            EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&config, &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
            nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
            nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

            nn::audio::WaveBuffer waveBuffer;
            waveBuffer.buffer = pcmBuffer;
            waveBuffer.size = pParameter->sampleCount * sizeof(pcmBuffer[0]);
            waveBuffer.startSampleOffset = 0;
            waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
            waveBuffer.loop = 0;
            waveBuffer.isEndOfStream = false;

            nn::audio::AppendWaveBuffer(&voice, &waveBuffer);

            // TODO: SIGLO-70073 の調査が終わった後にコメントを外す
            // NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

            NN_LOG("Start Test sampleRate(%d) i(%d)\n", pParameter->sampleRate, i);
            const auto beginTick = nn::os::GetSystemTick();
            while(nn::os::GetSystemTick() - beginTick < nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(100)))
            {
                if(nn::audio::GetReleasedWaveBuffer(&voice) != nullptr)
                {
                    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
                }

                // 手動レンダリングモードの時はレンダリングをこのスレッド上で実行する
                if(pParameter->executionMode == nn::audio::AudioRendererExecutionMode_ManualExecution)
                {
                    nn::audio::ExecuteAudioRendererRendering(handle);
                }

                // 偶数回目の時は SystemEvent がシグナル状態になるのを待つパターンでテスト
                if(i % 2 == 0)
                {
                    systemEvent.Wait();
                }
            }
            NN_LOG("End Playback sampleRate(%d) i(%d)\n", pParameter->sampleRate, i);

            nn::audio::StopAudioRenderer(handle);
            nn::audio::CloseAudioRenderer(handle);
            allocator.Free(pConfigBuffer);
            allocator.Free(pWorkBuffer);
            allocator.Finalize();
        }
    };

    const int ThreadCount = 2;
    TesterThreadInfo testerThreadInfo[ThreadCount];

    nn::mem::StandardAllocator allocator;
    const auto Parameter = GetParam();
    const auto StackSize = nn::util::align_up(1024 * 64,  nn::os::ThreadStackAlignment);
    const size_t ThreadWorkBufferSize = nn::util::align_up(nn::audio::GetAudioRendererWorkBufferSize(Parameter), nn::os::MemoryPageSize)
                                + nn::util::align_up(nn::audio::GetAudioRendererConfigWorkBufferSize(Parameter), nn::os::MemoryPageSize)
                                + 1024 * 64; // Allocator の管理領域など
    const auto WorkBufferSize = (StackSize + ThreadWorkBufferSize) * ThreadCount + 1024 * 64; // Allocator の管理領域など

    std::unique_ptr<uint8_t[]> workBuffer(new uint8_t[WorkBufferSize]);
    ASSERT_NE(nullptr, workBuffer.get());
    allocator.Initialize(workBuffer.get(), WorkBufferSize);

    for(int i = 0; i < ThreadCount; ++i)
    {
        testerThreadInfo[i].pThreadStack = allocator.Allocate(StackSize, nn::os::ThreadStackAlignment);
        ASSERT_NE(testerThreadInfo[i].pThreadStack, nullptr);
        NN_LOG("WorkBufferSize(%zd) ThreadWorkBufferSize(%zd) (%zd)\n", WorkBufferSize, ThreadWorkBufferSize, allocator.GetAllocatableSize());
        testerThreadInfo[i].pWorkBuffer = allocator.Allocate(ThreadWorkBufferSize);
        ASSERT_NE(testerThreadInfo[i].pWorkBuffer, nullptr);
        testerThreadInfo[i].workBufferSize = ThreadWorkBufferSize;
        testerThreadInfo[i].parameter = Parameter;

        // 各スレッドで出来るだけ異なるサンプルレートを利用させる
        int sampleRate;
        if(i % 2 == 0)
        {
            sampleRate = 32000;
        }
        else
        {
            sampleRate = 48000;
        }

        testerThreadInfo[i].parameter.sampleRate = sampleRate;
        testerThreadInfo[i].parameter.sampleCount = testerThreadInfo[i].parameter.sampleRate / 200;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&testerThreadInfo[i].thread, threadFunc, &testerThreadInfo[i], testerThreadInfo[i].pThreadStack, StackSize, nn::os::DefaultThreadPriority));
        const int coreNo = i + 1;
        nn::os::SetThreadCoreMask(&testerThreadInfo[i].thread, coreNo, 1llu << coreNo);
        nn::os::StartThread(&testerThreadInfo[i].thread);
    }
    for(int i = 0; i < ThreadCount; ++i)
    {
        nn::os::WaitThread(&testerThreadInfo[i].thread);
        nn::os::DestroyThread(&testerThreadInfo[i].thread);
        allocator.Free(testerThreadInfo[i].pWorkBuffer);
        allocator.Free(testerThreadInfo[i].pThreadStack);
    }

    allocator.Finalize();
} // NOLINT(readability/fn_size)

TEST(ExecuteAudioRendererRendering, Failure)
{
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    parameter.executionMode = nn::audio::AudioRendererExecutionMode_AutoExecution;

    {
        ScopedAudioRenderer sar(parameter);
        EXPECT_TRUE(nn::audio::ResultNotSupported::Includes(nn::audio::ExecuteAudioRendererRendering(sar.GetHandle())));
    }

    {
        nn::os::SystemEvent systemEvent;
        ScopedAudioRenderer sar(parameter, &systemEvent);
        EXPECT_TRUE(nn::audio::ResultNotSupported::Includes(nn::audio::ExecuteAudioRendererRendering(sar.GetHandle())));
    }
}

TEST(ExecuteAudioRendererRendering, Precondition)
{
    nn::audio::AudioRendererHandle handle = { nullptr };

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ExecuteAudioRendererRendering(handle), "");
}

TEST(ExecuteAudioRendererRendering, Success)
{
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererParameter parameter;
    SetDefaultParameter(&parameter);
    parameter.executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
    parameter.renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;

    nn::os::SystemEvent* pSystemEvents[] = { nullptr, &systemEvent };

    for(auto& pSystemEvent : pSystemEvents)
    {
        ScopedAudioRenderer sar(parameter, pSystemEvent);

        nn::audio::VoiceType voice;
        nn::audio::FinalMixType finalMix;
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::MemoryPoolType memoryPool;
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = 0;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(sar.GetHandle()));

        if(pSystemEvent != nullptr)
        {
            pSystemEvent->Wait();
        }
    }
}

/**
 * @brief       複数のオーディオレンダラの負荷制御を扱う正常系テストです。
 */
TEST(AudioRendererMultiInstanceProcessingTimeControl, Success)
{

    /*  TODO: DSP の処理落ちを検知する手段が現状ないので手動テスト
     *  # テスト方法
     *  * サイン波が音切れせずに聞こえることを確認
     *  # Note:
     *  * 複数のオーディオレンダラの合計処理時間が上限を超えた場合、どのように処理時間が制限されるかはプラットフォーム依存としている。(現状では優先度設定 API はない)
     *  * NX では、先に作成したオーディオレンダラが優先される仕様となっているため、二個目に作成したオーディオレンダラに過負荷を掛けたときに、
     *  * 最初に作成したオーディオレンダラの音が途切れないことを確認すればよい
     */

    struct TesterThreadInfo
    {
        nn::os::ThreadType thread;
        nn::audio::AudioRendererParameter parameter;
        void* pThreadStack;
        void* pWorkBuffer;
        size_t workBufferSize;
        float volume;
    };

    const int threadCount = 2;
    TesterThreadInfo testerThreadInfo[threadCount];

    nn::mem::StandardAllocator allocator;
    allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    const auto threadFunc = [](void* args)
    {
        // 再生時間
        const auto PlaybackDuration = nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(3));

        // Setup Allocator
        auto pTestThreadInfo = reinterpret_cast<TesterThreadInfo*>(args);
        nn::mem::StandardAllocator allocator;
        allocator.Initialize(pTestThreadInfo->pWorkBuffer, pTestThreadInfo->workBufferSize);

        // Setup AudioRenderer
        nn::os::SystemEvent systemEvent;
        nn::audio::AudioRendererHandle handle;
        nn::audio::AudioRendererConfig config;

        auto pParameter = reinterpret_cast<nn::audio::AudioRendererParameter*>(&pTestThreadInfo->parameter);
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(*pParameter);
        auto pWorkBuffer = allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, *pParameter, pWorkBuffer, workBufferSize));

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(*pParameter);
        auto pConfigBuffer = allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&config, *pParameter, pConfigBuffer, configBufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

        // Setup FinalMix
        nn::audio::FinalMixType finalMix;
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&config, &finalMix, 2));
        nn::audio::SetFinalMixVolume(&finalMix, pTestThreadInfo->volume);

        // Setup DeviceSink
        const int ChannelCount = 2;
        int8_t mainBus[ChannelCount] = { 0, 1 };
        nn::audio::DeviceSinkType deviceSink;
        nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, ChannelCount, "MainAudioOut");
        NN_ABORT_UNLESS(result.IsSuccess());

        // Setup MemoryPool for Voice
        nn::audio::MemoryPoolType memoryPool;
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity * 4];
        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&config, &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

        // Generate SineWave
        const auto SineFrequency = 480;
        const auto SineWaveSampleCount = pParameter->sampleRate / SineFrequency * 10;
        const float Pi = 3.1415926535897932384626433f;
        for (auto i = 0; i < SineWaveSampleCount; ++i)
        {
            pcmBuffer[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * SineFrequency * i / pParameter->sampleRate));
        }

        // Setup Voice
        const auto VoiceCountMax = 200;
        ASSERT_LE(pParameter->voiceCount, VoiceCountMax);

        nn::audio::VoiceType voices[VoiceCountMax];
        nn::audio::WaveBuffer waveBuffers[VoiceCountMax];

        for(int i = 0; i < pParameter->voiceCount; ++i)
        {
            auto& voice = voices[i];
            EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&config, &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
            nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
            nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
            nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, mainBus[0]);
            nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, mainBus[1]);

            auto& waveBuffer = waveBuffers[i];
            waveBuffer.buffer = pcmBuffer;
            waveBuffer.size = SineWaveSampleCount * sizeof(int16_t);
            waveBuffer.startSampleOffset = 0;
            waveBuffer.endSampleOffset = SineWaveSampleCount;
            waveBuffer.loop = true;
            waveBuffer.isEndOfStream = false;

            nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));
        const auto beginTick = nn::os::GetSystemTick();
        while(nn::os::GetSystemTick() - beginTick < PlaybackDuration)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));
            systemEvent.Wait();
        }

        nn::audio::StopAudioRenderer(handle);
        nn::audio::CloseAudioRenderer(handle);
        allocator.Free(pConfigBuffer);
        allocator.Free(pWorkBuffer);
        allocator.Finalize();
    };

    for(int i = 0; i < threadCount; ++i)
    {
        const size_t stackSize = 1024 * 1024 * 1;
        const size_t workBufferSize = 1024 * 1024 * 3;
        testerThreadInfo[i].pThreadStack = allocator.Allocate(stackSize, nn::os::ThreadStackAlignment);
        ASSERT_NE(testerThreadInfo[i].pThreadStack, nullptr);
        testerThreadInfo[i].pWorkBuffer = allocator.Allocate(workBufferSize);
        ASSERT_NE(testerThreadInfo[i].pWorkBuffer, nullptr);
        testerThreadInfo[i].workBufferSize = workBufferSize;
        nn::audio::InitializeAudioRendererParameter(&testerThreadInfo[i].parameter);

        testerThreadInfo[i].parameter.sinkCount = 1;
        testerThreadInfo[i].parameter.mixBufferCount = 2;

        if(i == 0)
        {
            testerThreadInfo[i].parameter.voiceCount = 1;
            testerThreadInfo[i].volume = 1.f;
        }
        else
        {
            testerThreadInfo[i].parameter.voiceCount = 200; // オーディオフレームあたりの処理時間上限を超える個数
            testerThreadInfo[i].volume = 0.0f;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&testerThreadInfo[i].thread, threadFunc, &testerThreadInfo[i], testerThreadInfo[i].pThreadStack, stackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&testerThreadInfo[i].thread);
    }

    for(int i = 0; i < threadCount; ++i)
    {
        nn::os::WaitThread(&testerThreadInfo[i].thread);
        nn::os::DestroyThread(&testerThreadInfo[i].thread);
        allocator.Free(testerThreadInfo[i].pWorkBuffer);
        allocator.Free(testerThreadInfo[i].pThreadStack);
    }

    allocator.Finalize();
} // NOLINT(readability/fn_size)

TEST(GetAudioRendererElapsedFrameCount, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioRendererElapsedFrameCount(nullptr), "");
}

TEST_P(ValidAudioRendererParameterTest, GetAudioRendererElapsedFrameCount)
{
    const auto parameter = GetParam();
    nn::os::SystemEvent systemEvent;
    ScopedAudioRenderer sar(parameter, &systemEvent, false);

    // Check initial value
    EXPECT_EQ(nn::audio::GetAudioRendererElapsedFrameCount(&sar.GetConfig()), 0);

    // Start AudioRenderer
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(sar.GetHandle()));

    {
        if(parameter.executionMode == nn::audio::AudioRendererExecutionMode_ManualExecution)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(sar.GetHandle()));
        }
        else
        {
            // Wait for 1 audio frame
            EXPECT_TRUE(systemEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(100)));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        // Check elapsedFrameCount is increased
        EXPECT_GE(nn::audio::GetAudioRendererElapsedFrameCount(&sar.GetConfig()), 1);
    }

    {
        // Stop AudioRenderer and wait for a while
        nn::audio::StopAudioRenderer(sar.GetHandle());
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        const auto elapsedFrameCount = nn::audio::GetAudioRendererElapsedFrameCount(&sar.GetConfig());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        // Check elapsedFrameCount is not increased after AudioRenderer is stopped
        EXPECT_EQ(elapsedFrameCount, nn::audio::GetAudioRendererElapsedFrameCount(&sar.GetConfig()));
    }

    // There is no method to check strictly that elapsedFrameCount is reset by StartAudioRenderer() so do logging only.
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(sar.GetHandle()));
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        NN_LOG("elapsedFrameCount: %d\n", nn::audio::GetAudioRendererElapsedFrameCount(&sar.GetConfig()));
    }
}
