﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include<string>
#include<cstring>
#include<list>
#include<algorithm>
#include<cmath>
#include<numeric>
#include <nnt.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nn/audio/audio_PerformanceMetricsTypes.private.h>
#include <nn/audioctrl.h>
#include <nn/audioctrl/audioctrl_AudioControllerDeviceControllerShimForProduction.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>
#include <memory> // std::unique_ptr
#include <random>

// To run this test, please enable "PerformanceMetrics for DSP Command" in the audio process.
#if 0

namespace
{

const int EffectChannelCountMax = 6;

class StatHelper
{
public:
    void Add(uint32_t value)
    {
        // m_Values.push_front(value);
        m_Values.push_back(value);

        m_Min = std::min(m_Min, static_cast<double>(value));
        m_Max = std::max(m_Max, static_cast<double>(value));
        m_Total += value;
    }

    double GetMin() const
    {
        return m_Min;
    }

    double GetMax() const
    {
        return m_Max;
    }

    double GetTotal() const
    {
        return m_Total;
    }

    int GetCount() const
    {
        return static_cast<int>(std::distance(m_Values.begin(), m_Values.end()));
    }

    double GetAverage() const
    {
        return m_Total / GetCount();
    }

    double GetStddiv() const
    {
        const auto average = GetAverage();
        double result = 0.0;

        for(const auto value : m_Values)
        {
            result += std::sqrt(std::pow(value - average, 2.0));
        }

        return result / GetCount();
    }

    double GetCenter()
    {
        std::sort(m_Values.begin(), m_Values.end());

        return m_Values[m_Values.size() / 2];
    }

private:
    double m_Min = std::numeric_limits<decltype(m_Min)>::max();
    double m_Max = std::numeric_limits<decltype(m_Max)>::min();
    double m_Total = 0.0;
    std::vector<double> m_Values;
};

struct StatResult
{
    double min;
    double max;
    double ave;
    double stddiv;
    int count;
};

class PerfResultSet
{
public:
    void Add(uint32_t measured, uint32_t estimated)
    {
        // NN_LOG("[PerfResultSet] measured:%u estimated:%u\n", measured, estimated);
        m_Measured.Add(measured);
        m_Estimated.Add(estimated);
        m_Error2.Add(static_cast<uint32_t>(std::pow(static_cast<double>(measured) - static_cast<double>(estimated), 2.0)));
    }

    StatResult GetMeasuredStatResult() const
    {
        return { m_Measured.GetMin(), m_Measured.GetMax(), m_Measured.GetAverage(), m_Measured.GetStddiv(), m_Measured.GetCount() };
    }

    StatResult GetEstimatedStatResult() const
    {
        return { m_Estimated.GetMin(), m_Estimated.GetMax(), m_Estimated.GetAverage(), m_Estimated.GetStddiv(), m_Estimated.GetCount() };
    }

    StatResult GetError2StatResult() const
    {
        return { m_Error2.GetMin(), m_Error2.GetMax(), m_Error2.GetAverage(), m_Error2.GetStddiv(), m_Error2.GetCount() };
    }

private:
    StatHelper m_Measured{};
    StatHelper m_Estimated{};
    StatHelper m_Error2{};
};

const char* ToString(nn::audio::PerformanceSysDetailType type)
{
    switch(type)
    {
    case nn::audio::PerformanceSysDetailType_Invalid:
        return "Invalid";
    case nn::audio::PerformanceSysDetailType_PcmInt16DataSource:
        return "PcmInt16DataSource";
    case nn::audio::PerformanceSysDetailType_AdpcmDataSource:
        return "AdpcmDataSource";
    case nn::audio::PerformanceSysDetailType_Volume:
        return "Volume";
    case nn::audio::PerformanceSysDetailType_VolumeRamp:
        return "VolumeRamp";
    case nn::audio::PerformanceSysDetailType_BiquadFilter:
        return "BiquadFilter";
    case nn::audio::PerformanceSysDetailType_Mix:
        return "Mix";
    case nn::audio::PerformanceSysDetailType_MixRamp:
        return "MixRamp";
    case nn::audio::PerformanceSysDetailType_MixRampGrouped:
        return "MixRampGrouped";
    case nn::audio::PerformanceSysDetailType_DepopPrepare:
        return "DepopPrepare";
    case nn::audio::PerformanceSysDetailType_DepopForMixBuffers:
        return "DepopForMixBuffers";
    case nn::audio::PerformanceSysDetailType_Delay:
        return "Delay";
    case nn::audio::PerformanceSysDetailType_Upsample:
        return "Upsample";
    case nn::audio::PerformanceSysDetailType_DownMix6chTo2ch:
        return "DownMix6chTo2ch";
    case nn::audio::PerformanceSysDetailType_Aux:
        return "Aux";
    case nn::audio::PerformanceSysDetailType_DeviceSink:
        return "DeviceSink";
    case nn::audio::PerformanceSysDetailType_CircularBufferSink:
        return "CircularBufferSink";
    case nn::audio::PerformanceSysDetailType_Reverb:
        return "Reverb";
    case nn::audio::PerformanceSysDetailType_I3dl2Reverb:
        return "I3dl2Reverb";
    case nn::audio::PerformanceSysDetailType_Performance:
        return "Performance";
    case nn::audio::PerformanceSysDetailType_ClearMixBuffer:
        return "ClearMixBuffer";
    case nn::audio::PerformanceSysDetailType_CopyMixBuffer:
        return "CopyMixBuffer";
    default:
        return "Unknown ";
    }
}

class ITarget
{
public:
    virtual ~ITarget(){};
    virtual void Update() = 0;
};

class Voice : public ITarget
{
public:
    Voice(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, int sampleRate, nn::audio::SampleFormat format)
        : m_Allocator(allocator),
          m_RandomEngine(),
          m_Config(pConfig)

    {
        const auto channelCount = 1;
        const auto parameterSize = sizeof(nn::audio::AdpcmParameter);
        const auto parameter = m_Allocator.Allocate(parameterSize, nn::audio::BufferAlignSize);

        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(pConfig, &m_Voice, sampleRate, channelCount, format, nn::audio::VoiceType::PriorityHighest, parameter, parameterSize));

        const auto sampleCount = sampleRate / 2; // 0.5 seconds

        // ここはクラッシュしない限りは適当な値でよいので Adpcm 時も int16_t 換算で計算する
        // MEMO: GetSampleByteSize() は Adpcm には使えない
        const auto dataSize = channelCount * sizeof(int16_t) * sampleCount;
        const auto bufferSize = nn::util::align_up(dataSize, nn::audio::BufferAlignSize);
        auto buffer = m_Allocator.Allocate(bufferSize, nn::audio::BufferAlignSize);
        EXPECT_NE(buffer, nullptr);
        for(auto& waveBuffer : m_WaveBuffers)
        {
            waveBuffer.buffer = buffer;
            waveBuffer.size = bufferSize;
            waveBuffer.startSampleOffset = 0;
            waveBuffer.endSampleOffset = sampleCount;
            waveBuffer.loop = false;
            waveBuffer.isEndOfStream = false;
            waveBuffer.pContext = nullptr;
            waveBuffer.contextSize = 0;

            nn::audio::AppendWaveBuffer(&m_Voice, &waveBuffer);
        }

        nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceVolume(&m_Voice, 1.0f);
    }

    int GetMixMixBufferCount(const nn::audio::FinalMixType* pFinalMix) const
    {
        return nn::audio::GetFinalMixBufferCount(pFinalMix);
    }

    int GetMixMixBufferCount(const nn::audio::SubMixType* pSubMix) const
    {
        return nn::audio::GetSubMixBufferCount(pSubMix);
    }

    template<typename Mix>
    void SetMixVolume(Mix* mix, float volume, int count)
    {
        nn::audio::SetVoiceDestination(m_Config, &m_Voice, mix);

        const auto mixMixBufferCount = GetMixMixBufferCount(mix);
        for(int destIndex = 0; destIndex < mixMixBufferCount; ++destIndex)
        {
            // 指定されたチャンネル以上は 0 を設定しておく
            const auto mixVolume = destIndex < count ? volume : 0.0f;
            nn::audio::SetVoiceMixVolume(&m_Voice, mix, mixVolume, 0, destIndex);
        }
    }

    void Update() NN_OVERRIDE
    {
        while(auto waveBuffer = nn::audio::GetReleasedWaveBuffer(&m_Voice))
        {
            // TORIAEZU: const_cast しておく
            nn::audio::AppendWaveBuffer(&m_Voice, const_cast<nn::audio::WaveBuffer*>(waveBuffer));
        }
    }

    void SetPitch(float pitch)
    {
        nn::audio::SetVoicePitch(&m_Voice, pitch);
    }

    void SetBiquadFilterEnabled(bool enabled)
    {
        nn::audio::BiquadFilterParameter filterParameter = { enabled, {720, 1439, 720}, {22684, -9350} };

        // コマンドの個数を稼ぐために 全段に設定を反映させる
        for(int index = 0; index < nn::audio::VoiceType::BiquadFilterCountMax; ++index)
        {
            nn::audio::SetVoiceBiquadFilterParameter(&m_Voice, index, filterParameter);
        }
    }

    const nn::audio::VoiceType* GetBase()
    {
        return &m_Voice;
    }

private:
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::VoiceType m_Voice;
    nn::audio::AudioRendererConfig* m_Config;
    nn::audio::WaveBuffer m_WaveBuffers[nn::audio::VoiceType::WaveBufferCountMax];
};

class Delay : public ITarget
{
public:
    Delay(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, nn::audio::FinalMixType* pFinalMix, int channelCountMax, nn::TimeSpan delayTimeMax)
        : m_Allocator(allocator),
          m_RandomEngine()
    {
        const auto sampleRate = nn::audio::GetFinalMixSampleRate(pFinalMix);
        const auto bufferSize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, sampleRate, channelCountMax);
        auto buffer = m_Allocator.Allocate(bufferSize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(buffer);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDelay(pConfig, &m_Delay, buffer, bufferSize, pFinalMix, delayTimeMax, channelCountMax));
    }

    void Update() NN_OVERRIDE
    {
        // RandomizeParameters();
    }

    void RandomizeParameters()
    {
        std::uniform_int_distribution<int64_t> delayTime(0, nn::audio::GetDelayTimeMax(&m_Delay).GetNanoSeconds());
        std::uniform_real_distribution<float> inGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> feedbackGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> dryGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> lowPassAmount(0.0f, 1.0f);

        const nn::audio::DelayParameterSet parameterSet = {
            nn::TimeSpan::FromNanoSeconds(delayTime(m_RandomEngine)),
            inGain(m_RandomEngine),
            feedbackGain(m_RandomEngine),
            dryGain(m_RandomEngine),
            lowPassAmount(m_RandomEngine)
        };

        nn::audio::SetDelayParameters(&m_Delay, &parameterSet);
    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetDelayEnabled(&m_Delay, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        NN_ASSERT_MINMAX(channelCount, 0, EffectChannelCountMax);
        int8_t channels[EffectChannelCountMax];
        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        nn::audio::SetDelayInputOutput(&m_Delay, channels, channels, channelCount);
    }

private:
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::DelayType m_Delay;
};

class Reverb : public ITarget
{
public:
    Reverb(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, nn::audio::FinalMixType* pFinalMix, int channelCountMax)
        : m_Allocator(allocator),
          m_RandomEngine()
    {
        const auto sampleRate = nn::audio::GetFinalMixSampleRate(pFinalMix);
        const auto bufferSize = nn::audio::GetRequiredBufferSizeForReverb(sampleRate, channelCountMax);
        auto buffer = m_Allocator.Allocate(bufferSize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(buffer);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddReverb(pConfig, &m_Reverb, buffer, bufferSize, pFinalMix, channelCountMax));
    }

    void Update() NN_OVERRIDE
    {
        // RandomizeParameters();
    }

    void RandomizeParameters()
    {
        std::uniform_int_distribution<int> earlyMode(static_cast<int>(nn::audio::ReverbType::EarlyMode_SmallRoom), static_cast<int>(nn::audio::ReverbType::EarlyMode_NoEarlyReflection));
        std::uniform_real_distribution<float> earlyGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> predelayTime(0.0f, 300.0f);
        std::uniform_int_distribution<int> lateMode(static_cast<int>(nn::audio::ReverbType::LateMode_Room), static_cast<int>(nn::audio::ReverbType::LateMode_MaximumDelay));
        std::uniform_real_distribution<float> lateGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> decayTimeSeconds(0.1f, 20.0f);
        std::uniform_real_distribution<float> highFreqDecayRatio(0.1f, 1.0f);
        std::uniform_real_distribution<float> coloration(0.0f, 1.0f);
        std::uniform_real_distribution<float> reverbGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> outGain(0.0f, 1.0f);
        std::uniform_real_distribution<float> dryGain(0.0f, 1.0f);

        const nn::audio::ReverbParameterSet parameterSet = {
            static_cast<nn::audio::ReverbType::EarlyMode>(earlyMode(m_RandomEngine)),
            earlyGain(m_RandomEngine),
            predelayTime(m_RandomEngine),
            static_cast<nn::audio::ReverbType::LateMode>(lateMode(m_RandomEngine)),
            lateGain(m_RandomEngine),
            decayTimeSeconds(m_RandomEngine),
            highFreqDecayRatio(m_RandomEngine),
            coloration(m_RandomEngine),
            reverbGain(m_RandomEngine),
            outGain(m_RandomEngine),
            dryGain(m_RandomEngine)
        };

        nn::audio::SetReverbParameters(&m_Reverb, &parameterSet);
    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetReverbEnabled(&m_Reverb, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        NN_ASSERT_MINMAX(channelCount, 0, EffectChannelCountMax);
        int8_t channels[EffectChannelCountMax];
        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        nn::audio::SetReverbInputOutput(&m_Reverb, channels, channels, channelCount);
    }

private:
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::ReverbType m_Reverb;
};

class Aux: public ITarget
{
public:
    Aux(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, const nn::audio::AudioRendererParameter& parameter, nn::audio::FinalMixType* pFinalMix, int mixBufferFrameCount, int channelCountMax)
        : m_Allocator(allocator),
          m_RandomEngine()
    {
        const auto sendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, mixBufferFrameCount, channelCountMax);
        void* sendBuffer = m_Allocator.Allocate(sendReturnBufferSize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(sendBuffer);
        void* returnBuffer = m_Allocator.Allocate(sendReturnBufferSize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(returnBuffer);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddAux(pConfig, &m_Aux, pFinalMix, sendBuffer, returnBuffer, sendReturnBufferSize));

        m_BufferSizeForReadAux = channelCountMax * parameter.sampleCount * sizeof(int32_t) * mixBufferFrameCount;
        m_BufferForReadAux = static_cast<int32_t*>(m_Allocator.Allocate(m_BufferSizeForReadAux));
        m_SampleCount = parameter.sampleCount;
        m_MixBufferFrameCount = mixBufferFrameCount;
    }

    void Update() NN_OVERRIDE
    {
        const auto readCount = nn::audio::ReadAuxSendBuffer(&m_Aux, m_BufferForReadAux, m_ChannelCount * m_SampleCount * m_MixBufferFrameCount);
        const auto writeCount = nn::audio::WriteAuxReturnBuffer(&m_Aux, m_BufferForReadAux, readCount);
        if(readCount != writeCount)
        {
            NN_LOG("Notice: Could not write back whole sample\n");
        }
    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetAuxEnabled(&m_Aux, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        // NN_ASSERT_MINMAX(channelCount, 0, EffectChannelCountMax);
        NN_ASSERT_MINMAX(channelCount, 0, nn::audio::MixBufferCountMax);
        int8_t channels[nn::audio::MixBufferCountMax];
        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        nn::audio::SetAuxInputOutput(&m_Aux, channels, channels, channelCount);

        m_ChannelCount = channelCount;
    }

private:
    int m_MixBufferFrameCount = 0;
    int m_SampleCount = 0;
    int m_ChannelCount = 0;
    int32_t* m_BufferForReadAux = nullptr;
    size_t m_BufferSizeForReadAux = 0u;
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::AuxType m_Aux;
};

class BiquadFilter: public ITarget
{
public:
    BiquadFilter(nn::audio::AudioRendererConfig* pConfig, nn::audio::FinalMixType* pFinalMix)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddBiquadFilter(pConfig, &m_BiquadFilter, pFinalMix));
    }

    void Update() NN_OVERRIDE
    {

    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetBiquadFilterEnabled(&m_BiquadFilter, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        NN_ASSERT_MINMAX(channelCount, 0, EffectChannelCountMax);
        int8_t input[EffectChannelCountMax];
        int8_t output[EffectChannelCountMax];

        std::iota(input, input + channelCount, static_cast<int8_t>(0));
        std::iota(output, output + channelCount, static_cast<int8_t>(0));

        // 入出力チャンネルが同じ状態だと無効化時にコピーしないので意図的にチャンネル番号を変える (1 チャンネルの時はあきらめる)
        if(channelCount > 1)
        {
            // 1, 2, ... , 0 のような並びにする
            std::rotate(output, output + 1, output + channelCount);
        }

        nn::audio::SetBiquadFilterInputOutput(&m_BiquadFilter, input, output, channelCount);
    }

private:
    nn::audio::BiquadFilterType m_BiquadFilter;
};

class BufferMixer: public ITarget
{
public:
    BufferMixer(nn::audio::AudioRendererConfig* pConfig, nn::audio::FinalMixType* pFinalMix)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddBufferMixer(pConfig, &m_BufferMixer, pFinalMix));
    }

    void Update() NN_OVERRIDE
    {

    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetBufferMixerEnabled(&m_BufferMixer, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        NN_ASSERT_MINMAX(channelCount, 0, nn::audio::MixBufferCountMax);
        int8_t channels[nn::audio::MixBufferCountMax];

        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        nn::audio::SetBufferMixerInputOutput(&m_BufferMixer, channels, channels, channelCount);
        for(const auto channel : channels)
        {
            nn::audio::SetBufferMixerVolume(&m_BufferMixer, channel, 0.5f);
        }
    }

private:
    nn::audio::BufferMixerType m_BufferMixer;
};

class I3dl2Reverb : public ITarget
{
public:
    I3dl2Reverb(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, nn::audio::FinalMixType* pFinalMix, int channelCountMax)
        : m_Allocator(allocator),
          m_RandomEngine()
    {
        const auto sampleRate = nn::audio::GetFinalMixSampleRate(pFinalMix);
        const auto bufferSize = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(sampleRate, channelCountMax);
        auto buffer = m_Allocator.Allocate(bufferSize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(buffer);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddI3dl2Reverb(pConfig, &m_I3dl2Reverb, buffer, bufferSize, pFinalMix, channelCountMax));
    }

    void Update() NN_OVERRIDE
    {
        // RandomizeParameters();
    }

    void RandomizeParameters()
    {
        std::uniform_real_distribution<float> roomGain(-10000.0f, 0.0f);
        std::uniform_real_distribution<float> roomHfGain(-10000.0f, 0.0f);
        std::uniform_real_distribution<float> lateReverbDecayTime(0.1f, 20.0f);
        std::uniform_real_distribution<float> lateReverbHfDecayRatio(0.1f, 2.0f);
        std::uniform_real_distribution<float> reflectionsGain(-10000.0f, 1000.0f);
        std::uniform_real_distribution<float> reflectionsDelayTime(0.0f, 0.3f);
        std::uniform_real_distribution<float> reverbGain(-10000.0f, 2000.0f);
        std::uniform_real_distribution<float> reverbDelayTime(0.0f, 0.1f);
        std::uniform_real_distribution<float> reverbDiffusion(0.1f, 100.0f);
        std::uniform_real_distribution<float> reverbDensity(0.1f, 100.0f);
        std::uniform_real_distribution<float> hfReference(20.0f, 20000.0f);
        std::uniform_real_distribution<float> dryGain(0.0f, 1.0f);

        const nn::audio::I3dl2ReverbParameterSet parameterSet = {
            roomGain(m_RandomEngine),
            roomHfGain(m_RandomEngine),
            lateReverbDecayTime(m_RandomEngine),
            lateReverbHfDecayRatio(m_RandomEngine),
            reflectionsGain(m_RandomEngine),
            reflectionsDelayTime(m_RandomEngine),
            reverbGain(m_RandomEngine),
            reverbDelayTime(m_RandomEngine),
            reverbDiffusion(m_RandomEngine),
            reverbDensity(m_RandomEngine),
            hfReference(m_RandomEngine),
            dryGain(m_RandomEngine)
        };

        nn::audio::SetI3dl2ReverbParameters(&m_I3dl2Reverb, &parameterSet);
    }

    void SetEnabled(bool isEnabled)
    {
        nn::audio::SetI3dl2ReverbEnabled(&m_I3dl2Reverb, isEnabled);
    }

    void SetChannelCount(int channelCount)
    {
        NN_ASSERT_MINMAX(channelCount, 0, EffectChannelCountMax);
        int8_t channels[EffectChannelCountMax];
        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        nn::audio::SetI3dl2ReverbInputOutput(&m_I3dl2Reverb, channels, channels, channelCount);
    }

private:
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::I3dl2ReverbType m_I3dl2Reverb;
};

class CircularBufferSink : public ITarget
{
public:
    CircularBufferSink(nn::mem::StandardAllocator& allocator, nn::audio::AudioRendererConfig* pConfig, const nn::audio::AudioRendererParameter& parameter, nn::audio::FinalMixType* pFinalMix, int channelCount)
        : m_Allocator(allocator),
          m_RandomEngine()
    {
        const int ChannelCountMax = nn::audio::MixBufferCountMax;
        NN_ASSERT_MINMAX(channelCount, 0, ChannelCountMax);
        int8_t channels[ChannelCountMax];
        std::iota(channels, channels + channelCount, static_cast<int8_t>(0));

        const auto circularBufferSinkFrameCount = 10;
        const auto format = nn::audio::SampleFormat_PcmInt16;
        const auto bufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&parameter, channelCount, circularBufferSinkFrameCount, format);
        auto buffer = m_Allocator.Allocate(bufferSize, nn::audio::BufferAlignSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(pConfig, &m_CircularBufferSink, pFinalMix, channels, channelCount, buffer, bufferSize, format));

        m_BufferSizeForReadSink = parameter.sampleCount * sizeof(int16_t) * channelCount * circularBufferSinkFrameCount;
        m_BufferForReadSink = m_Allocator.Allocate(m_BufferSizeForReadSink);
    }

    void Update() NN_OVERRIDE
    {
        nn::audio::ReadCircularBufferSink(&m_CircularBufferSink, m_BufferForReadSink, m_BufferSizeForReadSink);
    }

    nn::audio::CircularBufferSinkType* GetBase()
    {
        return &m_CircularBufferSink;
    }

private:
    nn::mem::StandardAllocator& m_Allocator;
    std::mt19937 m_RandomEngine;
    nn::audio::CircularBufferSinkType m_CircularBufferSink;
    void* m_BufferForReadSink;
    size_t m_BufferSizeForReadSink;
};

class Measurer
{
public:
    // TODO: コンストラクタを整理する
    template<typename NodeType>
    Measurer(const std::string propertyText, nn::audio::AudioRendererConfig* config, NodeType* node, ITarget* target, nn::audio::PerformanceSysDetailType targetSysDetail, int measureCount)
           : m_PropertyText(propertyText),
             m_Target(target),
             m_SysDetail(targetSysDetail),
             m_MeasureCount(measureCount)
    {
        nn::audio::SetPerformanceDetailTarget(config, node);
    }

    Measurer(const std::string propertyText, nn::audio::AudioRendererConfig* config, ITarget* target, nn::audio::PerformanceSysDetailType targetSysDetail, int measureCount)
           : m_PropertyText(propertyText),
             m_Target(target),
             m_SysDetail(targetSysDetail),
             m_MeasureCount(measureCount)
    {
        nn::audio::ClearPerformanceDetailTarget(config);
    }

    void Update(const nn::audio::PerformanceInfo& info)
    {
        if(m_Target)
        {
            m_Target->Update();
        }

        int detailCount = 0;
        const auto details = info.GetDetails(&detailCount);
        for(int i = 0; i < detailCount; ++i)
        {
            const auto sysDetail = reinterpret_cast<const nn::audio::PerformanceSysDetail*>(&details[i]);

            if(sysDetail->detailType == m_SysDetail)
            {
                m_ResultSet.Add(sysDetail->processingTime, sysDetail->estimatedProcessingTime);
            }

            if(m_UpdateCount % 200 == 0)
            {
                if(static_cast<nn::audio::PerformanceSysDetailType>(sysDetail->detailType) != nn::audio::PerformanceSysDetailType_Performance)
                {
                    // NN_LOG("[%s] \n", ToString(static_cast<nn::audio::PerformanceSysDetailType>(sysDetail->detailType)));
                }
            }
        }

        ++m_UpdateCount;
    }

    void PrintResult()
    {
        const auto measured = m_ResultSet.GetMeasuredStatResult();
        const auto estimated = m_ResultSet.GetEstimatedStatResult();

#if 0
        const auto error2 = m_ResultSet.GetError2StatResult();
        NN_LOG("[%s][%s] (min, max, ave, stddiv) ", ToString(m_SysDetail), m_PropertyText.c_str());
        NN_LOG("Mea:(%.2f, %.2f, %.2f, %.2f) ",
                measured.min,
                measured.max,
                measured.ave,
                measured.stddiv);

        NN_LOG("est:(%.2f, %.2f, %.2f, %.2f) ",
                estimated.min,
                estimated.max,
                estimated.ave,
                estimated.stddiv);

        NN_LOG("err2:(%.2f, %.2f, %.2f, %.2f) ",
                error2.min,
                error2.max,
                error2.ave,
                error2.stddiv);
#endif
        NN_LOG("Result, %20s, %15s, ", ToString(m_SysDetail), m_PropertyText.c_str());

        NN_LOG("%10.2f, %10.2f, %10.2f, %10.2f, %10.2f, %10.2f, %10.2f",
                measured.min,
                measured.max,
                measured.ave,
                estimated.max,
                estimated.max - measured.ave,
                (estimated.max - measured.ave) / 2880000 * 100,
                measured.stddiv
        );

        NN_LOG("\n");
    }

    bool IsCompleted()
    {
        return m_MeasureCount < m_ResultSet.GetMeasuredStatResult().count;
    }

private:
    int m_UpdateCount = 0;
    const std::string m_PropertyText;
    ITarget* m_Target;
    PerfResultSet m_ResultSet{};
    nn::audio::PerformanceSysDetailType m_SysDetail;
    int m_MeasureCount;
};

class Tester
{
public:
    Tester(void* workBuffer, size_t workBufferSize, int sampleRate, int mixBufferCount, int circularBufferSinkChannelCount)
    {
        m_Allocator.Initialize(workBuffer, workBufferSize);
        InitializeAudioRenderer(sampleRate, mixBufferCount, circularBufferSinkChannelCount);
    }

    ~Tester()
    {
        FinalizeAudioRenderer();
    }

    void InitializeAudioRenderer(int sampleRate, int mixBufferCount, int circularBufferSinkChannelCount)
    {
        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.sampleRate = sampleRate;
        parameter.sampleCount = sampleRate / 200;
        parameter.mixBufferCount = mixBufferCount;
        parameter.sinkCount = VoiceCount;
        parameter.subMixCount = SubMixCount;
        parameter.voiceCount = SinkCount;
        parameter.effectCount = EffectCount;
        parameter.performanceFrameCount = PerformanceFrameCount;

        const auto workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ASSERT_NOT_NULL(m_WorkBuffer);


        const auto configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        NN_ASSERT_NOT_NULL(m_ConfigBuffer);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        const auto finalMixBufferCount = nn::audio::MixBufferCountMax;
        const auto resultForAcquireFinalMix = nn::audio::AcquireFinalMix(&m_Config, &m_FinalMix, finalMixBufferCount);
        EXPECT_TRUE(resultForAcquireFinalMix);

        // Grouped コマンドを発行させないための SubMix なので 1 を指定しておく
        const auto subMixBufferCount = 1;
        const auto resultForAcquireSubMix = nn::audio::AcquireSubMix(&m_Config, &m_SubMix, sampleRate, subMixBufferCount);
        EXPECT_TRUE(resultForAcquireSubMix);
        nn::audio::SetSubMixDestination(&m_Config, &m_SubMix, &m_FinalMix);
        // MEMO: 必要ならば SetSubMixMixVolume() も呼び出す
        nn::audio::SetSubMixVolume(&m_SubMix, 1.0f);

        // Downmix コマンドを発行させるため
        const auto deviceSinkChannelCount = 6;
        nn::audio::DeviceSinkType deviceSink;
        int8_t mainBus[deviceSinkChannelCount];
        std::iota(mainBus, mainBus + deviceSinkChannelCount, static_cast<int8_t>(0));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&m_Config, &deviceSink, &m_FinalMix, mainBus, deviceSinkChannelCount, "MainAudioOut"));
        nn::audio::SetDownMixParameterEnabled(&deviceSink, true);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, &m_SystemEvent, parameter, m_WorkBuffer, workBufferSize));

        auto workBufferForAudioAllocator = m_Allocator.Allocate(WorkBufferSizeForAudioAllocator, nn::os::MemoryPageSize);
        ASSERT_NE(workBufferForAudioAllocator, nullptr);
        m_AllocatorForAudio.Initialize(workBufferForAudioAllocator, WorkBufferSizeForAudioAllocator);
        const auto resultForAcquireMemoryPool = nn::audio::AcquireMemoryPool(&m_Config, &m_MemoryPool, workBufferForAudioAllocator, WorkBufferSizeForAudioAllocator);
        ASSERT_TRUE(resultForAcquireMemoryPool);
        nn::audio::RequestAttachMemoryPool(&m_MemoryPool);

        m_Delay = std::make_unique<Delay>(m_AllocatorForAudio, &m_Config, &m_FinalMix, 6, nn::TimeSpan::FromSeconds(1));
        m_Reverb = std::make_unique<Reverb>(m_AllocatorForAudio, &m_Config, &m_FinalMix, 6);
        m_I3dl2Reverb = std::make_unique<I3dl2Reverb>(m_AllocatorForAudio, &m_Config, &m_FinalMix, 6);
        m_Aux = std::make_unique<Aux>(m_AllocatorForAudio, &m_Config, parameter, &m_FinalMix, 10, 6);
        m_BiquadFilter = std::make_unique<BiquadFilter>(&m_Config, &m_FinalMix);
        m_BufferMixer = std::make_unique<BufferMixer>(&m_Config, &m_FinalMix);

        const auto pcmInt16DummyVoiceCount = 40;
        for(int i = 0; i < pcmInt16DummyVoiceCount; ++i)
        {
            auto voice = std::make_unique<Voice>(m_AllocatorForAudio, &m_Config, sampleRate, nn::audio::SampleFormat_PcmInt16);
            voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
            m_DummyPcmInt16Voices.push_back(std::move(voice));
        }

        const auto adpcmDummyVoiceCount = 40;
        for(int i = 0; i < adpcmDummyVoiceCount; ++i)
        {
            auto voice = std::make_unique<Voice>(m_AllocatorForAudio, &m_Config, sampleRate, nn::audio::SampleFormat_Adpcm);
            voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
            m_DummyAdpcmVoices.push_back(std::move(voice));
        }

        m_PcmInt16Voice = std::make_unique<Voice>(m_AllocatorForAudio, &m_Config, sampleRate / 2, nn::audio::SampleFormat_PcmInt16);
        m_AdpcmVoice = std::make_unique<Voice>(m_AllocatorForAudio, &m_Config, sampleRate / 2, nn::audio::SampleFormat_Adpcm);

        m_CircularBufferSink = std::make_unique<CircularBufferSink>(m_AllocatorForAudio, &m_Config, parameter, &m_FinalMix, circularBufferSinkChannelCount);

        m_PerfBufferSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
        for(auto& perfBuffer : m_PerfBuffer)
        {
            perfBuffer = m_Allocator.Allocate(m_PerfBufferSize);
            ASSERT_NE(perfBuffer, nullptr);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));
        nn::audio::StartAudioRenderer(m_Handle);
    }

    void PerformMeasurement(Measurer* measurer)
    {
        const auto perfMeasureSkipFrameCount = 10;

        int frameCount = 0;
        for(;;)
        {
            auto lastBuffer = nn::audio::SetPerformanceFrameBuffer(&m_Config, m_PerfBuffer[m_PerfBufferIndex], m_PerfBufferSize);
            m_PerfBufferIndex = (m_PerfBufferIndex + 1) % PerfBufferCount;

            nn::audio::PerformanceInfo performanceInfo;
            if (lastBuffer != nullptr && performanceInfo.SetBuffer(lastBuffer, m_PerfBufferSize) && frameCount > perfMeasureSkipFrameCount)
            {
                measurer->Update(performanceInfo);
            }

            bool isSignaled = m_SystemEvent.TimedWait(nn::TimeSpan::FromSeconds(1));
            ASSERT_TRUE(isSignaled);

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));

            if(measurer->IsCompleted())
            {
                measurer->PrintResult();
                break;
            }

            ++frameCount;
        }
    }

    void PerformMeasurementForDelay(bool isEnabled, int channelCount)
    {
        m_Delay->SetEnabled(isEnabled);
        m_Delay->SetChannelCount(channelCount);

        const auto text = std::string(isEnabled ? "enabled " : "disabled ") + std::to_string(channelCount);

        auto measurer = std::make_unique<Measurer>(text, &m_Config, &m_FinalMix, m_Delay.get(), nn::audio::PerformanceSysDetailType_Delay, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForReverb(bool isEnabled, int channelCount)
    {
        m_Reverb->SetEnabled(isEnabled);
        m_Reverb->SetChannelCount(channelCount);

        const auto text = std::string(isEnabled ? "enabled " : "disabled ") + std::to_string(channelCount);

        auto measurer = std::make_unique<Measurer>(text, &m_Config, &m_FinalMix, m_Reverb.get(), nn::audio::PerformanceSysDetailType_Reverb, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForI3dl2Reverb(bool isEnabled, int channelCount)
    {
        m_I3dl2Reverb->SetEnabled(isEnabled);
        m_I3dl2Reverb->SetChannelCount(channelCount);

        const auto text = std::string(isEnabled ? "enabled " : "disabled ") + std::to_string(channelCount);

        auto measurer = std::make_unique<Measurer>(text, &m_Config, &m_FinalMix, m_I3dl2Reverb.get(), nn::audio::PerformanceSysDetailType_I3dl2Reverb, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForAux(bool isEnabled, int channelCount)
    {
        m_Aux->SetEnabled(isEnabled);
        m_Aux->SetChannelCount(channelCount);

        const auto text = std::string(isEnabled ? "enabled " : "disabled ") + std::to_string(channelCount);

        auto measurer = std::make_unique<Measurer>(text, &m_Config, &m_FinalMix, m_Aux.get(), nn::audio::PerformanceSysDetailType_Aux, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForDepopForMixBuffers()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, &m_FinalMix, nullptr, nn::audio::PerformanceSysDetailType_DepopForMixBuffers, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForPerformance()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, &m_FinalMix, nullptr, nn::audio::PerformanceSysDetailType_Performance, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForVolume()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, &m_FinalMix, nullptr, nn::audio::PerformanceSysDetailType_Volume, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForPcmInt16DataSource(float pitch)
    {
        m_PcmInt16Voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
        m_PcmInt16Voice->SetPitch(pitch);
        const auto text = std::to_string(pitch);
        auto measurer = std::make_unique<Measurer>(text, &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_PcmInt16DataSource, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForAdpcmDataSource(float pitch)
    {
        m_AdpcmVoice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
        m_AdpcmVoice->SetPitch(pitch);
        const auto text = std::to_string(pitch);
        auto measurer = std::make_unique<Measurer>(text, &m_Config, m_AdpcmVoice->GetBase(), m_AdpcmVoice.get(), nn::audio::PerformanceSysDetailType_AdpcmDataSource, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForBiquadFilter()
    {
        m_PcmInt16Voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
        m_PcmInt16Voice->SetBiquadFilterEnabled(true);
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_BiquadFilter, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForVolumeRamp()
    {
        m_PcmInt16Voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_VolumeRamp, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForMixRamp()
    {
        // Grouped の方にならないように 1 を指定
        m_PcmInt16Voice->SetMixVolume(&m_SubMix, 0.5f, 1);
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_MixRamp, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForMixRampGrouped()
    {
        // Grouped の方になるように最大チャンネルを指定
        m_PcmInt16Voice->SetMixVolume(&m_FinalMix, 0.5f, nn::audio::MixBufferCountMax);
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_MixRampGrouped, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForCopyMixBuffer()
    {
        m_BiquadFilter->SetChannelCount(nn::audio::BiquadFilterType::ChannelCountMax);
        m_BiquadFilter->SetEnabled(false);
        auto measurer = std::make_unique<Measurer>("", &m_Config, &m_FinalMix, nullptr, nn::audio::PerformanceSysDetailType_CopyMixBuffer, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForMix()
    {
        m_BufferMixer->SetChannelCount(nn::audio::MixBufferCountMax);
        m_BufferMixer->SetEnabled(true);
        auto measurer = std::make_unique<Measurer>("", &m_Config, &m_FinalMix, nullptr, nn::audio::PerformanceSysDetailType_Mix, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForClearMixBuffer()
    {
        const auto mixBufferCount = nn::audio::GetAudioRendererMixBufferCount(m_Handle);
        const auto text = std::to_string(mixBufferCount);
        auto measurer = std::make_unique<Measurer>(text, &m_Config, nullptr, nn::audio::PerformanceSysDetailType_ClearMixBuffer, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForDownMix6chTo2ch()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, nullptr, nn::audio::PerformanceSysDetailType_DownMix6chTo2ch, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForCircularBufferSink()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_CircularBufferSink.get(), nn::audio::PerformanceSysDetailType_CircularBufferSink, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForDepopPrepare()
    {
        m_PcmInt16Voice->SetMixVolume(&m_FinalMix, 1.0f, nn::audio::MixBufferCountMax);
        auto measurer = std::make_unique<Measurer>("", &m_Config, m_PcmInt16Voice->GetBase(), m_PcmInt16Voice.get(), nn::audio::PerformanceSysDetailType_DepopPrepare, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForUpsample()
    {
        auto measurer = std::make_unique<Measurer>("", &m_Config, nullptr, nn::audio::PerformanceSysDetailType_Upsample, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void PerformMeasurementForDeviceSink()
    {
#if defined(NN_BUILD_CONFIG_SPEC_NX) && (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
        const auto channelCount = nn::audio::GetActiveAudioDeviceChannelCountForOutput();
#else
        const auto channelCount = 6;
#endif
        const std::string text = std::to_string(channelCount);
        auto measurer = std::make_unique<Measurer>(text, &m_Config, nullptr, nn::audio::PerformanceSysDetailType_DeviceSink, MeasureCount);
        PerformMeasurement(measurer.get());
    }

    void FinalizeAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);
    }

private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::audio::MemoryPoolType m_MemoryPool;
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::SubMixType m_SubMix;
    nn::os::SystemEvent m_SystemEvent;
    nn::mem::StandardAllocator m_Allocator;
    nn::mem::StandardAllocator m_AllocatorForAudio;

    static const int PerfBufferCount = 4;
    void* m_PerfBuffer[PerfBufferCount] = { nullptr };
    int m_PerfBufferIndex = 0;
    size_t m_PerfBufferSize = 0u;

    std::unique_ptr<Delay> m_Delay;
    std::unique_ptr<Reverb> m_Reverb;
    std::unique_ptr<I3dl2Reverb> m_I3dl2Reverb;
    std::unique_ptr<Aux> m_Aux;
    std::unique_ptr<BiquadFilter> m_BiquadFilter;
    std::unique_ptr<BufferMixer> m_BufferMixer;
    std::vector<std::unique_ptr<Voice>> m_DummyPcmInt16Voices;
    std::vector<std::unique_ptr<Voice>> m_DummyAdpcmVoices;
    std::unique_ptr<Voice> m_PcmInt16Voice;
    std::unique_ptr<Voice> m_AdpcmVoice;
    std::unique_ptr<CircularBufferSink> m_CircularBufferSink;

    static const int VoiceCount = 100;
    static const int SinkCount = 100;
    static const int EffectCount = 100;
    static const int SubMixCount = 100;
    static const int PerformanceFrameCount = 100;
    static const size_t WorkBufferSizeForAudioAllocator = 1024 * 1024 * 10;
    static const int MeasureCount = 200;
};

nn::audioctrl::AudioOutputMode ToOutputMode(int channelCount)
{
    switch(channelCount)
    {
    case 1:
        return nn::audioctrl::AudioOutputMode_Pcm1ch;
    case 2:
        return nn::audioctrl::AudioOutputMode_Pcm2ch;
    case 6:
        return nn::audioctrl::AudioOutputMode_Pcm6ch;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#if defined(NN_BUILD_CONFIG_SPEC_NX) && (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
bool SetOutputDeviceChannelCount(int channelCount)
{
    NN_SDK_ASSERT(channelCount == 1 || channelCount == 2 || channelCount == 6);

    // 強制的に TV 出力設定にする
    nn::audioctrl::SetOutputTargetForProduction(nn::audioctrl::AudioTarget_Tv);

    nn::audioctrl::SetAudioOutputMode(nn::audioctrl::AudioTarget_Tv, ToOutputMode(channelCount));

    const auto retryCountMax = 10;
    for(auto retryCount = 0; retryCount < retryCountMax; ++retryCount)
    {
        if(nn::audio::GetActiveAudioDeviceChannelCountForOutput() == channelCount)
        {
            return true;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    return false;
}
#endif

} // namespace anonymous

TEST(VoiceDropPrecision, Measure)
{
    // テスト中は強制的に TV 出力にする
    const auto originalOutputMode = nn::audioctrl::GetAudioOutputMode(nn::audioctrl::AudioTarget_Tv);

    // テストで変更した設定を元に戻す
    NN_UTIL_SCOPE_EXIT
    {
        nn::audioctrl::SetAudioOutputMode(nn::audioctrl::AudioTarget_Tv, originalOutputMode);
        nn::audioctrl::SetOutputTargetForProduction(nn::audioctrl::AudioTarget_Invalid);
    };

    const size_t bufferSize = 1024 * 1024 * 128;
    auto buffer = std::make_unique<uint8_t[]>(bufferSize);

    for(const auto sampleRate : { 32000, 48000 })
    {
        // サンプルレートのみ
        {
            Tester tester(buffer.get(), bufferSize, sampleRate, 100, nn::audio::MixBufferCountMax);

            // Delay
            for(const auto isEnabled : { false, true })
            {
                for(const auto channelCount : { 1, 2, 4, 6 })
                {
                    tester.PerformMeasurementForDelay(isEnabled, channelCount);
                }
            }

            // Reverb
            for(const auto isEnabled : { false, true })
            {
                for(const auto channelCount : { 1, 2, 4, 6 })
                {
                    tester.PerformMeasurementForReverb(isEnabled, channelCount);
                }
            }

            // I3dl2Reverb
            for(const auto isEnabled : { false, true })
            {
                for(const auto channelCount : { 1, 2, 4, 6 })
                {
                    tester.PerformMeasurementForI3dl2Reverb(isEnabled, channelCount);
                }
            }

            // Aux
            for(const auto isEnabled : { false, true })
            {
                const auto channelCount = nn::audio::MixBufferCountMax;
                tester.PerformMeasurementForAux(isEnabled, channelCount);
            }

            tester.PerformMeasurementForDepopForMixBuffers();
            tester.PerformMeasurementForPerformance();
            tester.PerformMeasurementForVolume();
            tester.PerformMeasurementForBiquadFilter();
            tester.PerformMeasurementForCopyMixBuffer();
            tester.PerformMeasurementForDepopPrepare();
            tester.PerformMeasurementForVolumeRamp();
            tester.PerformMeasurementForMixRampGrouped();
            tester.PerformMeasurementForMixRamp();
            tester.PerformMeasurementForMix();

#if defined(NN_BUILD_CONFIG_SPEC_NX) && (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
            // 出力先を強制的に 2ch TV にしてテストを行う
            ASSERT_TRUE(SetOutputDeviceChannelCount(2));
            tester.PerformMeasurementForDownMix6chTo2ch();
#endif

            if(sampleRate == 32000)
            {
                tester.PerformMeasurementForUpsample();
            }
            else
            {
                // TORIAEZU: ダミーのメッセージを出しておく
                NN_LOG("Result, 0, 0, 0, 0, 0, 0, 0\n");
            }

            const auto pitchStep = 1.0f;
            for(float pitch = 1.0f; pitch <= nn::audio::VoiceType::GetPitchMax(); pitch += pitchStep)
            {
                tester.PerformMeasurementForPcmInt16DataSource(pitch);
            }

            for(float pitch = 1.0f; pitch <= nn::audio::VoiceType::GetPitchMax(); pitch += pitchStep)
            {
                tester.PerformMeasurementForAdpcmDataSource(pitch);
            }
        }

        // DeviceSink の出力チャンネル数に依存するコマンド
        for(const auto channelCount : { 2, 6 })
        {
#if defined(NN_BUILD_CONFIG_SPEC_NX) && (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
            ASSERT_TRUE(SetOutputDeviceChannelCount(channelCount));
#endif
            Tester tester(buffer.get(), bufferSize, sampleRate, 100, nn::audio::MixBufferCountMax);
            tester.PerformMeasurementForDeviceSink();
        }

        // MixBufferCount に依存するコマンド
        for(int mixBufferCount = 50; mixBufferCount < 100; mixBufferCount += 10)
        {
            Tester tester(buffer.get(), bufferSize, sampleRate, mixBufferCount, nn::audio::MixBufferCountMax);
            tester.PerformMeasurementForClearMixBuffer();
        }

        // CircularBufferSinkChannelCount に依存するコマンド
        for(int channelCount = 1; channelCount <= nn::audio::MixBufferCountMax; ++channelCount)
        {
            Tester tester(buffer.get(), bufferSize, sampleRate, 100, channelCount);
            tester.PerformMeasurementForCircularBufferSink();
        }
    }
}
#endif
