﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <cstdlib>    // std::aligned_alloc
#include <cstring>    // std::memcmp
#include <memory>
#include <string>

#include <nnt.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#include <nn/audio.h>
#include <nn/mem.h>

#include "../../../../Programs/Eris/Sources/Libraries/audio/common/audio_SortLogic.h"

namespace
{
struct StdAllocatorDeleter
{
    static void SetupDeleter(nn::mem::StandardAllocator* pAllocator)
    {
        m_Allocator = pAllocator;
    }
    static nn::mem::StandardAllocator* m_Allocator;
    void operator()(void* ptr) const
    {
        NN_ABORT_UNLESS_NOT_NULL(m_Allocator);
        m_Allocator->Free(ptr);
    }
};
nn::mem::StandardAllocator* StdAllocatorDeleter::m_Allocator = nullptr;
using StdUniquePtr = std::unique_ptr<void, StdAllocatorDeleter>;
}

TEST(DFS, ZeroElement)
{
    nn::audio::common::EdgeMatrix edges;
    nn::audio::common::NodeStates states;
    EXPECT_TRUE(states.Tsort(edges));
    for (auto itr = states.ResultBegin(); itr != states.ResultEnd(); ++itr)
    {
        ASSERT_TRUE(false);
    }
}

TEST(DFS, Success)
{
    static int8_t workBuffer[1024 * 1024];
    nn::mem::StandardAllocator allocator(workBuffer, 1024 * 1024);
    StdAllocatorDeleter::SetupDeleter(&allocator);
    {
        const int nodeCount = 7;
        nn::audio::common::EdgeMatrix edges;
        auto bufSize = nn::audio::common::EdgeMatrix::GetWorkBufferSize(nodeCount);
        StdUniquePtr buf(allocator.Allocate(bufSize, 1));
        edges.Initialize(buf.get(), bufSize, nodeCount);

        nn::audio::common::NodeStates states;
        auto statesSize = nn::audio::common::NodeStates::GetWorkBufferSize(nodeCount);
        StdUniquePtr statesBuf(allocator.Allocate(statesSize, 1));
        states.Initialize(statesBuf.get(), statesSize, nodeCount);

        edges.Connect(0, 1);
        edges.Connect(0, 3);
        edges.Connect(1, 4);
        edges.Connect(2, 5);
        edges.Connect(2, 4);
        edges.Connect(3, 1);
        // node 6 has no connection.

        EXPECT_TRUE(states.Tsort(edges));
        const int answer[] = { 6, 2, 5, 0, 3, 1, 4 };
        int index = 0;
        for (auto itr = states.ResultBegin(); itr != states.ResultEnd(); ++itr)
        {
            EXPECT_EQ(answer[index++], *itr);
        }
        std::cout << std::endl;

        // add loop path
        edges.Connect(4, 3);
        EXPECT_FALSE(states.Tsort(edges));
    }
    allocator.Finalize();
}
