﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm> // std::max, std::min
#include <nnt.h>
#include <cstdlib>
#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/audio.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/util/util_BitUtil.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>
#include "../../../Programs/Eris/Sources/Libraries/audio/common/audio_Util.h" // common::AudioOutInitializedMagic
#include <nn/nn_Abort.h>
#define NN_TEST_AUDIO_EXPECT_RANGE(value, begin, end)       \
    {                                                       \
        auto nn_test_audio_expect_range_value = (value);    \
        EXPECT_GE(nn_test_audio_expect_range_value, begin); \
        EXPECT_LT(nn_test_audio_expect_range_value, end);   \
    }
#include <nn/nn_Assert.h>
#include <cmath>
#include <nn/nn_SdkLog.h>
namespace {
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_Buffer[1024 * 1024];
static const int QF_FRACTIONAL_BIT_COUNT = 14;
const std::string g_AudioTestSourceFileName = "/shutter.wav";
const std::string g_AudioTestI3dl2Reverb1chFileName = "/testAudio_Effect_I3dl2Reverb1chShutter.wav";
const std::string g_AudioTestI3dl2Reverb2chFileName = "/testAudio_Effect_I3dl2Reverb2chShutter.wav";
const std::string g_AudioTestI3dl2Reverb6chFileNameBroken[] = { "/testAudio_Effect_I3dl2Reverb6chShutterA.wav",
    "/testAudio_Effect_I3dl2Reverb6chShutterB.wav", "/testAudio_Effect_I3dl2Reverb6chShutterC.wav" };
const std::string g_AudioTestI3dl2Reverb4chFileName = "/testAudio_Effect_I3dl2Reverb4chShutter.wav";
const std::string g_AudioTestI3dl2Reverb6chFileName = "/testAudio_Effect_I3dl2Reverb6chShutter.wav";
const float g_AudioTestQfPrecision = 1.0f / std::powf(2, static_cast<float>(QF_FRACTIONAL_BIT_COUNT));
nn::mem::StandardAllocator g_Allocator;
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[14 * 1024 * 1024];
const std::string g_AudioTestAuxReverb1chFileName = "/testAudio_Effect_Reverb1chShutter.wav";
const std::string g_AudioTestAuxReverb2chFileName = "/testAudio_Effect_Reverb2chShutter.wav";
const std::string g_AudioTestAuxReverb6chFileName ="/testAudio_Effect_Reverb6chShutter.wav";
const std::string g_AudioTestDelayed1chFileName = "/testAudio_Effect_Delay1ch.wav";
const std::string g_AudioTestDelayed2chFileName[] = {"/testAudio_Effect_Delay2chFinalMix.wav", "/testAudio_Effect_Delay2chSubMix.wav"};
const std::string g_AudioTestDelayed4chFileName[] = {"/testAudio_Effect_Delay4chFinalMix.wav", "/testAudio_Effect_Delay4chSubMix.wav"};
const std::string g_AudioTestDelayed6chFileName[] = {"/testAudio_Effect_Delay6chFinalMix.wav", "/testAudio_Effect_Delay6chSubMix.wav"};
}

/**
 * @brief       InitializeAuxI3dl2Reverb(AuxI3dl2ReverbType* pReverb, void* buffer, size_t bufferSize, int sampleRate, int channelCount) NN_NOEXCEPT
 */
TEST(InitializeAuxI3dl2Reverb, Success)
{
    nn::audio::AuxI3dl2ReverbType reverb;
    int sampleRate[2] = {32000, 48000};
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(48000, 6);
    int channelCount[4] = {1, 2, 4, 6};

    for(int i = 0; i < 4; ++i)
    {
        for(int j = 0; j < 2; ++j)
        {
            bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(sampleRate[j], channelCount[i]);
            EXPECT_NE(reverb._bufferSize, bufferSize);
            nn::audio::InitializeAuxI3dl2Reverb(&reverb, g_Buffer, bufferSize, sampleRate[j], channelCount[i]);
            EXPECT_GE(reverb._bufferSize, bufferSize);
        }
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(InitializeAuxI3dl2Reverb, PreCondition)
{
    nn::audio::AuxI3dl2ReverbType reverb;
    int sampleRate[2] = {32000, 48000};
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(48000, 6);
    int channelCount[4] = {1, 2, 4, 6};

    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxI3dl2Reverb(nullptr, g_Buffer, bufferSize, sampleRate[1], channelCount[3]), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxI3dl2Reverb(&reverb, nullptr, bufferSize, sampleRate[1], channelCount[3]), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxI3dl2Reverb(&reverb, g_Buffer, 0, sampleRate[1], channelCount[3]), "");
}
#endif

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAuxI3dl2ReverbChannelCountMax, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAuxI3dl2ReverbChannelCountMax(nullptr), "");
}
#endif

TEST(GetAuxI3dl2ReverbChannelCountMax, Success)
{
    nn::audio::AuxI3dl2ReverbType reverb;
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(48000, 6);
    const int channelCounts[] = { 1, 2, 4, 6 };
    const auto sampleCount = 48000;

    for(const auto channelCount : channelCounts)
    {
        nn::audio::InitializeAuxI3dl2Reverb(&reverb, g_Buffer, bufferSize, sampleCount, channelCount);
        EXPECT_EQ(channelCount, nn::audio::GetAuxI3dl2ReverbChannelCountMax(&reverb));
    }
}

/**
 * @brief       SetAuxI3dl2ReverbInputOutput(AuxI3dl2ReverbType* pReverb, const int8_t* input, const int8_t* outputs, int count) / GetAuxI3dl2ReverbInputOutput()
 */
TEST(SetAuxI3dl2ReverbInputOutput, Success)
{
    int8_t auxSubBus[2];
    int8_t finalBus[2];
    auxSubBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBus[nn::audio::ChannelMapping_FrontRight] = 1;
    finalBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    finalBus[nn::audio::ChannelMapping_FrontRight] = 1;
    int sampleRate = 32000;
    int channelCountMax = 2;
    nn::audio::AuxI3dl2ReverbType reverb;
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(sampleRate, channelCountMax);
    nn::audio::InitializeAuxI3dl2Reverb(&reverb, g_Buffer, bufferSize, sampleRate, channelCountMax);
    int8_t currentBusIn[2] = {3,4};
    int8_t currentBusOut[2] = {5,6};
    int currentOutCount = 0;
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_NE(currentBusIn[i],auxSubBus[i]);
        EXPECT_NE(currentBusOut[i], finalBus[i]);
    }
    EXPECT_NE(currentOutCount, channelCountMax);

    nn::audio::SetAuxI3dl2ReverbInputOutput(&reverb, auxSubBus, finalBus, channelCountMax);
    nn::audio::GetAuxI3dl2ReverbInputOutput(currentBusIn, currentBusOut, &currentOutCount, &reverb, 6);
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_EQ(auxSubBus[i], currentBusIn[i]);
        EXPECT_EQ(finalBus[i], currentBusOut[i]);
    }
    EXPECT_EQ(currentOutCount, channelCountMax);
}


/**
 * @brief       SetAuxI3dl2ReverbParameters() / GetAuxI3dl2ReverbParameters()
 */
TEST(SetAuxI3dl2ReverbParameters, Success)
{
    int channelCountMax = 2;
    int sampleRate = 48000;
    nn::audio::AuxI3dl2ReverbType reverb;
    nn::audio::I3dl2ReverbParameterSet paramSetOne;
    nn::audio::I3dl2ReverbParameterSet paramSetTwo;
    paramSetOne.roomHfGain = 0.0f;
    paramSetOne.hfReference = 0.0f;
    paramSetOne.lateReverbDecayTime = 0.0f;
    paramSetOne.lateReverbHfDecayRatio = 0.0f;
    paramSetOne.roomGain = 0.0f;
    paramSetOne.reflectionsGain = 0.0f;
    paramSetOne.reverbGain = 0.0f;
    paramSetOne.reverbDiffusion = 0.0f;
    paramSetOne.reflectionsDelayTime = 0.0f;
    paramSetOne.reverbDelayTime = 0.0f;
    paramSetOne.reverbDensity = 0.0f;
    paramSetOne.dryGain = 0.0f;

    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(sampleRate, channelCountMax);
    nn::audio::InitializeAuxI3dl2Reverb(&reverb, g_Buffer, bufferSize, sampleRate, channelCountMax);

    nn::audio::LoadI3dl2ReverbPreset(&paramSetTwo, nn::audio::I3dl2ReverbType::Preset_SmallRoom);
    EXPECT_NE(paramSetOne.roomHfGain, paramSetTwo.roomHfGain);
    EXPECT_NE(paramSetOne.hfReference, paramSetTwo.hfReference);
    EXPECT_NE(paramSetOne.lateReverbDecayTime, paramSetTwo.lateReverbDecayTime);
    EXPECT_NE(paramSetOne.lateReverbHfDecayRatio, paramSetTwo.lateReverbHfDecayRatio);
    EXPECT_NE(paramSetOne.roomGain, paramSetTwo.roomGain);
    EXPECT_NE(paramSetOne.reflectionsGain, paramSetTwo.reflectionsGain);
    EXPECT_NE(paramSetOne.reverbGain, paramSetTwo.reverbGain);
    EXPECT_NE(paramSetOne.reverbDiffusion, paramSetTwo.reverbDiffusion);
    EXPECT_NE(paramSetOne.reflectionsDelayTime, paramSetTwo.reflectionsDelayTime);
    EXPECT_NE(paramSetOne.reverbDelayTime, paramSetTwo.reverbDelayTime);
    EXPECT_NE(paramSetOne.reverbDensity, paramSetTwo.reverbDensity);
    EXPECT_NE(paramSetOne.dryGain, paramSetTwo.dryGain);

    nn::audio::SetAuxI3dl2ReverbParameters(&reverb, &paramSetTwo);
    paramSetOne = nn::audio::GetAuxI3dl2ReverbParameters(&reverb);

    EXPECT_EQ(paramSetOne.roomHfGain, paramSetTwo.roomHfGain);
    EXPECT_EQ(paramSetOne.hfReference, paramSetTwo.hfReference);
    EXPECT_EQ(paramSetOne.lateReverbDecayTime, paramSetTwo.lateReverbDecayTime);
    EXPECT_EQ(paramSetOne.lateReverbHfDecayRatio, paramSetTwo.lateReverbHfDecayRatio);
    EXPECT_EQ(paramSetOne.roomGain, paramSetTwo.roomGain);
    EXPECT_EQ(paramSetOne.reflectionsGain, paramSetTwo.reflectionsGain);
    EXPECT_EQ(paramSetOne.reverbGain, paramSetTwo.reverbGain);
    EXPECT_EQ(paramSetOne.reverbDiffusion, paramSetTwo.reverbDiffusion);
    EXPECT_EQ(paramSetOne.reflectionsDelayTime, paramSetTwo.reflectionsDelayTime);
    EXPECT_EQ(paramSetOne.reverbDelayTime, paramSetTwo.reverbDelayTime);
    EXPECT_EQ(paramSetOne.reverbDensity, paramSetTwo.reverbDensity);
    EXPECT_EQ(paramSetOne.dryGain, paramSetTwo.dryGain);
}

/**
 * @brief       LoadI3dl2ReverbPreset
 */
TEST(LoadI3dl2ReverbPreset, Success)
{
    nn::audio::I3dl2ReverbParameterSet paramSetOne;
    nn::audio::I3dl2ReverbParameterSet paramSetTwo;
    //MetalCorridor preset settings
    paramSetTwo.roomHfGain = -1000.0f;
    paramSetTwo.hfReference = 5000.0f;
    paramSetTwo.lateReverbDecayTime = 1.1f;
    paramSetTwo.lateReverbHfDecayRatio = 0.3f;
    paramSetTwo.roomGain = -1045.0f;
    paramSetTwo.reflectionsGain = -10000.0f;
    paramSetTwo.reverbGain = -444.0f;
    paramSetTwo.reverbDiffusion = 100.0f;
    paramSetTwo.reflectionsDelayTime = 0.006f;
    paramSetTwo.reverbDelayTime = 0.0f;
    paramSetTwo.reverbDensity = 0.0f;
    paramSetTwo.dryGain = 0.4f;

    nn::audio::LoadI3dl2ReverbPreset(&paramSetOne, nn::audio::I3dl2ReverbType::Preset_MetalCorridor);

    EXPECT_EQ(paramSetOne.roomGain, paramSetTwo.roomGain);
    EXPECT_EQ(paramSetOne.reflectionsGain, paramSetTwo.reflectionsGain);
    EXPECT_EQ(paramSetOne.reverbGain, paramSetTwo.reverbGain);
    EXPECT_EQ(paramSetOne.reverbDiffusion, paramSetTwo.reverbDiffusion);
    EXPECT_EQ(paramSetOne.reflectionsDelayTime, paramSetTwo.reflectionsDelayTime);
    EXPECT_EQ(paramSetOne.reverbDelayTime, paramSetTwo.reverbDelayTime);
    EXPECT_EQ(paramSetOne.reverbDensity, paramSetTwo.reverbDensity);
    EXPECT_EQ(paramSetOne.lateReverbDecayTime, paramSetTwo.lateReverbDecayTime);
    EXPECT_EQ(paramSetOne.lateReverbHfDecayRatio, paramSetTwo.lateReverbHfDecayRatio);
    EXPECT_EQ(paramSetOne.dryGain, paramSetTwo.dryGain);
    EXPECT_EQ(paramSetOne.roomHfGain, paramSetTwo.roomHfGain);
    EXPECT_EQ(paramSetOne.hfReference, paramSetTwo.hfReference);
}

TEST(TestI3dl2Reverb2chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestI3dl2Reverb2chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestI3dl2Reverb2chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(3);
    const int32_t BufferSize = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::WaveFile testWav;
    NN_SDK_LOG("Initializing FS %s\n", g_MountName.c_str());
    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    nn::audio::AuxI3dl2ReverbType i3dl2Reverb;
    size_t i3dl2ReverbSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(params.sampleRate, ChannelCount);
    void* i3dl2ReverbBuffer = allocator.Allocate(i3dl2ReverbSize);
    NN_ASSERT_NOT_NULL(i3dl2ReverbBuffer);

    int8_t bus[] = {0, 1};
    int auxI3dl2ReverbChannelCount = sizeof(bus);
    nn::audio::I3dl2ReverbParameterSet i3dl2Parameter;
    nn::audio::InitializeAuxI3dl2Reverb(&i3dl2Reverb, i3dl2ReverbBuffer, i3dl2ReverbSize, params.sampleRate, auxI3dl2ReverbChannelCount);
    nn::audio::LoadI3dl2ReverbPreset(&i3dl2Parameter, nn::audio::I3dl2ReverbType::Preset_MetalCorridor);
    nn::audio::SetAuxI3dl2ReverbParameters(&i3dl2Reverb, &i3dl2Parameter);
    nn::audio::SetAuxI3dl2ReverbInputOutput(&i3dl2Reverb, bus, bus, sizeof(bus));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, BufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);
    int32_t* readBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, bufferSize);

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    size_t dataSizeToWrite = nnt::audio::util::GetWaveSampleSize(GoldenFile) ;
    size_t dataSizeWritten = 0;
    // keep playing at least for recording period.
    int32_t frameSampleCount = params.sampleCount * ChannelCount;
    size_t frameDataSize = frameSampleCount * sizeof(int16_t);
    int32_t frameSampleCountPerChannel = params.sampleCount;
    int16_t* interleavedData = reinterpret_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * frameSampleCount));

    renderer.Start();
    while(dataSizeWritten < dataSizeToWrite)
    {
        int32_t readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
        int samplesPerFrame = params.sampleCount;
        int frameCount = readCount / ( samplesPerFrame * ChannelCount );
        for (auto frame = 0 ; frame < frameCount; ++frame)
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * ChannelCount;
            nn::audio::ProcessAuxI3dl2Reverb(&i3dl2Reverb, bufferBase, bufferBase, samplesPerFrame);
        }
        nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        //write to wave file
        int32_t sampleCounter = readCount;
        int32_t* src = readBuffer;
        while(sampleCounter > 0)
        {
            int16_t* interleavedDataAccessor = interleavedData;
            memset(interleavedData, 0, frameDataSize);
            for (auto idx = 0; idx < frameSampleCountPerChannel; idx++)
            {
                for (auto ch = 0; ch < ChannelCount; ++ch)
                {
                    int16_t smp = static_cast<int16_t>(src[frameSampleCountPerChannel * ch + idx]);
                    smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                    smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                    *interleavedDataAccessor++ = smp;
                }
            }
            sampleCounter -= frameSampleCount;
            src += frameSampleCount;
            if((dataSizeToWrite - dataSizeWritten) >= frameDataSize)
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), frameDataSize);
                dataSizeWritten += frameDataSize;
            }
            else
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), (dataSizeToWrite - dataSizeWritten));
                dataSizeWritten += (dataSizeToWrite - dataSizeWritten);
            }
        }
        renderer.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
    }
    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
} // NOLINT(impl/function_size)

/**
 * @brief       InitializeAuxReverb(AuxReverbType* pReverb, void* buffer, size_t bufferSize, int sampleRate, int channelCount) NN_NOEXCEPT
 */
TEST(InitializeAuxReverb, Success)
{
    nn::audio::AuxReverbType reverb;
    int sampleRate[2] = {32000, 48000};
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxReverb(48000, 6);
    nn::audio::InitializeAuxReverb(&reverb, g_WorkBuffer, bufferSize, 48000, 6);
    int channelCount[4] = {1, 2, 4, 6};

    for(int i = 0; i < ( sizeof(channelCount) / sizeof(channelCount[0]) ); ++i)
    {
        for(int j = 0; j < 2; ++j)
        {
            bufferSize = nn::audio::GetRequiredBufferSizeForAuxReverb(sampleRate[j], channelCount[i]);
            EXPECT_NE(reverb._bufferSize, bufferSize);
            nn::audio::InitializeAuxReverb(&reverb, g_WorkBuffer, bufferSize, sampleRate[j], channelCount[i]);
            EXPECT_GE(reverb._bufferSize, bufferSize);
        }
    }
}
// void ReverbDeathTests(nn::audio::AuxReverbType* pReverb, nn::audio::ReverbType::EarlyMode earlyMode, nn::audio::ReverbType::LateMode lateMode, float gain, float ratio, float value, nn::TimeSpan time) NN_NOEXCEPT
// {
    // { // Death Check
        // nn::audio::ReverbParameterSet paramSet;
        // paramSet.earlyMode = nn::audio::ReverbType::EarlyMode_SmallRoom;
        // paramSet.earlyGain = gain;
        // paramSet.predelayTimeMilliSeconds = static_cast<float>(time.GetMilliSeconds());
        // paramSet.lateMode = lateMode;
        // paramSet.lateGain = gain;
        // paramSet.decayTimeSeconds = static_cast<float>(time.GetSeconds());
        // paramSet.highFreqDecayRatio = ratio;
        // paramSet.coloration = value;
        // paramSet.reverbGain = gain;
        // paramSet.outGain = gain;
        // paramSet.dryGain = gain;
        // nn::audio::SetAuxReverbParameters(pReverb, &paramSet);

        // EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
        // EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAuxReverbParameters(pReverb) ,"");
    // }
// }

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAuxReverbChannelCountMax, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAuxReverbChannelCountMax(nullptr), "");
}
#endif

TEST(GetAuxReverbChannelCountMax, Success)
{
    nn::audio::AuxReverbType reverb;
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxReverb(48000, 6);
    const int channelCounts[] = { 1, 2, 4, 6 };
    const auto sampleCount = 48000;

    for(const auto channelCount : channelCounts)
    {
        nn::audio::InitializeAuxReverb(&reverb, g_Buffer, bufferSize, sampleCount, channelCount);
        EXPECT_EQ(channelCount, nn::audio::GetAuxReverbChannelCountMax(&reverb));
    }
}

void ReverbDeathTests(nn::audio::AuxReverbType* pReverb, nn::audio::ReverbType::EarlyMode earlyMode, nn::audio::ReverbType::LateMode lateMode, float gain, float ratio, float value, float time) NN_NOEXCEPT
{
    { // Death Check
        nn::audio::ReverbParameterSet arps;
        arps.earlyMode = earlyMode;
        arps.earlyGain = gain;
        arps.predelayTimeMilliSeconds = time;
        arps.lateMode = lateMode;
        arps.lateGain = gain;
        arps.decayTimeSeconds = time;
        arps.highFreqDecayRatio = ratio;
        arps.coloration = value;
        arps.reverbGain = gain;
        arps.outGain = gain;
        arps.dryGain = gain;
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &arps), "");
    }
}
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestAuxReverbParameters, Preconditions)
{
    {
        nn::audio::ReverbType::EarlyMode earlyMode = nn::audio::ReverbType::EarlyMode_Hall;
        nn::audio::ReverbType::LateMode lateMode = nn::audio::ReverbType::LateMode_Hall;
        float gain = 0.0f;
        float ratio = 0.0f;
        float value = 0.0f;
        float time = 0.0f;

        nn::audio::AuxReverbType* pReverb = nullptr;
        ReverbDeathTests(pReverb, earlyMode, lateMode, gain, ratio, value, time);

        nn::audio::AuxReverbType reverb;
        ::std::memset(&reverb, 0xff, sizeof(reverb));
        ReverbDeathTests(&reverb, earlyMode, lateMode, gain, ratio, value, time);
    }

    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};
    const int sampleRate = 32000;

    for (auto j = 0; j < sizeof(channels) / sizeof(int);  ++j)
    {
        auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        param.mixBufferCount = 1 + channels[j]; // SubMix(1) + FinalMix(channels[j])

        nn::audio::AuxReverbType reverb;
        nn::audio::AuxReverbType* pReverb = &reverb;
        nn::audio::ReverbParameterSet paramSet;
        paramSet.earlyMode = nn::audio::ReverbType::EarlyMode_SmallRoom;
        paramSet.earlyGain = 0.0f;
        paramSet.predelayTimeMilliSeconds = 0.0f;
        paramSet.lateMode = nn::audio::ReverbType::LateMode_Room;
        paramSet.lateGain = 0.0f;
        paramSet.decayTimeSeconds = 0.1f;
        paramSet.highFreqDecayRatio = 0.1f;
        paramSet.coloration = 0.0f;
        paramSet.reverbGain = 0.0f;
        paramSet.outGain = 0.0f;
        paramSet.dryGain = 0.0f;
        size_t reverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        nn::audio::InitializeAuxReverb(pReverb, reverbBuffer, reverbSize, sampleRate, channels[j]);
        nn::audio::SetAuxReverbInputOutput(pReverb, input, output, channels[j]);

        nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
        float invalidValues[] = { -0.1f,  1.1f };
        for (auto i = 0; i < sizeof(invalidValues) / sizeof(float); ++i)
        {
            float val = invalidValues[i];
            paramSet.earlyGain = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.earlyGain = 0.0f;
            paramSet.lateGain = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.lateGain = 0.0f;
            paramSet.reverbGain = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.reverbGain = 0.0f;
            paramSet.outGain = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.outGain = 0.0f;
            paramSet.dryGain = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.dryGain = 0.0f;
            paramSet.coloration = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.coloration = 0.0;
        }
        {
            paramSet.earlyMode = nn::audio::ReverbType::EarlyMode_Count;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet), "");
            paramSet.earlyMode = nn::audio::ReverbType::EarlyMode_SmallRoom;

            paramSet.lateMode = nn::audio::ReverbType::LateMode_Count;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.lateMode = nn::audio::ReverbType::LateMode_Room;

            float invalidTime = 301.f;
            paramSet.predelayTimeMilliSeconds = invalidTime;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.predelayTimeMilliSeconds = 0.1f;

            invalidTime = 0.1f;
            paramSet.decayTimeSeconds = invalidTime;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");

            invalidTime = 20.1f;
            paramSet.decayTimeSeconds = invalidTime;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            paramSet.decayTimeSeconds = invalidTime;

            float val = -0.0f;
            paramSet.highFreqDecayRatio = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
            val = 1.1f;
            paramSet.highFreqDecayRatio = val;
            EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxReverbParameters(pReverb, &paramSet) ,"");
        }

        effectAllocator.Free(reverbBuffer);
    }
}
#endif
TEST(TestAuxReverbParameters, Success)
{
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};
    const int sampleRate = 32000;
    for (auto j = 0; j < sizeof(channels) / sizeof(int);  ++j)
    {

        nn::audio::AuxReverbType reverb;
        nn::audio::AuxReverbType* pReverb = &reverb;
        nn::audio::ReverbParameterSet paramSet;
        paramSet.earlyMode = nn::audio::ReverbType::EarlyMode_SmallRoom;
        paramSet.earlyGain = 0.0f;
        paramSet.predelayTimeMilliSeconds = 0.0f;
        paramSet.lateMode = nn::audio::ReverbType::LateMode_Room;
        paramSet.lateGain = 0.0f;
        paramSet.decayTimeSeconds = 0.1f;
        paramSet.highFreqDecayRatio = 0.1f;
        paramSet.coloration = 0.0f;
        paramSet.reverbGain = 0.0f;
        paramSet.outGain = 0.0f;
        paramSet.dryGain = 0.0f;
        nn::audio::ReverbParameterSet returnedParams = paramSet;

        size_t reverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        nn::audio::InitializeAuxReverb(pReverb, reverbBuffer, reverbSize, sampleRate, channels[j]);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        nn::audio::SetAuxReverbInputOutput(&reverb, input, output, channels[j]);

        float validValues[] = { 0.0f,  1.0f };
        for (auto i = 0; i < sizeof(validValues) / sizeof(float); ++i)
        {
            float val = validValues[i];

            paramSet.earlyGain = val;
            paramSet.lateGain = val;
            paramSet.reverbGain = val;
            paramSet.outGain = val;
            paramSet.dryGain = val;
            paramSet.coloration = val;
            nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
            returnedParams = nn::audio::GetAuxReverbParameters(pReverb);
            EXPECT_NEAR(val, returnedParams.earlyGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.lateGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.reverbGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.outGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.dryGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.coloration, g_AudioTestQfPrecision);
        }

        {
            float validTime = 300.f;
            paramSet.predelayTimeMilliSeconds = validTime;
            nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
            returnedParams = nn::audio::GetAuxReverbParameters(pReverb);
            EXPECT_NEAR(validTime, returnedParams.predelayTimeMilliSeconds, g_AudioTestQfPrecision);

            validTime = 0.1f;
            paramSet.decayTimeSeconds = validTime;
            nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
            returnedParams = nn::audio::GetAuxReverbParameters(pReverb);
            EXPECT_NEAR(validTime, returnedParams.decayTimeSeconds, 1); // 内部で[秒]管理している都合、ここの精度は落ちる

            validTime = 20.0f;
            paramSet.decayTimeSeconds = validTime;
            nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
            returnedParams = nn::audio::GetAuxReverbParameters(pReverb);
            EXPECT_NEAR(validTime, returnedParams.decayTimeSeconds, g_AudioTestQfPrecision);

            float val = 0.1f;
            paramSet.highFreqDecayRatio = val;
            nn::audio::SetAuxReverbParameters(pReverb, &paramSet);
            returnedParams = nn::audio::GetAuxReverbParameters(pReverb);
            EXPECT_NEAR(val, paramSet.highFreqDecayRatio, g_AudioTestQfPrecision);
        }


        effectAllocator.Free(reverbBuffer);
    }
    effectAllocator.Finalize();
}
TEST(SetAuxReverbInputOutput, Success)
{
    int8_t auxSubBus[2];
    int8_t finalBus[2];
    auxSubBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBus[nn::audio::ChannelMapping_FrontRight] = 1;
    finalBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    finalBus[nn::audio::ChannelMapping_FrontRight] = 1;
    int sampleRate = 32000;
    int channelCountMax = 2;
    nn::audio::AuxReverbType reverb;
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxReverb(sampleRate, channelCountMax);
    nn::audio::InitializeAuxReverb(&reverb, g_Buffer, bufferSize, sampleRate, channelCountMax);
    int8_t currentBusIn[2] = {3,4};
    int8_t currentBusOut[2] = {5,6};
    int currentOutCount = 0;
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_NE(currentBusIn[i],auxSubBus[i]);
        EXPECT_NE(currentBusOut[i], finalBus[i]);
    }
    EXPECT_NE(currentOutCount, channelCountMax);

    nn::audio::SetAuxReverbInputOutput(&reverb, auxSubBus, finalBus, channelCountMax);
    nn::audio::GetAuxReverbInputOutput(currentBusIn, currentBusOut, &currentOutCount, &reverb, 6);
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_EQ(auxSubBus[i], currentBusIn[i]);
        EXPECT_EQ(finalBus[i], currentBusOut[i]);
    }
    EXPECT_EQ(currentOutCount, channelCountMax);
}

TEST(SetAuxReverbParameters, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    const int sampleRate = 32000;
    const int channelCount = 2;
    nn::audio::ReverbParameterSet auxParam0;
    nn::audio::ReverbParameterSet auxParam1;
    auxParam0.earlyMode = nn::audio::ReverbType::EarlyMode_SmallRoom;
    auxParam0.earlyGain = 0.0f;
    auxParam0.predelayTimeMilliSeconds = 0.0f;
    auxParam0.lateMode = nn::audio::ReverbType::LateMode_Room;
    auxParam0.lateGain = 0.0f;
    auxParam0.decayTimeSeconds = 0.0f;
    auxParam0.highFreqDecayRatio = 0.1f;
    auxParam0.coloration = 0.0f;
    auxParam0.reverbGain = 0.0f;
    auxParam0.outGain = 0.0f;
    auxParam0.dryGain = 0.0f;

    auxParam1.earlyMode = nn::audio::ReverbType::EarlyMode_Hall;
    auxParam1.earlyGain = 0.8f;
    auxParam1.predelayTimeMilliSeconds = 10.0f;
    auxParam1.lateMode = nn::audio::ReverbType::LateMode_Hall;
    auxParam1.lateGain = 0.7f;
    auxParam1.decayTimeSeconds = 1.0f;
    auxParam1.highFreqDecayRatio = 0.1f;
    auxParam1.coloration = 0.0f;
    auxParam1.reverbGain = 0.8f;
    auxParam1.outGain = 0.9f;
    auxParam1.dryGain = 0.9f;

    nn::audio::AuxReverbType auxReverb;
    size_t auxReverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(sampleRate, channelCount);
    void* auxReverbBuffer = allocator.Allocate(auxReverbSize);
    nn::audio::InitializeAuxReverb(&auxReverb, auxReverbBuffer, auxReverbSize, sampleRate, channelCount);
    nn::audio::SetAuxReverbParameters(&auxReverb, &auxParam1);
    auxParam0 = nn::audio::GetAuxReverbParameters(&auxReverb);
    EXPECT_NEAR(auxParam0.earlyMode, auxParam1.earlyMode, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.earlyGain, auxParam1.earlyGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.predelayTimeMilliSeconds, auxParam1.predelayTimeMilliSeconds, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.lateMode, auxParam1.lateMode, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.lateGain, auxParam1.lateGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.decayTimeSeconds, auxParam1.decayTimeSeconds, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.highFreqDecayRatio, auxParam1.highFreqDecayRatio, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.coloration, auxParam1.coloration, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.reverbGain, auxParam1.reverbGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.outGain, auxParam1.outGain, g_AudioTestQfPrecision);
    EXPECT_NEAR(auxParam0.dryGain, auxParam1.dryGain, g_AudioTestQfPrecision);
    allocator.Free(auxReverbBuffer);
    allocator.Finalize();
}

TEST(TestAuxReverb2chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestAuxReverb2chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestAuxReverb2chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t BufferSize = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::WaveFile testWav;
    NN_SDK_LOG("Initializing FS %s\n", g_MountName.c_str());
    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    nn::audio::AuxReverbType auxReverb;
    size_t auxReverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(params.sampleRate, ChannelCount);
    void* auxReverbBuffer = allocator.Allocate(auxReverbSize);
    NN_ASSERT_NOT_NULL(auxReverbBuffer);

    int8_t bus[] = {0, 1};
    int auxReverbChannelCount = sizeof(bus);
    nn::audio::InitializeAuxReverb(&auxReverb,auxReverbBuffer,auxReverbSize, params.sampleRate, auxReverbChannelCount);
    nn::audio::ReverbParameterSet arps;
    arps.coloration = 0.0f;
    arps.decayTimeSeconds = 2.0f;
    arps.dryGain = 0.5f;
    arps.earlyGain = 0.8f;
    arps.earlyMode = nn::audio::ReverbType::EarlyMode_Hall;
    arps.highFreqDecayRatio = 0.5f;
    arps.lateGain = 0.5f;
    arps.lateMode = nn::audio::ReverbType::LateMode_Hall;
    arps.outGain = 1.0f;
    arps.predelayTimeMilliSeconds = 10.0f;
    arps.reverbGain = 0.8f;

    nn::audio::SetAuxReverbParameters(&auxReverb, &arps);
    nn::audio::SetAuxReverbInputOutput(&auxReverb, bus, bus, sizeof(bus));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, BufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);
    int32_t* readBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, bufferSize);

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    size_t dataSizeToWrite = nnt::audio::util::GetWaveSampleSize(GoldenFile) ;
    size_t dataSizeWritten = 0;
    // keep playing at least for recording period.
    int32_t frameSampleCount = params.sampleCount * ChannelCount;
    size_t frameDataSize = frameSampleCount * sizeof(int16_t);
    int32_t frameSampleCountPerChannel = params.sampleCount;
    int16_t* interleavedData = reinterpret_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * frameSampleCount));

    renderer.Start();
    while(dataSizeWritten < dataSizeToWrite)
    {
        int32_t readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
        int samplesPerFrame = params.sampleCount;
        int frameCount = readCount / ( samplesPerFrame * ChannelCount );
        for (auto frame = 0 ; frame < frameCount; ++frame)
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * ChannelCount;
            nn::audio::ProcessAuxReverb(&auxReverb, bufferBase, bufferBase, samplesPerFrame);
        }
        nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        //write to wave file
        int32_t sampleCounter = readCount;
        int32_t* src = readBuffer;
        while(sampleCounter > 0)
        {
            int16_t* interleavedDataAccessor = interleavedData;
            memset(interleavedData, 0, frameDataSize);
            for (auto idx = 0; idx < frameSampleCountPerChannel; idx++)
            {
                for (auto ch = 0; ch < ChannelCount; ++ch)
                {
                    int16_t smp = static_cast<int16_t>(src[frameSampleCountPerChannel * ch + idx]);
                    smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                    smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                    *interleavedDataAccessor++ = smp;
                }
            }
            sampleCounter -= frameSampleCount;
            src += frameSampleCount;
            if((dataSizeToWrite - dataSizeWritten) >= frameDataSize)
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), frameDataSize);
                dataSizeWritten += frameDataSize;
            }
            else
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), (dataSizeToWrite - dataSizeWritten));
                dataSizeWritten += (dataSizeToWrite - dataSizeWritten);
            }
        }
        renderer.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
    }
    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
} // NOLINT(impl/function_size)

TEST(TestAuxReverb1chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestAuxReverb1chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestAuxReverb1chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t BufferSize = 100;
    const int32_t ChannelCount = 1;
    const int32_t BitRate = 16;

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::WaveFile testWav;
    NN_SDK_LOG("Initializing FS %s\n", g_MountName.c_str());
    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    nn::audio::AuxReverbType auxReverb;
    size_t auxReverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(params.sampleRate, ChannelCount);
    void* auxReverbBuffer = allocator.Allocate(auxReverbSize);
    NN_ASSERT_NOT_NULL(auxReverbBuffer);

    int8_t bus[] = {0};
    int auxReverbChannelCount = sizeof(bus);
    nn::audio::InitializeAuxReverb(&auxReverb,auxReverbBuffer,auxReverbSize, params.sampleRate, auxReverbChannelCount);
    nn::audio::ReverbParameterSet arps;
    arps.coloration = 0.0f;
    arps.decayTimeSeconds = 2.0f;
    arps.dryGain = 0.5f;
    arps.earlyGain = 0.8f;
    arps.earlyMode = nn::audio::ReverbType::EarlyMode_Hall;
    arps.highFreqDecayRatio = 0.5f;
    arps.lateGain = 0.5f;
    arps.lateMode = nn::audio::ReverbType::LateMode_Hall;
    arps.outGain = 1.0f;
    arps.predelayTimeMilliSeconds = 10.0f;
    arps.reverbGain = 0.8f;

    nn::audio::SetAuxReverbParameters(&auxReverb, &arps);
    nn::audio::SetAuxReverbInputOutput(&auxReverb, bus, bus, sizeof(bus));
    nn::audio::SetAuxReverbInputOutput(&auxReverb, bus, bus, sizeof(bus));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, BufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0};
    int8_t outputIndex[ChannelCount] = {0};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);
    int32_t* readBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, bufferSize);

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    size_t dataSizeToWrite = nnt::audio::util::GetWaveSampleSize(GoldenFile) ;
    size_t dataSizeWritten = 0;
    // keep playing at least for recording period.
    int32_t frameSampleCount = params.sampleCount * ChannelCount;
    size_t frameDataSize = frameSampleCount * sizeof(int16_t);
    int32_t frameSampleCountPerChannel = params.sampleCount;
    int16_t* interleavedData = reinterpret_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * frameSampleCount));

    renderer.Start();
    while(dataSizeWritten < dataSizeToWrite)
    {
        int32_t readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
        int samplesPerFrame = params.sampleCount;
        int frameCount = readCount / ( samplesPerFrame * ChannelCount );
        for (auto frame = 0 ; frame < frameCount; ++frame)
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * ChannelCount;
            nn::audio::ProcessAuxReverb(&auxReverb, bufferBase, bufferBase, samplesPerFrame);
        }
        nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        //write to wave file
        int32_t sampleCounter = readCount;
        int32_t* src = readBuffer;
        while(sampleCounter > 0)
        {
            int16_t* interleavedDataAccessor = interleavedData;
            memset(interleavedData, 0, frameDataSize);
            for (auto idx = 0; idx < frameSampleCountPerChannel; idx++)
            {
                for (auto ch = 0; ch < ChannelCount; ++ch)
                {
                    int16_t smp = static_cast<int16_t>(src[frameSampleCountPerChannel * ch + idx]);
                    smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                    smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                    *interleavedDataAccessor++ = smp;
                }
            }
            sampleCounter -= frameSampleCount;
            src += frameSampleCount;
            if((dataSizeToWrite - dataSizeWritten) >= frameDataSize)
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), frameDataSize);
                dataSizeWritten += frameDataSize;
            }
            else
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), (dataSizeToWrite - dataSizeWritten));
                dataSizeWritten += (dataSizeToWrite - dataSizeWritten);
            }
        }
        renderer.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
    }
    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
} // NOLINT(impl/function_size)

TEST(TestAuxReverb6chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestAuxReverb6chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestAuxReverb6chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t BufferSize = 100;
    const int32_t ChannelCount = 6;
    const int32_t BitRate = 16;

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::WaveFile testWav;
    NN_SDK_LOG("Initializing FS %s\n", g_MountName.c_str());
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    nn::audio::AuxReverbType auxReverb;
    size_t auxReverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(params.sampleRate, ChannelCount);
    void* auxReverbBuffer = allocator.Allocate(auxReverbSize);
    NN_ASSERT_NOT_NULL(auxReverbBuffer);

    int8_t bus[] = {0, 1, 2, 3, 4, 5};
    int auxReverbChannelCount = sizeof(bus);
    nn::audio::InitializeAuxReverb(&auxReverb,auxReverbBuffer,auxReverbSize, params.sampleRate, auxReverbChannelCount);
    nn::audio::ReverbParameterSet arps;
    arps.coloration = 0.0f;
    arps.decayTimeSeconds = 2.0f;
    arps.dryGain = 0.5f;
    arps.earlyGain = 0.8f;
    arps.earlyMode = nn::audio::ReverbType::EarlyMode_Hall;
    arps.highFreqDecayRatio = 0.5f;
    arps.lateGain = 0.5f;
    arps.lateMode = nn::audio::ReverbType::LateMode_Hall;
    arps.outGain = 1.0f;
    arps.predelayTimeMilliSeconds = 10.0f;
    arps.reverbGain = 0.8f;

    nn::audio::SetAuxReverbParameters(&auxReverb, &arps);
    nn::audio::SetAuxReverbInputOutput(&auxReverb, bus, bus, sizeof(bus));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, BufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1, 2, 3, 4, 5};
    int8_t outputIndex[ChannelCount] = {0, 1, 2, 3, 4, 5};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);
    int32_t* readBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, bufferSize);

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 2);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 3);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 4);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 5);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    size_t dataSizeToWrite = nnt::audio::util::GetWaveSampleSize(GoldenFile) ;
    size_t dataSizeWritten = 0;
    // keep playing at least for recording period.
    int32_t frameSampleCount = params.sampleCount * ChannelCount;
    size_t frameDataSize = frameSampleCount * sizeof(int16_t);
    int32_t frameSampleCountPerChannel = params.sampleCount;
    int16_t* interleavedData = reinterpret_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * frameSampleCount));

    renderer.Start();
    while(dataSizeWritten < dataSizeToWrite)
    {
        int32_t readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
        int samplesPerFrame = params.sampleCount;
        int frameCount = readCount / ( samplesPerFrame * ChannelCount );
        for (auto frame = 0 ; frame < frameCount; ++frame)
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * ChannelCount;
            nn::audio::ProcessAuxReverb(&auxReverb, bufferBase, bufferBase, samplesPerFrame);
        }
        nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        //write to wave file
        int32_t sampleCounter = readCount;
        int32_t* src = readBuffer;
        while(sampleCounter > 0)
        {
            int16_t* interleavedDataAccessor = interleavedData;
            memset(interleavedData, 0, frameDataSize);
            for (auto idx = 0; idx < frameSampleCountPerChannel; idx++)
            {
                for (auto ch = 0; ch < ChannelCount; ++ch)
                {
                    int16_t smp = static_cast<int16_t>(src[frameSampleCountPerChannel * ch + idx]);
                    smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                    smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                    *interleavedDataAccessor++ = smp;
                }
            }
            sampleCounter -= frameSampleCount;
            src += frameSampleCount;
            if((dataSizeToWrite - dataSizeWritten) >= frameDataSize)
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), frameDataSize);
                dataSizeWritten += frameDataSize;
            }
            else
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), (dataSizeToWrite - dataSizeWritten));
                dataSizeWritten += (dataSizeToWrite - dataSizeWritten);
            }
        }
        renderer.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
    }
    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
} // NOLINT(impl/function_size)

TEST(InitializeAuxDelay, Success)
{
    nn::audio::AuxDelayType delay;
    const int sampleRates[] = {32000, 48000};
    size_t bufferSize = 0;
    const int channelCounts[] = {1, 2, 4, 6};
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(100);
    for(const auto channelCount : channelCounts)
    {
        for(const auto sampleRate : sampleRates)
        {
            bufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(delayTime, sampleRate, channelCount);
            nn::audio::InitializeAuxDelay(&delay, g_Buffer, bufferSize, delayTime, sampleRate, channelCount);
            EXPECT_GE(delay._bufferSize, bufferSize);
        }
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(InitializeAuxDelay, PreCondition)
{
    nn::audio::AuxDelayType delay;
    int sampleRate[2] = {32000, 48000};
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(10);
    nn::TimeSpan invalidDelayTime = nn::TimeSpan::FromMilliSeconds(-10);
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(delayTime, 48000, 6);
    int channelCount[4] = {1, 2, 4, 6};
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxDelay(nullptr, g_Buffer, bufferSize, delayTime, sampleRate[1], channelCount[3]), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxDelay(&delay, nullptr, bufferSize, delayTime, sampleRate[1], channelCount[3]), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxDelay(&delay, g_Buffer, 0, delayTime, sampleRate[1], channelCount[3]), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::InitializeAuxDelay(&delay, g_Buffer, 0, invalidDelayTime, sampleRate[1], channelCount[3]), "");
}
#endif

TEST(TestSetAuxDelayInputOutput, Success)
{
    int8_t auxSubBus[2];
    int8_t finalBus[2];
    auxSubBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBus[nn::audio::ChannelMapping_FrontRight] = 1;
    finalBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    finalBus[nn::audio::ChannelMapping_FrontRight] = 1;

    int sampleRate = 32000;
    int channelCountMax = 2;
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(100);
    nn::TimeSpan maxDelay = nn::TimeSpan::FromMilliSeconds(300);
    nn::audio::AuxDelayType delay;
    size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(maxDelay, sampleRate, channelCountMax);
    nn::audio::InitializeAuxDelay(&delay, g_Buffer, bufferSize, delayTime, sampleRate, channelCountMax);
    int8_t currentBusIn[2] = {3, 4};
    int8_t currentBusOut[2] = {5, 6};
    int currentOutCount = 0;
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_NE(currentBusIn[i], auxSubBus[i]);
        EXPECT_NE(currentBusOut[i], finalBus[i]);
    }
    EXPECT_NE(currentOutCount, channelCountMax);

    nn::audio::SetAuxDelayInputOutput(&delay, auxSubBus, finalBus, channelCountMax);
    nn::audio::GetAuxDelayInputOutput(currentBusIn, currentBusOut, &currentOutCount, &delay, 6);
    for(int i = 0; i < 2; ++i)
    {
        EXPECT_EQ(auxSubBus[i], currentBusIn[i]);
        EXPECT_EQ(finalBus[i], currentBusOut[i]);
    }
    EXPECT_EQ(currentOutCount, channelCountMax);
}

TEST(TestSetAuxDelayParams, Success)
{
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};
    const int sampleRate = 32000;
    for (int j = 0; j < sizeof(channels) / sizeof(int);  ++j)
    {
        nn::audio::AuxDelayType delay;
        nn::audio::AuxDelayType* pDelay = & delay;
        nn::audio::DelayParameterSet paramSet;
        paramSet.delayTime = nn::TimeSpan::FromMilliSeconds(400);
        paramSet.inGain = 0.5f;
        paramSet.feedbackGain = 0.4f;
        paramSet.dryGain = 0.5f;
        paramSet.channelSpread = 0.5f;
        paramSet.lowPassAmount = 0.5f;
        nn::audio::DelayParameterSet returnedParams = paramSet;

        size_t bufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(paramSet.delayTime, sampleRate, channels[j]);
        void* delayBuffer = effectAllocator.Allocate(bufferSize);
        nn::audio::InitializeAuxDelay(pDelay, delayBuffer, bufferSize, paramSet.delayTime, sampleRate, channels[j]);
        NN_ASSERT_NOT_NULL(delayBuffer);
        nn::audio::SetAuxDelayInputOutput(pDelay, input, output, channels[j]);
        float validValues[] = {0.0f, 1.0f };
        nn::TimeSpan validDelayValues[] = {nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(100)};
        for(int i = 0; i < sizeof(validValues) / sizeof(float); ++i)
        {
            float val = validValues[i];
            nn::TimeSpan delayValue = validDelayValues[i];
            paramSet.delayTime = delayValue;
            paramSet.inGain = val;
            paramSet.feedbackGain = val;
            paramSet.dryGain = val;
            paramSet.channelSpread = val;
            paramSet.lowPassAmount = val;
            nn::audio::SetAuxDelayParameters(pDelay, &paramSet);
            returnedParams = nn::audio::GetAuxDelayParameters(pDelay);
            EXPECT_EQ(delayValue, returnedParams.delayTime);
            EXPECT_NEAR(val, returnedParams.inGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.feedbackGain, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.channelSpread, g_AudioTestQfPrecision);
            EXPECT_NEAR(val, returnedParams.lowPassAmount, g_AudioTestQfPrecision);
        }

        effectAllocator.Free(delayBuffer);
    }
    effectAllocator.Finalize();
}

TEST(TestAuxDelayWaveComp2ch, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed2chFileName[0];
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed2chFileName[0];
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t BufferSize = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::WaveFile testWav;
    NN_SDK_LOG("Initializing FS %s\n", g_MountName.c_str());
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    nn::audio::AuxDelayType delay;
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(200);
    size_t delayBufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(nn::TimeSpan::FromSeconds(2), params.sampleRate, ChannelCount);
    void* delayBuffer = allocator.Allocate(delayBufferSize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    int8_t bus[] = {0, 1};
    nn::audio::InitializeAuxDelay(&delay, delayBuffer, delayBufferSize, delayTime, params.sampleRate, ChannelCount);
    nn::audio::DelayParameterSet delayParam;
    delayParam.lowPassAmount = 0.9f;
    delayParam.channelSpread = 0.256f;
    delayParam.dryGain = 0.5f;
    delayParam.inGain = 0.4f;
    delayParam.feedbackGain = 0.7f;
    delayParam.delayTime = delayTime;

    nn::audio::SetAuxDelayParameters(&delay, &delayParam);
    nn::audio::SetAuxDelayInputOutput(&delay, bus, bus, sizeof(bus));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, BufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);
    int32_t* readBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, bufferSize);

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    size_t dataSizeToWrite = nnt::audio::util::GetWaveSampleSize(GoldenFile) ;
    size_t dataSizeWritten = 0;
    // keep playing at least for recording period.
    int32_t frameSampleCount = params.sampleCount * ChannelCount;
    size_t frameDataSize = frameSampleCount * sizeof(int16_t);
    int32_t frameSampleCountPerChannel = params.sampleCount;
    int16_t* interleavedData = reinterpret_cast<int16_t*>(allocator.Allocate(sizeof(int16_t) * frameSampleCount));

    renderer.Start();
    while(dataSizeWritten < dataSizeToWrite)
    {
        int32_t readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
        int samplesPerFrame = params.sampleCount;
        int frameCount = readCount / ( samplesPerFrame * ChannelCount );
        for (auto frame = 0 ; frame < frameCount; ++frame)
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * ChannelCount;
            nn::audio::ProcessAuxDelay(&delay, bufferBase, bufferBase, samplesPerFrame);
        }
        nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        //write to wave file
        int32_t sampleCounter = readCount;
        int32_t* src = readBuffer;
        while(sampleCounter > 0)
        {
            int16_t* interleavedDataAccessor = interleavedData;
            memset(interleavedData, 0, frameDataSize);
            for (auto idx = 0; idx < frameSampleCountPerChannel; idx++)
            {
                for (auto ch = 0; ch < ChannelCount; ++ch)
                {
                    int16_t smp = static_cast<int16_t>(src[frameSampleCountPerChannel * ch + idx]);
                    smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                    smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                    *interleavedDataAccessor++ = smp;
                }
            }
            sampleCounter -= frameSampleCount;
            src += frameSampleCount;
            if((dataSizeToWrite - dataSizeWritten) >= frameDataSize)
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), frameDataSize);
                dataSizeWritten += frameDataSize;
            }
            else
            {
                testWav.Write(reinterpret_cast<int8_t*>(interleavedData), (dataSizeToWrite - dataSizeWritten));
                dataSizeWritten += (dataSizeToWrite - dataSizeWritten);
            }
        }
        renderer.Update();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
    }
    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    testWav.Close();
    allocator.Free(delayBuffer);
    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    allocator.Finalize();

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile, 1);
} // NOLINT(impl/function_size)

TEST(TestAuxDelayInsufficientBufferSize, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AuxDelayType delay;
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(200);
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    int ChannelCount = 2;
    size_t delayBufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(delayTime, params.sampleRate, ChannelCount);
    void* delayBuffer = allocator.Allocate(delayBufferSize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    nn::audio::InitializeAuxDelay(&delay, delayBuffer, delayBufferSize, delayTime, params.sampleRate, ChannelCount);

    allocator.Free(delayBuffer);
    allocator.Finalize();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestAuxDelayInsufficientBufferSize, PreCondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::AuxDelayType delay;
    nn::TimeSpan delayTime = nn::TimeSpan::FromMilliSeconds(200);
    nn::TimeSpan delayTimeLong = nn::TimeSpan::FromMilliSeconds(400);
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    int ChannelCount = 2;
    size_t delayBufferSize = nn::audio::GetRequiredBufferSizeForAuxDelay(delayTime, params.sampleRate, ChannelCount);
    void* delayBuffer = allocator.Allocate(delayBufferSize);
    NN_ASSERT_NOT_NULL(delayBuffer);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::InitializeAuxDelay(&delay, delayBuffer, delayBufferSize, delayTimeLong, params.sampleRate, ChannelCount), "");

    allocator.Free(delayBuffer);
    allocator.Finalize();
}
#endif
