﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"
#include <mutex>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>

#include <nn/mem.h>

namespace {

    const int MemoryHeapSize = 32 * 1024 * 1024;

    class SoundArchivePlayerSetup : public nnt::atk::util::AtkCommonSetup
    {
    public:
        // テスト対象である SoundArchivePlayer の初期化をクラス外で行う
        void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam param;
            Initialize(param, allocator);
        }

        void Initialize(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            InitializeSoundSystem(param.GetSoundSystemParam(), allocator);
            InitializeSoundHeap(param.GetSoundHeapSize(), allocator);
            InitializeSoundArchive(allocator);
            InitializeSoundDataManager(allocator);
        }

        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            FinalizeSoundDataManager(allocator);
            FinalizeSoundArchive(allocator);
            FinalizeSoundHeap(allocator);
            FinalizeSoundSystem(allocator);
        }
    };

    struct AddonSoundArchiveSet
    {
        nn::atk::AddonSoundArchive addonSoundArchive;
        void* pMemoryForAddonSoundArchive;
        nn::atk::SoundDataManager addonSoundDataManager;
        void* pMemoryForAddonSoundDataManager;
    };

    nnt::atk::util::FsCommonSetup g_FsSetup;
    SoundArchivePlayerSetup g_AtkSetup;

    struct SoundArchivePlayerBufferSet
    {
        void* pMemoryForSoundArchivePlayer;
        std::size_t memSizeForSoundArchivePlayer;
        std::size_t memSizeForStreamBuffer;
        void* pMemoryForStreamBuffer;
    };

    struct TestArgs
    {
        int testFrame;
        int sequenceRestartFrame;
        int streamRestartFrame;
        int waveRestartFrame;
        nn::atk::SoundArchivePlayer* pSoundArchivePlayer;
        nn::os::Semaphore* pSemaphore;
    };

    struct FinalizedThreadCounter
    {
        int threadCount;
        nn::os::Semaphore* pSemaphore;
    };

    static char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN  char g_HeapMemoryForMemoryPool[MemoryHeapSize];
    nn::mem::StandardAllocator  g_AllocatorForMemoryPool;
    nn::os::Mutex g_Mutex(true);

    const size_t ThreadStackSize = 4096 * 2;
}

namespace
{
    void AllocateSoundArchivePlayerBuffer(SoundArchivePlayerBufferSet* pOutBufferSet, nn::atk::SoundArchivePlayer& soundArchivePlayer, nn::atk::SoundArchive& soundArchive)
    {
        // SoundArchivePlayer の初期化のための設定
        pOutBufferSet->memSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize(&soundArchive);
        pOutBufferSet->pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, pOutBufferSet->memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
        pOutBufferSet->memSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize(&soundArchive);
        pOutBufferSet->pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory(g_AllocatorForMemoryPool, pOutBufferSet->memSizeForStreamBuffer, nn::audio::BufferAlignSize);
    }

    void AllocateSoundArchivePlayerBuffer(SoundArchivePlayerBufferSet* pOutBufferSet)
    {
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        AllocateSoundArchivePlayerBuffer(pOutBufferSet, soundArchivePlayer, soundArchive);
    }

    void FreeSoundArchivePlayerBuffer(SoundArchivePlayerBufferSet* pOutBufferSet)
    {
        g_AllocatorForMemoryPool.Free(pOutBufferSet->pMemoryForStreamBuffer);
        g_Allocator.Free(pOutBufferSet->pMemoryForSoundArchivePlayer);
    }

    void LoadData()
    {
        // 音源データの読み込み
        g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
        g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
        g_AtkSetup.LoadData(WSD_SNARE, "WSD_SNARE");
    }


    void PlaySoundArchivePlayer(void *args)
    {
        NN_ABORT_UNLESS_NOT_NULL(args);

        TestArgs* config = reinterpret_cast<TestArgs*>(args);
        NN_ABORT_UNLESS_GREATER(config->testFrame, 0);
        NN_ABORT_UNLESS_GREATER(config->sequenceRestartFrame, 0);
        NN_ABORT_UNLESS_GREATER(config->streamRestartFrame, 0);
        NN_ABORT_UNLESS_GREATER(config->waveRestartFrame, 0);

        if(config->pSemaphore != nullptr)
        {
            config->pSemaphore->Acquire();
        }

        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
        nn::atk::SoundArchivePlayer* pSoundArchivePlayer = config->pSoundArchivePlayer;
        NN_ABORT_UNLESS_NOT_NULL(pSoundArchivePlayer);
        nn::atk::SoundArchivePlayer& soundArchivePlayer = *pSoundArchivePlayer;
        SoundArchivePlayerBufferSet bufferSet;
        AllocateSoundArchivePlayerBuffer(&bufferSet, soundArchivePlayer, soundArchive);

        {
            // SoundArchivePlayer の初期化は、 DriverCommand への push を行うので排他制御して呼ぶ
            std::lock_guard<nn::os::Mutex> lock(g_Mutex);
            EXPECT_TRUE(soundArchivePlayer.Initialize(
                &soundArchive,
                &soundDataManager,
                bufferSet.pMemoryForSoundArchivePlayer, bufferSet.memSizeForSoundArchivePlayer,
                bufferSet.pMemoryForStreamBuffer, bufferSet.memSizeForStreamBuffer));
        }

        {
            // SoundSystem におけるグローバルに影響する Set 関数やエフェクト着脱に関する API は、 DriverCommand への push を行うので排他制御する
            std::lock_guard<nn::os::Mutex> lock(g_Mutex);
            nn::atk::SoundSystem::SetAuxBusVolume(nn::atk::AuxBus_A, 0.9f, nn::TimeSpan::FromMilliSeconds(500));
            nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);
            nn::atk::SoundSystem::SetMasterVolume(1.0f, 250);
        }

        EXPECT_TRUE(soundArchivePlayer.IsAvailable());
        EXPECT_TRUE(&soundArchivePlayer.GetSoundArchive() == &soundArchive);

        nn::atk::SoundHandle seqSoundHandle;
        nn::atk::SoundHandle seSoundHandle;
        nn::atk::SoundHandle strmSoundHandle;

        for (int i = 0; i < config->testFrame; i++)
        {
            // DriverCommand に影響する API は排他制御して呼ぶ
            if (i % config->sequenceRestartFrame == 0)
            {
                {
                    std::lock_guard<nn::os::Mutex> lock(g_Mutex);
                    seqSoundHandle.Stop(0);
                    EXPECT_TRUE(soundArchivePlayer.StartSound(&seqSoundHandle, SEQ_MARIOKART).IsSuccess());

                    nn::atk::SequenceSoundHandle tempSoundHandle(&seqSoundHandle);
                    // SequenceSoundHandle に特有の Set 関数は DriverCommand への push を行うので排他制御する
                    // ここでは、それらに該当するいくつかの API をテストする
                    tempSoundHandle.SetTempoRatio(0.9f);
                    tempSoundHandle.SetChannelPriority(54);
                    tempSoundHandle.SetTrackOutputSurroundPan(nn::atk::OutputDevice_Main, 1, 1.0f);
                }
                // SoundHandle による Set 関数は、 SoundArchivePlayer::Update による処理で初めてコマンドが push されるので、排他制御せずに呼ぶ
                // ここでは、それらに該当するいくつかの API をテストする
                seqSoundHandle.Pause(false, 200);
                seqSoundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter2048, 0.1f);
                seqSoundHandle.SetPitch(1.2f);
            }
            if (i % config->waveRestartFrame == 0)
            {
                {
                    std::lock_guard<nn::os::Mutex> lock(g_Mutex);
                    seSoundHandle.Stop(0);
                    EXPECT_TRUE(soundArchivePlayer.StartSound(&seSoundHandle, SE_YOSHI).IsSuccess());

                    nn::atk::WaveSoundHandle tempSoundHandle(&seSoundHandle);
                    // WaveSoundHandle に特有の Set 関数は DriverCommand への push を行うので排他制御する
                    // ここでは、それらに該当するいくつかの API をテストする
                    tempSoundHandle.SetChannelPriority(55);
                }
                // SoundHandle による Set 関数は、 SoundArchivePlayer::Update による処理で初めてコマンドが push されるので、排他制御せずに呼ぶ
                // ここでは、それらに該当するいくつかの API をテストする
                seSoundHandle.ResetOutputLine();
                seSoundHandle.SetPlayerPriority(68);
                seSoundHandle.SetEffectSend(nn::atk::AuxBus_A, 0.5f);
            }
            if (i % config->streamRestartFrame == 0)
            {
                {
                    std::lock_guard<nn::os::Mutex> lock(g_Mutex);
                    strmSoundHandle.Stop(0);
                    EXPECT_TRUE(soundArchivePlayer.StartSound(&strmSoundHandle, STRM_MARIOKART_INTRO).IsSuccess());

                    nn::atk::StreamSoundHandle tempSoundHandle(&seSoundHandle);
                    // StreamSoundHandle に特有の Set 関数は DriverCommand への push を行うので排他制御する
                    // ここでは、それらに該当するいくつかの API をテストする
                    tempSoundHandle.SetTrackOutputPan(nn::atk::OutputDevice_Main, 1, 0.2f);
                    tempSoundHandle.SetTrackOutputMainSend(nn::atk::OutputDevice_Main, 1, 1.2f);
                }
                // SoundHandle による Set 関数は、 SoundArchivePlayer::Update による処理で初めてコマンドが push されるので、排他制御せずに呼ぶ
                // ここでは、それらに該当するいくつかの API をテストする
                strmSoundHandle.SetMainSend(0.9f);
                strmSoundHandle.SetLowPassFilterFrequency(-0.1f);
                strmSoundHandle.SetVolume(1.0f);
            }
            {
                std::lock_guard<nn::os::Mutex> lock(g_Mutex);
                soundArchivePlayer.Update();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }

        {
            // SoundArchivePlayer の終了処理も、 DriverCommand への push を行うので排他制御して呼ぶ
            std::lock_guard<nn::os::Mutex> lock(g_Mutex);
            soundArchivePlayer.Finalize();
        }

        FreeSoundArchivePlayerBuffer(&bufferSet);

        if(config->pSemaphore != nullptr)
        {
            config->pSemaphore->Release();
        }
    } // NOLINT(impl/function_size)
}

TEST(MultiSoundArchivePlayer, LongRun_SingleThreadOperationTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nn::audio::MemoryPoolType memoryPool;

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPool, g_HeapMemoryForMemoryPool, MemoryHeapSize);

    SoundArchivePlayerBufferSet bufferSet;
    AllocateSoundArchivePlayerBuffer(&bufferSet);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    nn::atk::SoundArchivePlayer anotherSoundArchivePlayer;
    SoundArchivePlayerBufferSet anotherBufferSet;
    AllocateSoundArchivePlayerBuffer(&anotherBufferSet, anotherSoundArchivePlayer, soundArchive);

    EXPECT_TRUE(soundArchivePlayer.Initialize(
                    &soundArchive,
                    &soundDataManager,
                    bufferSet.pMemoryForSoundArchivePlayer, bufferSet.memSizeForSoundArchivePlayer,
                    bufferSet.pMemoryForStreamBuffer, bufferSet.memSizeForStreamBuffer));
    EXPECT_TRUE(anotherSoundArchivePlayer.Initialize(
                    &soundArchive,
                    &soundDataManager,
                    anotherBufferSet.pMemoryForSoundArchivePlayer, anotherBufferSet.memSizeForSoundArchivePlayer,
                    anotherBufferSet.pMemoryForStreamBuffer, anotherBufferSet.memSizeForStreamBuffer));

    EXPECT_TRUE(soundArchivePlayer.IsAvailable());
    EXPECT_TRUE(anotherSoundArchivePlayer.IsAvailable());

    EXPECT_TRUE(&soundArchivePlayer.GetSoundArchive() == &soundArchive);
    EXPECT_TRUE(&anotherSoundArchivePlayer.GetSoundArchive() == &soundArchive);

    for(int i = 0; i < 200; i++)
    {
        if (i == 0)
        {
            nn::atk::SoundHandle soundHandle;
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SEQ_MARIOKART_PLAYERHEAP).IsSuccess());
            EXPECT_TRUE(anotherSoundArchivePlayer.StartSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
        }
        if (i % 30 == 0)
        {
            nn::atk::SoundHandle soundHandle;
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess());
            EXPECT_TRUE(anotherSoundArchivePlayer.StartSound(&soundHandle, SE_YOSHI_PLAYERHEAP).IsSuccess());
        }
        if (i % 60 == 0)
        {
            nn::atk::SoundHandle soundHandle;
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess());
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART_48k).IsSuccess());
        }

        soundArchivePlayer.Update();
        anotherSoundArchivePlayer.Update();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    soundArchivePlayer.Finalize();
    anotherSoundArchivePlayer.Finalize();

    FreeSoundArchivePlayerBuffer(&bufferSet);
    FreeSoundArchivePlayerBuffer(&anotherBufferSet);
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
    g_AllocatorForMemoryPool.Finalize();
}

TEST(MultiSoundArchivePlayer, LongRun_MultiThreadOperationTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nn::audio::MemoryPoolType memoryPool;

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPool, g_HeapMemoryForMemoryPool, MemoryHeapSize);

    nn::os::ThreadType thread[2];

    nn::Result result;
    nn::atk::SoundArchivePlayer subSoundArchivePlayer;
    TestArgs config[2] =
    {
        {
            1000,
            95,
            15,
            40,
            &g_AtkSetup.GetSoundArchivePlayer(),
            nullptr
        },
        {
            800,
            66,
            30,
            25,
            &subSoundArchivePlayer,
            nullptr
        }
    };

    void* threadStack[2];

    threadStack[0] = g_Allocator.Allocate(ThreadStackSize, nn::os::ThreadStackAlignment);
    threadStack[1] = g_Allocator.Allocate(ThreadStackSize, nn::os::ThreadStackAlignment);

    result = nn::os::CreateThread(&thread[0], PlaySoundArchivePlayer, reinterpret_cast<void*>(&config[0]), threadStack[0], ThreadStackSize, nn::os::DefaultThreadPriority);
    ASSERT_TRUE(result.IsSuccess());
    result = nn::os::CreateThread(&thread[1], PlaySoundArchivePlayer, reinterpret_cast<void*>(&config[1]), threadStack[1], ThreadStackSize, nn::os::DefaultThreadPriority);
    ASSERT_TRUE(result.IsSuccess());

    nn::os::StartThread(&thread[0]);
    nn::os::StartThread(&thread[1]);

    nn::os::WaitThread(&thread[0]);
    nn::os::WaitThread(&thread[1]);

    nn::os::DestroyThread(&thread[0]);
    nn::os::DestroyThread(&thread[1]);

    g_Allocator.Free(threadStack[0]);
    g_Allocator.Free(threadStack[1]);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
    g_AllocatorForMemoryPool.Finalize();
}

TEST(MultiSoundArchivePlayer, LongRun_MultiThreadMultiCoreOperationTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_HeapMemoryForMemoryPool, MemoryHeapSize);
    nn::audio::MemoryPoolType memoryPool;

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPool, g_HeapMemoryForMemoryPool, MemoryHeapSize);

    nn::os::ThreadType thread[3];
    nn::atk::SoundArchivePlayer soundArchivePlayer[2];
    TestArgs config[3] =
    {
        {
            3500,
            55,
            60,
            75,
            &g_AtkSetup.GetSoundArchivePlayer(),
            nullptr
        },
        {
            3000,
            110,
            120,
            150,
            &soundArchivePlayer[0],
            nullptr
        },
        {
            2900,
            40,
            30,
            20,
            &soundArchivePlayer[1],
            nullptr
        }
    };

    void* threadStack[3];
    threadStack[0] = g_Allocator.Allocate(ThreadStackSize, nn::os::ThreadStackAlignment);
    threadStack[1] = g_Allocator.Allocate(ThreadStackSize, nn::os::ThreadStackAlignment);
    threadStack[2] = g_Allocator.Allocate(ThreadStackSize, nn::os::ThreadStackAlignment);

    nn::Result result;
    result = nn::os::CreateThread(&thread[0], PlaySoundArchivePlayer, reinterpret_cast<void*>(&config[0]), threadStack[0], ThreadStackSize, nn::os::DefaultThreadPriority, 0);
    ASSERT_TRUE(result.IsSuccess());
    result = nn::os::CreateThread(&thread[1], PlaySoundArchivePlayer, reinterpret_cast<void*>(&config[1]), threadStack[1], ThreadStackSize, nn::os::DefaultThreadPriority, 1);
    ASSERT_TRUE(result.IsSuccess());
    result = nn::os::CreateThread(&thread[2], PlaySoundArchivePlayer, reinterpret_cast<void*>(&config[2]), threadStack[2], ThreadStackSize, nn::os::DefaultThreadPriority, 2);
    ASSERT_TRUE(result.IsSuccess());

    nn::os::StartThread(&thread[0]);
    nn::os::StartThread(&thread[1]);
    nn::os::StartThread(&thread[2]);;

    nn::os::WaitThread(&thread[0]);
    nn::os::WaitThread(&thread[1]);
    nn::os::WaitThread(&thread[2]);

    nn::os::DestroyThread(&thread[0]);
    nn::os::DestroyThread(&thread[1]);
    nn::os::DestroyThread(&thread[2]);

    g_Allocator.Free(threadStack[0]);
    g_Allocator.Free(threadStack[1]);
    g_Allocator.Free(threadStack[2]);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
    g_AllocatorForMemoryPool.Finalize();
}
