﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_FileLogger.h>
#include <nnt/atkUtil/testAtk_ProfileStatistics.h>
#include <nnt/atkUtil/testAtk_SoundThreadStatistics.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nnt/atkUtil/testAtk_Constants.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {

    const int MemoryHeapSize = 32 * 1024 * 1024;
    const char InitialPerformanceLog[] = "Number of Voices,Atk Frame(us),Ren Total Frame(us),Voice Frame(us),Voice Frame Per Voice(us),Sub Mix Frame(us),Final Mix Frame(us),Sink Frame(us),Sound Thread Interval(us)\n";
    const char InitialSoundThreadPerformanceLog[] = "SoundThread Process(us),Update LowLevelVoice(us),FrameProcess(us),UpdateRenderer(us),UserEffect(us),Wait Renderer Event(us)\n";
    const int ProcessTimeThreshold = 3000; // 5000[us] * 0.6
    const int RenderingSampleRate_48k = 48000;
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    int GetTestFrame(nn::TimeSpan time)
    {
        int result = 0;
        for (;;)
        {
            if(time >= WaitTime)
            {
                time = time - WaitTime;
                result++;
            }
            else
            {
                if(time != nn::TimeSpan(0))
                {
                    result++;
                }
                break;
            }
        }
        return result;
    }

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    const char* TestNameSuffix = "vcmd";
#else
    const char* TestNameSuffix = "";
#endif

    static char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;

    nn::atk::SoundProfile  g_SoundProfile[32];
    nn::atk::ProfileReader g_ProfileReader;
    nn::os::Tick           g_PreviousAtkFrameProcessBegin = nn::os::Tick(0);

    nn::atk::SoundThreadUpdateProfile g_SoundThreadUpdateProfile[32];

    struct CheckTestConfig
    {
        int   rendererSampleRate;

        bool  isHandlePitchEnabled;
        bool  isHandleLpfEnabled;
        bool  isHandleBiquadEnabled;

        float handlePitch;
        float handleLpf;
        int   handleBiquadFilterType;
        float handleBiquadFilterValue;

        CheckTestConfig() NN_NOEXCEPT
            : rendererSampleRate(RenderingSampleRate_48k)
            , isHandlePitchEnabled(false)
            , isHandleLpfEnabled(false)
            , isHandleBiquadEnabled(false)
            , handlePitch(1.0f)
            , handleLpf(0.0f)
            , handleBiquadFilterType(nn::atk::BiquadFilterType_None)
            , handleBiquadFilterValue(0.0f)
        {}
    };

    void ApplyCheckTestConfig(nn::atk::SoundHandle& handle, CheckTestConfig& config) NN_NOEXCEPT
    {
        if ( config.isHandlePitchEnabled )
        {
            handle.SetPitch(config.handlePitch);
        }
        if ( config.isHandleLpfEnabled )
        {
            handle.SetLowPassFilterFrequency(config.handleLpf);
        }
        if ( config.isHandleBiquadEnabled )
        {
            handle.SetBiquadFilter(config.handleBiquadFilterType, config.handleBiquadFilterValue);
        }
    }

    class FsProfilerSetup : public nnt::atk::util::FsCommonSetup
    {
    public:
        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            // Contents ディレクトリの作成
            nn::Result result = nn::fs::MountHostRoot();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            // + 1 は終端文字の分
            char contentsDirectory[nn::fs::EntryNameLengthMax + 1];
            NN_ABORT_UNLESS(nnt::atk::util::GetContentsDirectoryPath(contentsDirectory, sizeof(contentsDirectory)), "Cannot get contents directory path.");
            nnt::atk::util::CreateDirectory(contentsDirectory);
            nn::fs::UnmountHostRoot();

            // TestResults ディレクトリの作成
            // + 1 は終端文字の分
            char testResultsDirectory[nn::fs::EntryNameLengthMax + 1];
            nnt::atk::util::GetTestResultsDirectoryPath(testResultsDirectory, sizeof(testResultsDirectory));
            result = nn::fs::MountHostRoot();
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            nnt::atk::util::CreateDirectory(testResultsDirectory);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            nn::fs::UnmountHostRoot();

            nnt::atk::util::FsCommonSetup::InitializeParam param;
            param.SetContentsDirectoryMounted(true);
            param.SetTestResultsDirectoryMounted(true);
            nnt::atk::util::FsCommonSetup::Initialize(param);
        }
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::atk::util::FsCommonSetup::Finalize();
        }
    };

    class AtkProfilerSetup : public nnt::atk::util::AtkCommonSetup
    {
    public:
        virtual void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            InitializeImpl(allocator, false, RenderingSampleRate_48k);
        }
        void InitializeImpl(nn::mem::StandardAllocator& allocator, int rendererSampleRate) NN_NOEXCEPT
        {
            InitializeImpl(allocator, false, rendererSampleRate);
        }
        void InitializeImpl(nn::mem::StandardAllocator& allocator, bool isVoiceDropEnabled) NN_NOEXCEPT
        {
            InitializeImpl(allocator, isVoiceDropEnabled, RenderingSampleRate_48k);
        }
        void InitializeImpl(nn::mem::StandardAllocator& allocator, bool isVoiceDropEnabled, int rendererSampleRate) NN_NOEXCEPT
        {
            //  g_ProfileReader に直前の TEST の結果が入っていることがあるため、一旦全部読みます
            int profileCountMax = static_cast<int>(sizeof(g_SoundProfile) / sizeof(g_SoundProfile[0]));
            g_ProfileReader.Read( g_SoundProfile, profileCountMax );

            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableProfiler = true;
            soundSystemParam.enableVoiceDrop = isVoiceDropEnabled;
            soundSystemParam.voiceCountMax = nn::atk::SoundSystem::VoiceCountMax;
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            initializeParam.GetSoundSystemParam().rendererSampleRate = rendererSampleRate;

            // ProfileReader の登録
            nnt::atk::util::AtkCommonSetup::Initialize(initializeParam, allocator);
            nn::atk::SoundSystem::RegisterProfileReader(g_ProfileReader);
            g_PreviousAtkFrameProcessBegin = nn::os::GetSystemTick();
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            // ProfileReader の登録解除
            nn::atk::SoundSystem::UnregisterProfileReader(g_ProfileReader);
            nnt::atk::util::AtkCommonSetup::Finalize(allocator);
        }
        void InitializeWithoutSoundSystem(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
        {
            nnt::atk::util::AtkCommonSetup::InitializeSoundHeap(param.GetSoundHeapSize(), allocator);
            nnt::atk::util::AtkCommonSetup::InitializeSoundArchive(param.GetSoundArchivePath(), allocator);
            nnt::atk::util::AtkCommonSetup::InitializeSoundDataManager(allocator);
            nnt::atk::util::AtkCommonSetup::InitializeSoundArchivePlayer(allocator);

        }
        void InitializeSoundSystemImpl(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
        {
            nnt::atk::util::AtkCommonSetup::InitializeSoundSystem(param.GetSoundSystemParam(), allocator);
        }
        void FinalizeWithoutSoundSystem(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
        {
            nnt::atk::util::AtkCommonSetup::FinalizeSoundArchivePlayer(allocator);
            nnt::atk::util::AtkCommonSetup::FinalizeSoundDataManager(allocator);
            nnt::atk::util::AtkCommonSetup::FinalizeSoundArchive(allocator);
            nnt::atk::util::AtkCommonSetup::FinalizeSoundHeap(allocator);
        }
        void FinalizeSoundSystemImpl(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
        {
            nnt::atk::util::AtkCommonSetup::FinalizeSoundSystem(allocator);
        }
    };

    class AtkDetailProfilerSetup : public AtkProfilerSetup
    {
    public:
        AtkDetailProfilerSetup()
        : m_SoundThreadUpdateProfileReaderBuffer(nullptr)
        {
        }

        virtual void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            InitializeImpl(allocator, false, true, RenderingSampleRate_48k);
        }
        void InitializeImpl(nn::mem::StandardAllocator& allocator, bool isVoiceDropEnabled, bool isDetailProfileEnabled, int rendererSampleRate) NN_NOEXCEPT
        {
            //  g_ProfileReader に直前の TEST の結果が入っていることがあるため、一旦全部読みます
            int profileCountMax = static_cast<int>(sizeof(g_SoundProfile) / sizeof(g_SoundProfile[0]));
            g_ProfileReader.Read( g_SoundProfile, profileCountMax );

            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableProfiler = true;
            soundSystemParam.enableVoiceDrop = isVoiceDropEnabled;
            soundSystemParam.enableDetailSoundThreadProfile = isDetailProfileEnabled;
            soundSystemParam.voiceCountMax = nn::atk::SoundSystem::VoiceCountMax;

            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            initializeParam.GetSoundSystemParam().rendererSampleRate = rendererSampleRate;

            // ProfileReader の登録
            nnt::atk::util::AtkCommonSetup::Initialize(initializeParam, allocator);
            nn::atk::SoundSystem::RegisterProfileReader(g_ProfileReader);
            g_PreviousAtkFrameProcessBegin = nn::os::GetSystemTick();

            // SoundThread 用プロファイラの初期化
            size_t bufferSize = m_SoundThreadUpdateProfileReader.GetRequiredMemorySize(32);
            m_SoundThreadUpdateProfileReaderBuffer = allocator.Allocate(bufferSize);
            m_SoundThreadUpdateProfileReader.Initialize(m_SoundThreadUpdateProfileReaderBuffer, bufferSize, 32);
            nn::atk::SoundSystem::RegisterSoundThreadUpdateProfileReader(m_SoundThreadUpdateProfileReader);
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            // SoundThread 用プロファイラの終了
            nn::atk::SoundSystem::UnregisterSoundThreadUpdateProfileReader(m_SoundThreadUpdateProfileReader);
            m_SoundThreadUpdateProfileReader.Finalize();
            allocator.Free(m_SoundThreadUpdateProfileReaderBuffer);

            // ProfileReader の登録解除
            nn::atk::SoundSystem::UnregisterProfileReader(g_ProfileReader);

            nnt::atk::util::AtkCommonSetup::Finalize(allocator);
        }

        nn::atk::SoundThreadUpdateProfileReader& GetSoundThreadUpdateProfileReader()
        {
            return m_SoundThreadUpdateProfileReader;
        }

    private:
        void* m_SoundThreadUpdateProfileReaderBuffer;
        nn::atk::SoundThreadUpdateProfileReader m_SoundThreadUpdateProfileReader;
    };

    FsProfilerSetup g_FsSetup;
    nnt::atk::util::AtkCommonSetup g_AtkCommonSetup;
    AtkProfilerSetup g_AtkSetup;
    AtkDetailProfilerSetup g_AtkDetailProfilerSetup;
}

void WriteProfileLog(nnt::atk::util::FileLogger& logger, nn::atk::SoundProfile& profile)
{
    float voiceFramePerVoice = profile.totalVoiceCount == 0 ? 0.0f : static_cast<float>(profile.voiceProcess.GetSpan().GetMicroSeconds()) / profile.totalVoiceCount;

    nn::Result result = logger.Write("%d,%lld,%lld,%lld,%f,%lld,%lld,%lld,%lld\n",
                profile.totalVoiceCount,
                profile.nwFrameProcess.GetSpan().GetMicroSeconds(),
                profile.rendererFrameProcess.GetSpan().GetMicroSeconds(),
                profile.voiceProcess.GetSpan().GetMicroSeconds(),
                voiceFramePerVoice,
                profile.mainMixProcess.GetSpan().GetMicroSeconds(),
                profile.finalMixProcess.GetSpan().GetMicroSeconds(),
                profile.sinkProcess.GetSpan().GetMicroSeconds(),
                profile.GetNwFrameProcessInterval(g_PreviousAtkFrameProcessBegin).GetMicroSeconds());
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");
}

void WriteSoundThreadUpdateProfileLog(nnt::atk::util::FileLogger& logger, nn::atk::SoundThreadUpdateProfile& profile)
{
    nn::Result result = logger.Write("%lld,%lld,%lld,%lld,%lld,%lld\n",
        profile.soundThreadProcess.GetSpan().GetMicroSeconds(),
        profile._updateLowLevelVoiceProcess.GetSpan().GetMicroSeconds(),
        profile._frameProcess.GetSpan().GetMicroSeconds(),
        profile._userEffectFrameProcess.GetSpan().GetMicroSeconds(),
        profile._updateRendererProcess.GetSpan().GetMicroSeconds(),
        profile._waitRendererEventProcess.GetSpan().GetMicroSeconds());
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");
}

void WaitForEffectClear(nn::atk::EffectBase& effect)
{
    const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(5000);
    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
        nn::os::SleepThread(WaitTime);
        if(effect.IsRemovable()) break;
    }
    EXPECT_TRUE(effect.IsRemovable()) << "Effect clear timed out.";
}

int RecordProfileReaderInfo(nnt::atk::util::FileLogger& fileLogger, nnt::atk::util::ProfileStatistics& statistics)
{
    // ProfileReader に関する情報の記録
    int profileCountMax = static_cast<int>(sizeof(g_SoundProfile) / sizeof(g_SoundProfile[0]));
    int profileCount = g_ProfileReader.Read(g_SoundProfile, profileCountMax);
    EXPECT_GE(profileCount, 0);
    EXPECT_LE(profileCount, profileCountMax);
    for (int j=0; j < profileCount; j++)
    {
        nn::atk::SoundProfile profile = g_SoundProfile[j];

        WriteProfileLog(fileLogger, profile);
        statistics.AddProfile(profile, g_PreviousAtkFrameProcessBegin);

        g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
    }

    return profileCount;
}

int RecordSoundThreadUpdateProfileReaderInfo(AtkDetailProfilerSetup& setup, nnt::atk::util::FileLogger& fileLogger, nnt::atk::util::SoundThreadStatistics& statistics)
{
    // ProfileReader に関する情報の記録
    int profileCountMax = static_cast<int>(sizeof(g_SoundThreadUpdateProfile) / sizeof(g_SoundThreadUpdateProfile[0]));
    int profileCount = setup.GetSoundThreadUpdateProfileReader().Read(g_SoundThreadUpdateProfile, profileCountMax);
    EXPECT_GE(profileCount, 0);
    EXPECT_LE(profileCount, profileCountMax);
    for (int j=0; j < profileCount; j++)
    {
        nn::atk::SoundThreadUpdateProfile profile = g_SoundThreadUpdateProfile[j];

        WriteSoundThreadUpdateProfileLog(fileLogger, profile);
        statistics.AddProfile(profile);
    }

    return profileCount;
}

template<typename Effect>
void ProfileSequenceMarioKart(Effect* pEffect, size_t requiredEffectBufferSize, const char* const effectLabel)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    bool isEffectEnabled = pEffect != nullptr;

    // + 1 は終端文字の分
    char profileResultPath[nn::fs::EntryNameLengthMax + 1];
    if(isEffectEnabled)
    {
        nn::util::SNPrintf(profileResultPath, sizeof(profileResultPath), "%s:/%s/Atk%sProfileMonitor/%s/SeqMarioKart_%s.csv",
            g_FsSetup.TestResultsMountName, nnt::atk::util::GetTargetName(), TestNameSuffix, nnt::atk::util::BuildType, effectLabel);
    }
    else
    {
        nn::util::SNPrintf(profileResultPath, sizeof(profileResultPath), "%s:/%s/Atk%sProfileMonitor/%s/SeqMarioKart_None.csv",
            g_FsSetup.TestResultsMountName, nnt::atk::util::GetTargetName(), TestNameSuffix, nnt::atk::util::BuildType);
    }
    nnt::atk::util::FileLogger profileLogger(profileResultPath);
    NN_ABORT_UNLESS(profileLogger.IsOpened(), "Cannot open file.");

    char soundThreadUpdateProfileResultPath[nn::fs::EntryNameLengthMax + 1];
    if(isEffectEnabled)
    {
        nn::util::SNPrintf(soundThreadUpdateProfileResultPath, sizeof(soundThreadUpdateProfileResultPath), "%s:/%s/Atk%sProfileMonitor/%s/SeqMarioKart_%s_SoundThread.csv",
            g_FsSetup.TestResultsMountName, nnt::atk::util::GetTargetName(), TestNameSuffix, nnt::atk::util::BuildType, effectLabel);
    }
    else
    {
        nn::util::SNPrintf(soundThreadUpdateProfileResultPath, sizeof(soundThreadUpdateProfileResultPath), "%s:/%s/Atk%sProfileMonitor/%s/SeqMarioKart_None_SoundThread.csv",
            g_FsSetup.TestResultsMountName, nnt::atk::util::GetTargetName(), TestNameSuffix, nnt::atk::util::BuildType);
    }
    nnt::atk::util::FileLogger soundThreadUpdateProfileLogger(soundThreadUpdateProfileResultPath);
    NN_ABORT_UNLESS(soundThreadUpdateProfileLogger.IsOpened(), "Cannot open file.");

    g_AtkDetailProfilerSetup.Initialize(g_Allocator);
    g_AtkDetailProfilerSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    void* effectBuffer = nullptr;
    nn::audio::MemoryPoolType memoryPoolForEffect;
    if (isEffectEnabled)
    {
        size_t alignedBufferSize = nn::util::align_up( requiredEffectBufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        effectBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alignedBufferSize, nn::audio::MemoryPoolType::AddressAlignment);

        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForEffect, effectBuffer, alignedBufferSize);
        pEffect->SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, pEffect, effectBuffer, requiredEffectBufferSize));
    }

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkDetailProfilerSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    NN_ABORT_UNLESS(soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART).IsSuccess(), "StartSound(SEQ_MARIOKART) failed.");

    if(isEffectEnabled)
    {
        soundHandle.SetMainSend(-1.0f);
        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
    }

    NN_LOG("Start SEQ_MARIOKART.\n");
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);

    nn::Result result = profileLogger.Write(InitialPerformanceLog);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

    result = soundThreadUpdateProfileLogger.Write(InitialSoundThreadPerformanceLog);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot write file.");

    int totalProfileCount = 0;
    int totalSoundThreadProfileCount = 0;

    // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
    nnt::atk::util::ProfileStatistics profileStatistics(1);
    nnt::atk::util::SoundThreadStatistics soundThreadStatistics(0);

    // SEQ_MARIOKART のイントロ部分を再生し終えるまで回す (SEQ_MARIOKART のタイムベースは 96)
    while(sequenceSoundHandle.GetTick() <= 8 * 96 * 4)
    {
        nnt::atk::util::UpdateAndWait(soundArchivePlayer);

        totalProfileCount += RecordProfileReaderInfo(profileLogger, profileStatistics);
        totalSoundThreadProfileCount += RecordSoundThreadUpdateProfileReaderInfo(g_AtkDetailProfilerSetup, soundThreadUpdateProfileLogger, soundThreadStatistics);
    }

    EXPECT_NE(0, totalProfileCount);

    char labelName[nn::fs::EntryNameLengthMax];
    if(isEffectEnabled)
    {
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "SeqMarioKart%s%s", TestNameSuffix, effectLabel);
    }
    else
    {
        nn::util::SNPrintf(labelName, sizeof(labelName) - 1, "SeqMarioKart%s", TestNameSuffix);
    }
    profileStatistics.PrintMeanProfile(labelName);
    profileStatistics.PrintMaxProfile(labelName);

    const std::string TestName = "SoundProfileTest";
    char statisticsLabel[nn::fs::EntryNameLengthMax];
    if(isEffectEnabled)
    {
        nn::util::SNPrintf(statisticsLabel, sizeof(statisticsLabel) - 1, "Atk_%s_SeqMarioKart_%s", (TestName + TestNameSuffix).c_str(), effectLabel);
    }
    else
    {
        nn::util::SNPrintf(statisticsLabel, sizeof(statisticsLabel) - 1, "Atk_%s_SeqMarioKart_None", (TestName + TestNameSuffix).c_str());
    }

    soundThreadStatistics.PrintMeanProfile(statisticsLabel);
    soundThreadStatistics.PrintMaxProfile(statisticsLabel);

    if(isEffectEnabled)
    {
        pEffect->SetEnabled(false);
        WaitForEffectClear(*pEffect);

        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForEffect);
        g_Allocator.Free(effectBuffer);
    }

    g_AtkDetailProfilerSetup.Finalize(g_Allocator);
    profileLogger.Close();
    soundThreadUpdateProfileLogger.Close();
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

TEST(AudioProfile, LogSeqMarioKartProfileTest)
{
    nnt::atk::util::OnPreAtkTest();

    // ビルドを通すために、ダミーで nn::atk::EffectDelay を利用
    ProfileSequenceMarioKart<nn::atk::EffectDelay>(nullptr, 0, "");
}

TEST(AudioProfile, LogSeqMarioKartWithDelayProfileTest)
{
    nnt::atk::util::OnPreAtkTest();

    nn::atk::EffectDelay delay;
    delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
    delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_48000);
    ProfileSequenceMarioKart(&delay, delay.GetRequiredMemSize(), "Delay");
}

TEST(AudioProfile, LogSeqMarioKartWithReverbProfileTest)
{
    nnt::atk::util::OnPreAtkTest();

    nn::atk::EffectReverb reverb;
    reverb.SetChannelMode(nn::atk::EffectReverb::ChannelMode_2Ch);
    reverb.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000);
    ProfileSequenceMarioKart(&reverb, reverb.GetRequiredMemSize(), "Reverb");
}

TEST(AudioProfile, LogSeqMarioKartWithI3dl2ReverbProfileTest)
{
    nnt::atk::util::OnPreAtkTest();

    nn::atk::EffectI3dl2Reverb reverb;
    reverb.SetChannelMode(nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch);
    reverb.SetSampleRate(nn::atk::EffectI3dl2Reverb::SampleRate_48000);
    ProfileSequenceMarioKart(&reverb, reverb.GetRequiredMemSize(), "I3dl2Reverb");

}

TEST(AudioProfile, LogMemorySizeForSoundArchivePlayerTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    const std::string TestName = "MemorySizeForSoundArchivePlayerTest";
    const int StreamCacheBufferSize = 128 * 1024; // AtkStreamSound サンプルと同じ値を使用

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // FsSoundArchive
    nn::atk::FsSoundArchive fsSoundArchive;
    char soundArchivePath[128];
    NN_ABORT_UNLESS(nnt::atk::util::GetCommonSoundArchivePath(soundArchivePath, sizeof(soundArchivePath), nnt::atk::util::FsCommonSetup::RomMountName), "Cannot get sound archive path.");
    NN_ABORT_UNLESS(fsSoundArchive.Open(soundArchivePath), "cannot open SoundArchive(%s)\n", soundArchivePath);
    size_t memSizeForfsSoundArchiveInfoBlockSize = fsSoundArchive.GetHeaderSize();
    void* pMemoryForInfoBlock = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForfsSoundArchiveInfoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize);
    NN_ABORT_UNLESS(fsSoundArchive.LoadHeader(pMemoryForInfoBlock, memSizeForfsSoundArchiveInfoBlockSize), "cannot load FsSoundArchive InfoBlock");
    // SoundDataManager
    nn::atk::SoundDataManager soundDataManager;
    size_t memSizeForSoundDataManager = soundDataManager.GetRequiredMemSize(&fsSoundArchive);
    // SoundArchivePlayer
    nn::atk::SoundArchivePlayer soundArchivePlayer;
    size_t memSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize(&fsSoundArchive);
    size_t memSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize(&fsSoundArchive);
    size_t memSizeForStreamCacheBuffer = soundArchivePlayer.GetRequiredStreamCacheSize(&fsSoundArchive, StreamCacheBufferSize);
    size_t memSizeForStreamInstance = soundArchivePlayer.GetRequiredStreamInstanceSize(&fsSoundArchive);
    size_t memSizeForStreamSoundHeader = nn::atk::SoundArchivePlayer::GetRequiredWorkBufferSizeToReadStreamSoundHeader();

    // 各バッファサイズのダンプ
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-FsSoundArchiveBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForfsSoundArchiveInfoBlockSize);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-SoundDataManagerBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForSoundDataManager);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-SoundArchivePlayerBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForSoundArchivePlayer);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-StreamBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForStreamBuffer);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-StreamCacheSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForStreamCacheBuffer);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-StreamInstanceSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForStreamInstance);
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-StreamSoundHeaderSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), memSizeForStreamSoundHeader);

    fsSoundArchive.Close();
    g_Allocator.Free(pMemoryForInfoBlock);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(AudioProfile, LogMemorySizeForSoundSystemTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam param;
    size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
    void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);
    nn::atk::SoundSystem::Initialize(param, reinterpret_cast<uintptr_t>(pMemoryForSoundSystem), memSizeForSoundSystem);

    // 各バッファサイズのダンプ
    NN_LOG("##teamcity[buildStatisticValue key='AudioRendererBufferSize%s(byte)' value='%zu']\n",      TestNameSuffix, nn::atk::SoundSystem::GetAudioRendererBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='NnVoiceAllocatorBufferSize%s(byte)' value='%zu']\n",   TestNameSuffix, nn::atk::SoundSystem::GetLowLevelVoiceAllocatorBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='MultiVoiceManagerBufferSize%s(byte)' value='%zu']\n",  TestNameSuffix, nn::atk::SoundSystem::GetMultiVoiceManagerBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='ChannelManagerBufferSize%s(byte)' value='%zu']\n",     TestNameSuffix, nn::atk::SoundSystem::GetChannelManagerBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='SoundThreadStackSize%s(byte)' value='%zu']\n",         TestNameSuffix, param.soundThreadStackSize);
    NN_LOG("##teamcity[buildStatisticValue key='TaskThreadStackSize%s(byte)' value='%zu']\n",          TestNameSuffix, param.taskThreadStackSize);
    NN_LOG("##teamcity[buildStatisticValue key='PerformanceFrameBufferSize%s(byte)' value='%zu']\n",   TestNameSuffix, nn::atk::SoundSystem::GetPerformanceFrameBufferSize() * nn::atk::detail::MaxPerformanceBufferCount);
    NN_LOG("##teamcity[buildStatisticValue key='SoundThreadCommandTotalBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, nn::atk::SoundSystem::GetSoundThreadCommandTotalBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='SoundThreadCommandBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, param.soundThreadCommandBufferSize);
    NN_LOG("##teamcity[buildStatisticValue key='TaskThreadCommandTotalBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, nn::atk::SoundSystem::GetTaskThreadCommandTotalBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='TaskThreadCommandBufferSize%s(byte)' value='%zu']\n",  TestNameSuffix, param.taskThreadCommandBufferSize);
#ifdef NNT_ATK_ENABLE_VOICE_COMMAND
    const int voiceCommandManagerCount = 2;
    NN_LOG("##teamcity[buildStatisticValue key='LowLevelVoiceCommandTotalBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, nn::atk::SoundSystem::GetLowLevelVoiceCommandTotalBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='VoiceReplyCommandTotalBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, nn::atk::SoundSystem::GetVoiceReplyCommandTotalBufferSize());
    NN_LOG("##teamcity[buildStatisticValue key='VoiceCommandBufferSize%s(byte)' value='%zu']\n",       TestNameSuffix, param.voiceCommandBufferSize * voiceCommandManagerCount);
#endif

    // 合計のバッファサイズのダンプ
    size_t totalBufferSize = 0;
    totalBufferSize += nn::atk::SoundSystem::GetAudioRendererBufferSize();
    totalBufferSize += nn::atk::SoundSystem::GetLowLevelVoiceAllocatorBufferSize();
    totalBufferSize += nn::atk::SoundSystem::GetMultiVoiceManagerBufferSize();
    totalBufferSize += nn::atk::SoundSystem::GetChannelManagerBufferSize();
    totalBufferSize += param.soundThreadStackSize;
    totalBufferSize += param.taskThreadStackSize;
    totalBufferSize += nn::atk::SoundSystem::GetPerformanceFrameBufferSize() * nn::atk::detail::MaxPerformanceBufferCount;
    totalBufferSize += nn::atk::SoundSystem::GetSoundThreadCommandTotalBufferSize();
    totalBufferSize += nn::atk::SoundSystem::GetTaskThreadCommandTotalBufferSize();
#ifdef NNT_ATK_ENABLE_VOICE_COMMAND
    totalBufferSize += nn::atk::SoundSystem::GetLowLevelVoiceCommandTotalBufferSize() + nn::atk::SoundSystem::GetVoiceReplyCommandTotalBufferSize();
#endif
    NN_LOG("##teamcity[buildStatisticValue key='TotalBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, totalBufferSize);
    NN_LOG("##teamcity[buildStatisticValue key='RequiredBufferSize%s(byte)' value='%zu']\n", TestNameSuffix, memSizeForSoundSystem);

    nn::atk::SoundSystem::Finalize();

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(AudioProfile, LogMemorySizeForSubMixTest)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // サブミックスの違いによるバッファサイズの違いのテスト
    const std::string TestName = "MemorySizeForSubMixTest";
    const int FinalMixChannelCount = 6;
    const int FinalMixBusCount = 1;
    {
        // デフォルト構成 (非カスタム)
        const char* itemLabel= "DefaultSubMix";
        nn::atk::SoundSystem::SoundSystemParam param;
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = 0;
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
    }
    {
        // デフォルト(6ch x 4bus) 相当のカスタム構成
        const char* itemLabel= "DefaultCustomSubMix";
        const int SrcBusCount = 4;
        const int SrcChannelCount = 6;
        const int SubMixCount = 1;
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCustomSubMix = true;
        param.subMixTotalChannelCount = SrcBusCount * SrcChannelCount;
        param.subMixCount = SubMixCount;
        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        atkParam.SetSoundSystemParam(param);
        g_AtkCommonSetup.Initialize(atkParam, g_Allocator);
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = nn::atk::SubMix::GetRequiredMemorySize(SrcBusCount, SrcChannelCount, FinalMixBusCount, FinalMixChannelCount);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
        g_AtkCommonSetup.Finalize(g_Allocator);
    }
    {
        // 24ch x 1bus のカスタム構成
        const char* itemLabel= "24ch1BusCustomSubMix";
        const int SrcBusCount = 1;
        const int SrcChannelCount = 24;
        const int SubMixCount = 1;
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCustomSubMix = true;
        param.subMixTotalChannelCount = SrcBusCount * SrcChannelCount;
        param.subMixCount = SubMixCount;
        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        atkParam.SetSoundSystemParam(param);
        g_AtkCommonSetup.Initialize(atkParam, g_Allocator);
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = nn::atk::SubMix::GetRequiredMemorySize(SrcBusCount, SrcChannelCount, FinalMixBusCount, FinalMixChannelCount);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
        g_AtkCommonSetup.Finalize(g_Allocator);
    }
    {
        // 1ch x 24bus のカスタム構成
        const char* itemLabel= "1ch24BusCustomSubMix";
        const int SrcBusCount = 24;
        const int SrcChannelCount = 1;
        const int SubMixCount = 1;
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCustomSubMix = true;
        param.subMixTotalChannelCount = SrcBusCount * SrcChannelCount;
        param.subMixCount = SubMixCount;
        param.busCountMax = nn::atk::OutputReceiver::BusCountMax;
        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        atkParam.SetSoundSystemParam(param);
        g_AtkCommonSetup.Initialize(atkParam, g_Allocator);
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = nn::atk::SubMix::GetRequiredMemorySize(SrcBusCount, SrcChannelCount, FinalMixBusCount, FinalMixChannelCount);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
        g_AtkCommonSetup.Finalize(g_Allocator);
    }
    {
        // SE用追加サブミックス + BGM用サブミックス (非カスタム)
        const char* itemLabel = "AdditionalSubMix";
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableAdditionalEffectBus = true;
        param.enableAdditionalSubMix = true;
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = 0;
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
    }
    {
        // SE用追加サブミックス(6ch x 1bus) + BGM用サブミックス(2ch x 3bus) 相当のカスタム構成
        const char* itemLabel= "AdditionalCustomSubMix";
        const int SeSubMixBusCount = 4;
        const int SeSubMixChannelCount = 6;
        const int SePostSubMixBusCount = 1;
        const int SePostSubMixChannelCount = 6;
        const int BgmSubMixBusCount = 3;
        const int BgmSubMixChannelCount = 2;
        const int SubMixCount = 3;
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCustomSubMix = true;
        param.subMixTotalChannelCount = (SeSubMixBusCount * SeSubMixChannelCount) + (SePostSubMixBusCount * SePostSubMixChannelCount) + (BgmSubMixBusCount * BgmSubMixChannelCount);
        param.subMixCount = SubMixCount;
        nnt::atk::util::AtkCommonSetup::InitializeParam atkParam;
        atkParam.SetSoundSystemParam(param);
        g_AtkCommonSetup.Initialize(atkParam, g_Allocator);
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSeSubMix = nn::atk::SubMix::GetRequiredMemorySize(SeSubMixBusCount, SeSubMixChannelCount, SePostSubMixBusCount, SePostSubMixChannelCount);
        size_t memSizeForSePostSubMix = nn::atk::SubMix::GetRequiredMemorySize(SePostSubMixBusCount, SePostSubMixChannelCount, FinalMixBusCount, FinalMixChannelCount);
        size_t memSizeForBgmSubMix = nn::atk::SubMix::GetRequiredMemorySize(BgmSubMixBusCount, BgmSubMixChannelCount, FinalMixBusCount, FinalMixChannelCount);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSeSubMix + memSizeForSePostSubMix + memSizeForBgmSubMix);
        g_AtkCommonSetup.Finalize(g_Allocator);
    }
    {
        // ファイナルミックスのみのカスタム構成
        const char* itemLabel= "FinalMixOnly";
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCustomSubMix = true;
        param.subMixTotalChannelCount = 0;
        param.subMixCount = 0;
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        size_t memSizeForSubMix = 0;
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SoundSystemBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSoundSystem);
        NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_SubMixBufferSize(byte)' value='%zu']\n", (TestName + TestNameSuffix).c_str(), itemLabel, memSizeForSubMix);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

template<typename Effect>
void PlayableCountCheck(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize, CheckTestConfig config)
{
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    g_AtkSetup.InitializeImpl( g_Allocator, config.rendererSampleRate );
    g_AtkSetup.LoadData( itemId, itemLabel );

    // エフェクトが指定されているときはエフェクトを追加
    nn::audio::MemoryPoolType memoryPoolForEffect;
    void* effectBuffer = nullptr;
    bool isEffectEnabled = pEffect != nullptr;
    if ( isEffectEnabled )
    {
        effectBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, requiredEffectBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForEffect, effectBuffer, requiredEffectBufferSize);
        pEffect->SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, pEffect, effectBuffer, requiredEffectBufferSize));
    }

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    for ( ;; )
    {
        if ( !soundArchivePlayer.StartSound( &soundHandle, itemId ).IsSuccess() )
        {
            NN_LOG( "StartSound(%s) failed.\n", itemLabel );
            FAIL();
            break;
        }

        ApplyCheckTestConfig(soundHandle, config);

        if ( isEffectEnabled )
        {
            soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        }

        nnt::atk::util::UpdateAndWait(soundArchivePlayer);
        int profileCountMax = static_cast<int>(sizeof( g_SoundProfile ) / sizeof( g_SoundProfile[0] ));
        int profileCount = g_ProfileReader.Read( g_SoundProfile, profileCountMax );
        if ( profileCount > 0 )
        {
            bool isOverThreshold = false;
            int profileIndex = 0;
            for ( auto i = 0; i < profileCount; ++i )
            {
                // オーディオフレームを使い切ることはできないため、5000 us の 6 割の値である 3000us と比較しています（6割に根拠はありません）
                // ボイスドロップによる無限ループの可能性を排除するため、ボイスの最大値でも制限をかけています
                if ( g_SoundProfile[i].rendererFrameProcess.GetSpan().GetMicroSeconds() > ProcessTimeThreshold ||
                     g_SoundProfile[i].totalVoiceCount >= static_cast<uint32_t>(nn::atk::SoundSystem::GetVoiceCountMax()) )
                {
                    isOverThreshold = true;
                    profileIndex = i;
                    break;
                }
            }

            if ( isOverThreshold )
            {
                NN_LOG( "##teamcity[buildStatisticValue key='TotalVoiceCount_%s_PlayableCountCheck%s(num)' value='%d']\n", itemLabel, TestNameSuffix, g_SoundProfile[profileIndex].totalVoiceCount );
                NN_LOG( "##teamcity[buildStatisticValue key='RendererFrameProcess_%s_PlayableCountCheck%s(us)' value='%d']\n", itemLabel, TestNameSuffix, g_SoundProfile[profileIndex].rendererFrameProcess.GetSpan().GetMicroSeconds() );
                break;
            }

        }
    }

    if ( isEffectEnabled )
    {
        pEffect->SetEnabled(false);
        WaitForEffectClear(*pEffect);

        nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForEffect);

        g_Allocator.Free(effectBuffer);
    }

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

template<typename Effect>
void PlayableCountCheck(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize)
{
    CheckTestConfig config;
    PlayableCountCheck<Effect>(itemId, itemLabel, pEffect, requiredEffectBufferSize, config);
}

TEST( AudioProfile, LogPlayableCountCheckTest )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    // エフェクトなし（ビルドを通すために、ダミーとして nn::atk::EffectDelay の型を使っています）
    PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP", nullptr, 0);
    PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16", nullptr, 0);
    // エフェクトなし（32kレンダリング）
    {
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_REN32K", nullptr, 0, config);
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_REN32K", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ増加）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMax();
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMAX", nullptr, 0, config);
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMAX", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ減少）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMin();
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMIN", nullptr, 0, config);
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMIN", nullptr, 0, config);
    }
    // エフェクトなし（LPF）
    {
        CheckTestConfig config;
        config.isHandleLpfEnabled = true;
        config.handleLpf = -1.0f;
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_LPF", nullptr, 0, config);
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_LPF", nullptr, 0, config);
    }
    // エフェクトなし（Biquad512）
    {
        CheckTestConfig config;
        config.isHandleBiquadEnabled = true;
        config.handleBiquadFilterType = nn::atk::BiquadFilterType_BandPassFilter512;
        config.handleBiquadFilterValue = 1.0f;
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_BIQUAD512", nullptr, 0, config);
        PlayableCountCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_BIQUAD512", nullptr, 0, config);
    }

    // Delay 追加
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
        delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_48000);
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP", &delay, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_48000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16", &delay, alignedEffectBufferSize );
    }
    // Delay 追加（32kレンダリング）
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
        delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_32000);
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP_REN32K", &delay, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_32000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16_REN32K", &delay, alignedEffectBufferSize, config );
    }

    // Reverb 追加
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP", &reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16", &reverb, alignedEffectBufferSize );
    }
    // Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP_REN32K", &reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16_REN32K", &reverb, alignedEffectBufferSize, config );
    }

    // I3dl2 Reverb 追加
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP", &i3dl2Reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        PlayableCountCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16", &i3dl2Reverb, alignedEffectBufferSize );
    }
    // I3dl2 Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        PlayableCountCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
} // NOLINT(impl/function_size)

template<typename Effect>
void LoadCheck( nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize, CheckTestConfig config )
{
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    g_AtkSetup.InitializeImpl( g_Allocator, config.rendererSampleRate );
    g_AtkSetup.LoadData( itemId, itemLabel );

    // エフェクトが指定されているときはエフェクトを追加
    nn::audio::MemoryPoolType memoryPoolForEffect;
    void* effectBuffer = nullptr;
    bool isEffectEnabled = pEffect != nullptr;
    if ( isEffectEnabled )
    {
        effectBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, requiredEffectBufferSize, nn::audio::MemoryPoolType::AddressAlignment );
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPoolForEffect, effectBuffer, requiredEffectBufferSize );
        pEffect->SetEnabled( true );
        EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, pEffect, effectBuffer, requiredEffectBufferSize ) );
    }

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    // 50 ボイス同時再生
    for ( int voiceCount = 0; voiceCount < 50; ++voiceCount )
    {
        if ( !soundArchivePlayer.StartSound( &soundHandle, itemId ).IsSuccess() )
        {
            NN_LOG( "StartSound(%s) failed.\n", itemLabel );
            FAIL();
            break;
        }

        ApplyCheckTestConfig(soundHandle, config);

        if ( isEffectEnabled )
        {
            soundHandle.SetEffectSend( nn::atk::AuxBus_A, 1.0f );
        }
    }

    // 最初の 1 サンプル目は GetNwFrameProcessInterval() が異常値を示すため除外
    nnt::atk::util::ProfileStatistics statistics( 1 );
    int totalProfileCount = 0;

    // 100 ゲームフレームほど再生することで、安定した計測結果となることを期待しています。
    for ( int frameCount = 0; frameCount < 100; ++frameCount )
    {
        nnt::atk::util::UpdateAndWait(soundArchivePlayer);
        int profileCountMax = static_cast<int>(sizeof( g_SoundProfile ) / sizeof( g_SoundProfile[0] ));
        int profileCount = g_ProfileReader.Read( g_SoundProfile, profileCountMax );
        EXPECT_GE( profileCount, 0 );
        EXPECT_LE( profileCount, profileCountMax );
        for ( int j = 0; j<profileCount; j++ )
        {
            nn::atk::SoundProfile profile = g_SoundProfile[j];

            statistics.AddProfile( profile, g_PreviousAtkFrameProcessBegin );

            g_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
        }

        totalProfileCount += profileCount;
    }

    EXPECT_NE( 0, totalProfileCount );

    char labelName[nn::fs::EntryNameLengthMax];
    nn::util::SNPrintf( labelName, sizeof( labelName ) - 1, "LoadCheck_%s_%s", itemLabel, TestNameSuffix );

    statistics.PrintMeanProfile( labelName );
    statistics.PrintMaxProfile( labelName );

    if ( isEffectEnabled )
    {
        pEffect->SetEnabled( false );
        WaitForEffectClear( *pEffect );

        nn::atk::SoundSystem::ClearEffect( nn::atk::AuxBus_A );
        nn::atk::SoundSystem::DetachMemoryPool( &memoryPoolForEffect );

        g_Allocator.Free( effectBuffer );
    }

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

template<typename Effect>
void LoadCheck(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize)
{
    CheckTestConfig config;
    LoadCheck<Effect>(itemId, itemLabel, pEffect, requiredEffectBufferSize, config);
}

TEST( AudioProfile, LogLoadCheckTest )
{
    nnt::atk::util::OnPreAtkTest();
    // エフェクトなし（ビルドを通すために、ダミーとして nn::atk::EffectDelay の型を使っています）
    LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "SE_SIN440_LOOP", nullptr, 0 );
    LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16", nullptr, 0 );
    // エフェクトなし（32kレンダリング）
    {
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_REN32K", nullptr, 0, config);
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_REN32K", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ増加）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMax();
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMAX", nullptr, 0, config);
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMAX", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ減少）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMin();
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMIN", nullptr, 0, config);
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMIN", nullptr, 0, config);
    }
    // エフェクトなし（LPF）
    {
        CheckTestConfig config;
        config.isHandleLpfEnabled = true;
        config.handleLpf = -1.0f;
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_LPF", nullptr, 0, config);
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_LPF", nullptr, 0, config);
    }
    // エフェクトなし（Biquad512）
    {
        CheckTestConfig config;
        config.isHandleBiquadEnabled = true;
        config.handleBiquadFilterType = nn::atk::BiquadFilterType_BandPassFilter512;
        config.handleBiquadFilterValue = 1.0f;
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_BIQUAD512", nullptr, 0, config);
        LoadCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_BIQUAD512", nullptr, 0, config);
    }

    // Delay 追加
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_48000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP", &delay, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_48000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16", &delay, alignedEffectBufferSize );
    }
    // Delay 追加（32kレンダリング）
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_32000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP_REN32K", &delay, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_32000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16_REN32K", &delay, alignedEffectBufferSize, config );
    }

    // Reverb 追加
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP", &reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16", &reverb, alignedEffectBufferSize );
    }
    // Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP_REN32K", &reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16_REN32K", &reverb, alignedEffectBufferSize, config );
    }

    // I3dl2 Reverb 追加
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP", &i3dl2Reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        LoadCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16", &i3dl2Reverb, alignedEffectBufferSize );
    }
    // I3dl2 Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );

        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        LoadCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
} // NOLINT(impl/function_size)

#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_SPEC_NX)
void WaitFrameProcess() NN_NOEXCEPT
{
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
    nn::os::SleepThread(WaitTime);
}

// ボイスの発音数が playSoundCount と同数になるのを待ちます。
// 十分な時間を待っても同数にならなかった場合は false を返します。
bool WaitVoicePlay(int playSoundCount) NN_NOEXCEPT
{
    const auto loopFrame = GetTestFrame(nn::TimeSpan::FromSeconds(1));
    for (auto elapsedFrame = 0; elapsedFrame <= loopFrame; ++elapsedFrame)
    {
        while (g_ProfileReader.Read(g_SoundProfile, 1) > 0)
        {
            if (g_SoundProfile[0].totalVoiceCount == playSoundCount)
            {
                return true;
            }
        }
        WaitFrameProcess();
    }
    return false;
}

// ボイスドロップが発生し playSoundCount を下回らないかを確認するため、十分な時間待ちます。再生を維持出来たボイス数を返します。
int WaitVoiceDrop(int playSoundCount) NN_NOEXCEPT
{
    const auto loopFrame = GetTestFrame(nn::TimeSpan::FromMilliSeconds(200));
    const int ProfileCountMax = static_cast<int>(sizeof(g_SoundProfile) / sizeof(g_SoundProfile[0]));
    for (auto elapsedFrame = 0; elapsedFrame <= loopFrame; ++elapsedFrame)
    {
        int profileCount = g_ProfileReader.Read(g_SoundProfile, ProfileCountMax);
        for (auto i = 0; i < profileCount; ++i)
        {
            if (g_SoundProfile[i].totalVoiceCount < playSoundCount)
            {
                return g_SoundProfile[i].totalVoiceCount;
            }
        }
        WaitFrameProcess();
    }
    return playSoundCount;
}

// ボイスの発音数が 0 以上になるのを待ちます。
// 十分な時間を待っても 0 以上にならなかった場合は false を返します。
bool WaitVoicePlay() NN_NOEXCEPT
{
    const auto loopFrame = GetTestFrame(nn::TimeSpan::FromSeconds(1));
    for (auto elapsedFrame = 0; elapsedFrame <= loopFrame; ++elapsedFrame)
    {
        while (g_ProfileReader.Read(g_SoundProfile, 1) > 0)
        {
            if (g_SoundProfile[0].totalVoiceCount != 0)
            {
                return true;
            }
        }
        WaitFrameProcess();
    }
    return false;
}

TEST(AudioProfile, VoiceDropTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    const std::string TestName = "VoiceDropTest";

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // ボイスドロップの有効化
    g_AtkSetup.InitializeImpl(g_Allocator, true);

    g_AtkSetup.LoadData(SE_SIN440_LOOP, "SE_SIN440_LOOP");
    g_AtkSetup.LoadData(SE_SIN440_LOOP_LOWPRIORITY, "SE_SIN440_LOOP_LOWPRIORITY");

    int playSoundCount = 0;
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    // ボイスドロップテストのチェックを開始するボイス数を、ボイスを上限まで一斉再生した後に残ったボイスの数とする。
    for (int i = 0; i < nn::atk::SoundSystem::GetVoiceCountMax(); ++i)
    {
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_SIN440_LOOP).IsSuccess());
        playSoundCount++;
    }
    soundArchivePlayer.Update();
    WaitVoicePlay();
    for(;;)
    {
        auto resultSoundCount = WaitVoiceDrop(playSoundCount);
        if (playSoundCount == resultSoundCount)
        {
            break;
        }
        playSoundCount = resultSoundCount;
    }

    // ボイスドロップが発生するまで、ループ波形を１つずつ再生していく
    for ( ;; )
    {
        // モノラル波形の再生要求を行い、サウンド再生要求数とボイス数が同数になるまで待つ
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_SIN440_LOOP).IsSuccess());
        playSoundCount++;
        soundArchivePlayer.Update();
        NN_LOG("[%s] Start SE_SIN440_LOOP [soundCount:%2d]\n", (TestName + TestNameSuffix).c_str(), playSoundCount);
        WaitVoicePlay(playSoundCount);

        // ボイスがドロップしないかを確認するため待つ
        int resultWaitDrop = WaitVoiceDrop(playSoundCount);
        if (resultWaitDrop != playSoundCount)
        {
            soundArchivePlayer.Update(); // BasicSound::Finalize() が行われるよう、アーカイブプレイヤーを更新
            playSoundCount = resultWaitDrop;
            NN_LOG("[%s] voiceDrop occured [voiceCount:%2d]\n", (TestName + TestNameSuffix).c_str(), playSoundCount);
            break;
        }
    }
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s_VoiceDroppingCount' value='%d']\n", (TestName + TestNameSuffix).c_str(), playSoundCount);

    // 低優先度のサウンドの再生要求を行い、サウンド再生要求数とボイス数が同数になるまで待つ
    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_SIN440_LOOP_LOWPRIORITY).IsSuccess());
    playSoundCount++;
    soundArchivePlayer.Update();
    NN_LOG("[%s] Start SE_SIN440_LOOP_LOWPRIORITY [soundCount:%2d]\n", (TestName + TestNameSuffix).c_str(), playSoundCount);
    WaitVoicePlay(playSoundCount);

    // ボイスドロップを待ち、低優先度のサウンドが優先度負けで停止している事を確認する
    int resultWaitDrop = WaitVoiceDrop(playSoundCount);
    NN_ABORT_UNLESS_NOT_EQUAL(resultWaitDrop, playSoundCount);
    soundArchivePlayer.Update(); // BasicSound::Finalize() が行われるよう、アーカイブプレイヤーを更新
    NN_LOG("[%s] voiceDrop occured [voiceCount:%2d]\n", (TestName + TestNameSuffix).c_str(), resultWaitDrop);
    EXPECT_FALSE(soundHandle.IsAttachedSound());

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

void CountUpDroppedLowLevelVoiceProcess(uintptr_t ptr)
{
    int* pDroppedVoiceCount = reinterpret_cast<int*>(ptr);
    *pDroppedVoiceCount += nn::atk::SoundSystem::GetDroppedLowLevelVoiceCount();
}

TEST(AudioProfile, DroppedLowLevelVoiceCountTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    const std::string TestName = "DroppedLowLevelVoiceCountTest";

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // ボイスドロップの有効化、ボイスドロップを早く起こすためのレンダリングリミット設定
    g_AtkSetup.InitializeImpl(g_Allocator, true);
    NN_ABORT_UNLESS(nn::atk::detail::driver::HardwareManager::GetInstance().SetAudioRendererRenderingTimeLimit(15).IsSuccess());

    // ドロップしたボイスの総数をカウントするためのコールバックを設定
    int droppedVoiceCount = 0;
    nn::atk::SoundSystem::SetSoundThreadEndUserCallback(CountUpDroppedLowLevelVoiceProcess, reinterpret_cast<uintptr_t>(&droppedVoiceCount));

    g_AtkSetup.LoadData(SE_SIN440_LOOP, "SE_SIN440_LOOP");

    const int PlayRequestCountMax = 30;
    int playRequestCount = 0;
    int playingCount = 0;
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    while (playRequestCount < PlayRequestCountMax)
    {
        // モノラル波形の再生要求を行い、再生数とボイス数が同数になるまで待つ
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_SIN440_LOOP).IsSuccess());
        playRequestCount++;
        playingCount++;
        soundArchivePlayer.Update();
        WaitVoicePlay(playingCount);

        // ボイスがドロップしないかを確認するため待つ
        int resultWaitDrop = WaitVoiceDrop(playingCount);
        if (resultWaitDrop != playingCount)
        {
            soundArchivePlayer.Update(); // BasicSound::Finalize() が行われるよう、アーカイブプレイヤーを更新
            playingCount = resultWaitDrop;
            NN_LOG("[%s] voiceDrop occured [playingCount:%2d]\n", (TestName + TestNameSuffix).c_str(), playingCount);
        }

        // 再生数とドロップ数の和が、再生要求数と同じになる事を確認
        NN_LOG("[%s] [playingCount:%2d][droppedCount:%2d][requestCount:%2d]\n",
            (TestName + TestNameSuffix).c_str(), playingCount, droppedVoiceCount, playRequestCount);
        EXPECT_EQ(playingCount + droppedVoiceCount, playRequestCount);
    }

    nn::atk::SoundSystem::ClearSoundThreadEndUserCallback();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

template<typename Effect>
void VoiceDropCheck(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize, CheckTestConfig config) NN_NOEXCEPT
{
    const std::string TestName = "VoiceDropCheckTest";

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    // ボイスドロップの有効化、レンダリングサンプルレートの設定
    g_AtkSetup.InitializeImpl(g_Allocator, true, config.rendererSampleRate);
    g_AtkSetup.LoadData(itemId, itemLabel);

    // エフェクトが指定されているときはエフェクトを追加
    nn::audio::MemoryPoolType memoryPoolForEffect;
    void* effectBuffer = nullptr;
    bool isEffectEnabled = pEffect != nullptr;
    if ( isEffectEnabled )
    {
        effectBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, requiredEffectBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForEffect, effectBuffer, requiredEffectBufferSize);
        pEffect->SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, pEffect, effectBuffer, requiredEffectBufferSize));
    }

    int playSoundCount = 0;
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    // ボイスドロップテストのチェックを開始するボイス数を、ボイスを上限まで一斉再生した後に残ったボイスの数とする。
    for (int i = 0; i < nn::atk::SoundSystem::GetVoiceCountMax(); ++i)
    {
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, itemId).IsSuccess());
        // テスト設定とエフェクトの適用
        ApplyCheckTestConfig(soundHandle, config);
        if (isEffectEnabled)
        {
            soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        }
        playSoundCount++;
    }
    soundArchivePlayer.Update();
    WaitVoicePlay();
    for(;;)
    {
        auto resultSoundCount = WaitVoiceDrop(playSoundCount);
        if (resultSoundCount == playSoundCount)
        {
            break;
        }
        playSoundCount = resultSoundCount;
    }

    // ボイスドロップが発生するまでループ波形を１つずつ再生していく
    for ( ;; )
    {
        // モノラル波形の再生要求を行い、サウンド再生要求数とボイス数が同数になるまで待つ
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, itemId).IsSuccess());
        // テスト設定とエフェクトの適用
        ApplyCheckTestConfig(soundHandle, config);
        if (isEffectEnabled)
        {
            soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        }
        playSoundCount++;
        soundArchivePlayer.Update();
        NN_LOG("[%s] Start %s [soundCount:%2d]\n", itemLabel, (TestName + TestNameSuffix).c_str(), playSoundCount);
        WaitVoicePlay(playSoundCount);

        // ボイスがドロップしないかを確認するため待つ
        int resultWaitDrop = WaitVoiceDrop(playSoundCount);
        if (resultWaitDrop != playSoundCount)
        {
            soundArchivePlayer.Update(); // BasicSound::Finalize() が行われるよう、アーカイブプレイヤーを更新
            playSoundCount = resultWaitDrop;
            NN_LOG("[%s] voiceDrop occured [voiceCount:%2d]\n", (TestName + TestNameSuffix).c_str(), playSoundCount);
            break;
        }
    }
    NN_LOG("##teamcity[buildStatisticValue key='Atk_%s-%s_VoiceDroppingCount' value='%d']\n", (TestName + TestNameSuffix).c_str(), itemLabel, playSoundCount);

    if ( isEffectEnabled )
    {
        pEffect->SetEnabled( false );
        WaitForEffectClear( *pEffect );
        nn::atk::SoundSystem::ClearEffect( nn::atk::AuxBus_A );
        nn::atk::SoundSystem::DetachMemoryPool( &memoryPoolForEffect );
        g_Allocator.Free( effectBuffer );
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

template<typename Effect>
void VoiceDropCheck(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel, Effect* pEffect, size_t requiredEffectBufferSize)
{
    CheckTestConfig config;
    VoiceDropCheck(itemId, itemLabel, pEffect, requiredEffectBufferSize, config);
}

TEST(AudioProfile, VoiceDropCheckTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    // エフェクトなし（ビルドを通すために、ダミーとして nn::atk::EffectDelay の型を使っています）
    VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP", nullptr, 0);
    VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16", nullptr, 0);
    // エフェクトなし（32kレンダリング）
    {
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_REN32K", nullptr, 0, config);
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_REN32K", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ増加）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMax();
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMAX", nullptr, 0, config);
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMAX", nullptr, 0, config);
    }
    // エフェクトなし（ピッチ減少）
    {
        CheckTestConfig config;
        config.isHandlePitchEnabled = true;
        config.handlePitch = nn::audio::VoiceType::GetPitchMin();
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_PITCHMIN", nullptr, 0, config);
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_PITCHMIN", nullptr, 0, config);
    }
    // エフェクトなし（LPF）
    {
        CheckTestConfig config;
        config.isHandleLpfEnabled = true;
        config.handleLpf = -1.0f;
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_LPF", nullptr, 0, config);
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_LPF", nullptr, 0, config);
    }
    // エフェクトなし（Biquad512）
    {
        CheckTestConfig config;
        config.isHandleBiquadEnabled = true;
        config.handleBiquadFilterType = nn::atk::BiquadFilterType_BandPassFilter512;
        config.handleBiquadFilterValue = 1.0f;
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP, "SE_SIN440_LOOP_BIQUAD512", nullptr, 0, config);
        VoiceDropCheck<nn::atk::EffectDelay>(SE_SIN440_LOOP_PCM16, "SE_SIN440_LOOP_PCM16_BIQUAD512", nullptr, 0, config);
    }

    // Delay 追加
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
        delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_48000);
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP", &delay, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_48000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16", &delay, alignedEffectBufferSize );
    }
    // Delay 追加（32kレンダリング）
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode(nn::atk::EffectDelay::ChannelMode_2Ch);
        delay.SetSampleRate(nn::atk::EffectDelay::SampleRate_32000);
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP, "Delay-SE_SIN440_LOOP_REN32K", &delay, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectDelay delay;
        delay.SetChannelMode( nn::atk::EffectDelay::ChannelMode_2Ch );
        delay.SetSampleRate( nn::atk::EffectDelay::SampleRate_32000 );
        size_t effectbufferSize = delay.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectDelay>( SE_SIN440_LOOP_PCM16, "Delay-SE_SIN440_LOOP_PCM16_REN32K", &delay, alignedEffectBufferSize, config );
    }
    // Reverb 追加
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP", &reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_48000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16", &reverb, alignedEffectBufferSize );
    }
    // Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP, "Reverb-SE_SIN440_LOOP_REN32K", &reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectReverb reverb;
        reverb.SetChannelMode( nn::atk::EffectReverb::ChannelMode_2Ch );
        reverb.SetSampleRate( nn::atk::EffectReverb::SampleRate_32000 );
        size_t effectbufferSize = reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectReverb>( SE_SIN440_LOOP_PCM16, "Reverb-SE_SIN440_LOOP_PCM16_REN32K", &reverb, alignedEffectBufferSize, config );
    }

    // I3dl2 Reverb 追加
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP", &i3dl2Reverb, alignedEffectBufferSize );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_48000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        VoiceDropCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16", &i3dl2Reverb, alignedEffectBufferSize );
    }
    // I3dl2 Reverb 追加（32kレンダリング）
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP, "I3dl2Reverb-SE_SIN440_LOOP_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
    {
        nn::atk::EffectI3dl2Reverb i3dl2Reverb;
        i3dl2Reverb.SetChannelMode( nn::atk::EffectI3dl2Reverb::ChannelMode_2Ch );
        i3dl2Reverb.SetSampleRate( nn::atk::EffectI3dl2Reverb::SampleRate_32000 );
        size_t effectbufferSize = i3dl2Reverb.GetRequiredMemSize();
        size_t alignedEffectBufferSize = nn::util::align_up( effectbufferSize, nn::audio::MemoryPoolType::SizeGranularity );
        CheckTestConfig config;
        config.rendererSampleRate = 32000;
        VoiceDropCheck<nn::atk::EffectI3dl2Reverb>( SE_SIN440_LOOP_PCM16, "I3dl2Reverb-SE_SIN440_LOOP_PCM16_REN32K", &i3dl2Reverb, alignedEffectBufferSize, config );
    }
} // NOLINT(impl/function_size)
#endif

TEST(AudioProfile, VoiceCountWithoutCacheTest)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    // 初期化
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.InitializeImpl( g_Allocator, RenderingSampleRate_48k );
    g_AtkSetup.LoadData(SE_SIN440_LOOP, "SE_SIN440_LOOP");

    // .bfwav の読み込み
    void* bufferForBfwav[nn::atk::SoundSystem::VoiceCountMax];
    nn::audio::MemoryPoolType memoryPoolForBfwav[nn::atk::SoundSystem::VoiceCountMax];
    {
        nn::fs::FileHandle fileHandle;
        int64_t fileSize;
        char bfwavFilePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(bfwavFilePath, sizeof(bfwavFilePath), "%s:/sin440_loop.dspadpcm.bfwav", g_FsSetup.RomMountName);
        NN_ABORT_UNLESS(nn::fs::OpenFile(&fileHandle, bfwavFilePath, nn::fs::OpenMode_Read).IsSuccess());
        NN_ABORT_UNLESS(nn::fs::GetFileSize(&fileSize, fileHandle).IsSuccess());
        size_t alignedFileSize = nn::util::align_up(static_cast<size_t>(fileSize), nn::audio::MemoryPoolType::SizeGranularity);
        for (int i = 0; i < nn::atk::SoundSystem::VoiceCountMax; i++)
        {
            bufferForBfwav[i] = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, static_cast<size_t>(alignedFileSize), nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS(nn::fs::ReadFile(fileHandle, 0, bufferForBfwav[i], static_cast<size_t>(fileSize)).IsSuccess());
            nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForBfwav[i], bufferForBfwav[i], alignedFileSize);
        }
        nn::fs::CloseFile(fileHandle);
    }

    // DSP処理時間が閾値を越えるまで発音数を増やす
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    int playVoiceCount = 0;
    for ( ;; )
    {
        if( playVoiceCount < nn::atk::SoundSystem::VoiceCountMax )
        {
            nn::atk::SoundHandle soundHandle;
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_WaveSoundInfo;
            startInfo.waveSoundInfo.waveAddress = bufferForBfwav[playVoiceCount];
            if ( !soundArchivePlayer.StartSound( &soundHandle, SE_SIN440_LOOP, &startInfo ).IsSuccess() )
            {
                NN_LOG("StartSound failed.\n");
                FAIL();
                break;
            }
            playVoiceCount++;
        }
        nnt::atk::util::UpdateAndWait(soundArchivePlayer);

        // DSP処理時間が閾値を越えてないかチェック
        int profileCountMax = static_cast<int>(sizeof( g_SoundProfile ) / sizeof( g_SoundProfile[0] ));
        int profileCount = g_ProfileReader.Read( g_SoundProfile, profileCountMax );
        if ( profileCount > 0 )
        {
            bool isOverThreshold = false;
            int profileIndex = 0;
            for ( auto i = 0; i < profileCount; ++i )
            {
                // TODO: playVoiceCount が nn::atk::SoundSystem::VoiceCountMax で、ボイスドロップが 1 ボイスでも起きたときに、
                //       無限ループに陥る不具合を修正する (SIGLO-83159 の対応後)
                if ( g_SoundProfile[i].rendererFrameProcess.GetSpan().GetMicroSeconds() > ProcessTimeThreshold ||
                     g_SoundProfile[i].totalVoiceCount >= static_cast<uint32_t>(nn::atk::SoundSystem::GetVoiceCountMax()) )
                {
                    isOverThreshold = true;
                    profileIndex = i;
                    break;
                }
            }

            if ( isOverThreshold )
            {
                NN_LOG( "##teamcity[buildStatisticValue key='Atk_VoiceCountWithoutCacheTest%s_TotalVoiceCount(num)' value='%d']\n", TestNameSuffix, g_SoundProfile[profileIndex].totalVoiceCount );
                NN_LOG( "##teamcity[buildStatisticValue key='Atk_VoiceCountWithoutCacheTest%s_RendererFrameProcess(us)' value='%d']\n", TestNameSuffix, g_SoundProfile[profileIndex].rendererFrameProcess.GetSpan().GetMicroSeconds() );
                break;
            }
        }
    }

    // 終了処理
    soundArchivePlayer.GetSoundPlayer(PLAYER_FOR_COUNTTEST).StopAllSound(0);
    nnt::atk::util::UpdateAndWait(soundArchivePlayer);
    for (int i = 0; i < nn::atk::SoundSystem::VoiceCountMax; i++)
    {
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForBfwav[i]);
        g_Allocator.Free(bufferForBfwav[i]);
    }
    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

void SubMixCountCheck(int subMixCount, int channelCount, int busCount, bool isParallel, bool isTeamCityLogEnabled)
{
    const std::string TestName = "SubMixCountCheckTest";
    const int SubMixCountMax = 10;
    nn::atk::SubMix subMix[SubMixCountMax];
    void* pMemoryForSubMix[SubMixCountMax];
    nn::atk::ProfileReader profileReader;

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // ※※ 現状、ユーザ定義サブミックスはサウンドシステムの直前に終了しないと失敗する問題があるため、
    // ※※ サウンドシステム→サブミックス→サウンドシステム以外、の順番で初期化を行い、終了処理をその逆順で行います。
    // サウンドシステムの初期化
    nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
    nn::atk::SoundSystem::SoundSystemParam& soundSystemParam = initializeParam.GetSoundSystemParam();
    soundSystemParam.enableCustomSubMix = true;
    soundSystemParam.subMixCount = SubMixCountMax;
    soundSystemParam.subMixTotalChannelCount = SubMixCountMax * channelCount * busCount;
    soundSystemParam.enableProfiler = true;
    g_AtkSetup.InitializeSoundSystemImpl(initializeParam, g_Allocator);

    // サブミックスの初期化
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    const int FinalMixBusCount = finalMix.GetBusCount();
    const int FinalMixChannelCount = finalMix.GetChannelCount();
    nn::atk::SubMix* pPreviousSubMix = nullptr;
    for (int subMixIndex = 0; subMixIndex < subMixCount; subMixIndex++)
    {
        nn::atk::SubMix* pCurrentSubMix = &subMix[subMixIndex];
        if (isParallel)
        {
            // 並列の場合、全てファイナルミックスに出力する
            size_t memorySizeForSubMix = pCurrentSubMix->GetRequiredMemorySize(busCount, channelCount, FinalMixBusCount, FinalMixChannelCount);
            pMemoryForSubMix[subMixIndex] = g_Allocator.Allocate(memorySizeForSubMix);
            NN_ABORT_UNLESS(pCurrentSubMix->Initialize(busCount, channelCount, FinalMixBusCount, FinalMixChannelCount, pMemoryForSubMix[subMixIndex], memorySizeForSubMix));
            NN_UNUSED(pPreviousSubMix);
            pCurrentSubMix->SetDestination(&finalMix);
            for(int busIndex = 0; busIndex < pCurrentSubMix->GetBusCount(); busIndex++)
            {
                pCurrentSubMix->SetSend( busIndex, 0, 1.0f );
            }
        }
        else
        {
            // 直列の場合、最初はファイナルミックスに接続し、残りは直前のサブミックスに接続する
            int dstBusCount     = (pCurrentSubMix == nullptr) ? FinalMixBusCount : busCount;
            int dstChannelCount = (pCurrentSubMix == nullptr) ? FinalMixChannelCount : channelCount;
            size_t memorySizeForSubMix = pCurrentSubMix->GetRequiredMemorySize(busCount, channelCount, dstBusCount, dstChannelCount);
            pMemoryForSubMix[subMixIndex] = g_Allocator.Allocate(memorySizeForSubMix);
            NN_ABORT_UNLESS(pCurrentSubMix->Initialize(busCount, channelCount, dstBusCount, dstChannelCount, pMemoryForSubMix[subMixIndex], memorySizeForSubMix));
            if (subMixIndex == 0)
            {
                pCurrentSubMix->SetDestination(&finalMix);
                for(int busIndex = 0; busIndex < pCurrentSubMix->GetBusCount(); busIndex++)
                {
                    pCurrentSubMix->SetSend( busIndex, 0, 1.0f );
                }
            }
            else
            {
                pCurrentSubMix->SetDestination(pPreviousSubMix);
            }
            pPreviousSubMix = pCurrentSubMix;
        }
    }

    // サウンドシステム以外の初期化
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    g_AtkSetup.InitializeWithoutSoundSystem(initializeParam, g_Allocator);
    nn::atk::SoundSystem::RegisterProfileReader(profileReader);
    g_AtkSetup.LoadData(SE_SIN440_LOOP, "SE_SIN440_LOOP");
    nnt::atk::util::UpdateAndWait(soundArchivePlayer);

    // サウンドの再生
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo startInfo;
    startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_OutputReceiver;
    startInfo.pOutputReceiver = &subMix[subMixCount - 1]; // 最後に追加したサブミックスに出力する
    soundArchivePlayer.StartSound(&soundHandle, SE_SIN440_LOOP, &startInfo);
    nnt::atk::util::UpdateAndWait(soundArchivePlayer);

    // パフォーマンスの確認
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    int profileCountMax = static_cast<int>(sizeof( g_SoundProfile ) / sizeof( g_SoundProfile[0] ));
    int profileCount = profileReader.Read( g_SoundProfile, profileCountMax );
    if ( profileCount > 0 )
    {
        int profileIndex = 0;
        for ( auto i = 0; i < profileCount; ++i )
        {
            if ( g_SoundProfile[i].rendererFrameProcess.GetSpan().GetMicroSeconds() > ProcessTimeThreshold ||
                 g_SoundProfile[i].totalVoiceCount >= static_cast<uint32_t>(nn::atk::SoundSystem::GetVoiceCountMax()) )
            {
                // 最初のプロファイルを取得する
                profileIndex = i;
                break;
            }
        }

        // 結果をログ出力する
        NN_LOG( "Atk_SubMixCheckTestFor%s%s_TotalSubMixCount subMix:%2d channel:%2d bus:%2d\n", isParallel ? "Parallel" : "Serial" ,TestNameSuffix, subMixCount, channelCount, busCount );
        if (isTeamCityLogEnabled)
        {
            NN_LOG( "##teamcity[buildStatisticValue key='Atk_SubMixCheckTestFor%s%s_RendererFrameProcess(us)' value='%d']\n", isParallel ? "Parallel" : "Serial" ,TestNameSuffix, g_SoundProfile[profileIndex].rendererFrameProcess.GetSpan().GetMicroSeconds() );
        }
        else
        {
            NN_LOG( "Atk_SubMixCheckTestFor%s%s_RendererFrameProcess %d(us)\n", isParallel ? "Parallel" : "Serial" ,TestNameSuffix, g_SoundProfile[profileIndex].rendererFrameProcess.GetSpan().GetMicroSeconds() );
        }
    }

    // サウンドシステム以外の終了処理
    nn::atk::SoundSystem::UnregisterProfileReader(profileReader);
    g_AtkSetup.FinalizeWithoutSoundSystem(g_Allocator);

    // サブミックスの終了処理、後から付けたサブミックスから終了処理を行う
    const auto TimeOut   = nn::TimeSpan::FromMilliSeconds(5 * 1000);
    nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0);
    for (int subMixIndex = subMixCount - 1; subMixIndex >= 0;)
    {
        if (subMix[subMixIndex].IsFinalizable())
        {
            subMix[subMixIndex].Finalize();
            g_Allocator.Free(pMemoryForSubMix[subMixIndex]);
            subMixIndex--;
        }
        else
        {
            nn::os::SleepThread(WaitTime);
            elapsed += WaitTime;
            EXPECT_GE(TimeOut, elapsed);
        }
    }

    // サウンドシステムの終了処理
    g_AtkSetup.FinalizeSoundSystemImpl(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

void SubMixCountCheckForParallel(int subMixCount, int channelCount, int busCount, bool isTeamCityLogEnabled)
{
    // サブミックスを並列に出力する場合
    SubMixCountCheck(subMixCount, channelCount, busCount, true, isTeamCityLogEnabled);
}

void SubMixCountCheckForSerial(int subMixCount, int channelCount, int busCount, bool isTeamCityLogEnabled)
{
    // サブミックスを直列に出力する場合
    SubMixCountCheck(subMixCount, channelCount, busCount, false, isTeamCityLogEnabled);
}

TEST(AudioProfile, SubMixCheckTest)
{
    const bool IsDebugMode     = false; // true にすると、サブミックスを１つずつ増やして処理時間のログ出力を行います。
    const int  SubMixCountMax  = 10;
    const int  ChannelCount    = 6;
    const int  BusCount        = 2;

    for (int subMixNum = 1; subMixNum <= SubMixCountMax; subMixNum++ )
    {
        bool isLastResult = (subMixNum == SubMixCountMax);
        if (IsDebugMode || isLastResult)
        {
            // 最終結果だけ TeamCity に出力する
            SubMixCountCheckForParallel(subMixNum, ChannelCount, BusCount, isLastResult);
            SubMixCountCheckForSerial(subMixNum, ChannelCount, BusCount, isLastResult);
        }
    }
}
