﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"
#include "testAtk_TestEffectUtil.h"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

typedef    nnt::atk::effectUtil::EffectAuxTest    EffectAuxTest;

namespace {
const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup g_FsSetup;
nnt::atk::util::AtkCommonSetup g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;
}

// TODO: チャンネル設定以外のテスト項目を追加する
#ifndef NN_SDK_BUILD_RELEASE
TEST(Effect, AuxChannelSettingDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (int channelCount = 1; channelCount <= nn::atk::EffectAux::ChannelCountMax; ++channelCount)
    {
        EffectAuxTest aux;
        nn::atk::ChannelIndex* pInvalidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount));
        NN_ABORT_UNLESS_NOT_NULL(pInvalidChannelSetting);
        // チャンネルの重複
        if (channelCount > 1)
        {
            for (int i = 0; i < channelCount; ++i)
            {
                if (i == 0 || i == 1)
                {
                    pInvalidChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
                else
                {
                    pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
                }
            }
            EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelIndex(pInvalidChannelSetting, channelCount), ".*");
        }

        // チャンネル設定の値に -1 が含まれる
        for (int i = 0; i < channelCount; ++i)
        {
            pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i - 1);
        }
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelIndex(pInvalidChannelSetting, channelCount), ".*");

        // チャンネル設定の値に nn::atk::ChannelIndex_Count が含まれる
        for (int i = 0; i < channelCount; ++i)
        {
            if (i == channelCount - 1)
            {
                pInvalidChannelSetting[i] = nn::atk::ChannelIndex_Count;
            }
            else
            {
                pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
            }
        }
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelIndex(pInvalidChannelSetting, channelCount), ".*");

        // チャンネル数が設定できる最大値よりも多い
        nn::atk::ChannelIndex* pValidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount));
        NN_ABORT_UNLESS_NOT_NULL(pValidChannelSetting);
        for (int i = 0; i < channelCount; ++i)
        {
            pValidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
        }
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelCount(nn::atk::EffectAux::ChannelCountMax + 1), ".*");
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelIndex(pInvalidChannelSetting, nn::atk::EffectAux::ChannelCountMax + 1), ".*");

        // チャンネル数が設定できる最小値よりも少ない
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelCount(0), ".*");
        EXPECT_DEATH_IF_SUPPORTED(aux.SetChannelIndex(pValidChannelSetting, 0), ".*");

        g_Allocator.Free(reinterpret_cast<void*>(pInvalidChannelSetting));
        g_Allocator.Free(reinterpret_cast<void*>(pValidChannelSetting));
    }
    g_Allocator.Finalize();
}
#endif

TEST(Effect, AuxSetChannelCountTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (int channelCount = 1; channelCount <= nn::atk::EffectAux::ChannelCountMax; ++channelCount)
    {
        EffectAuxTest aux;

        EXPECT_TRUE(aux.SetChannelCount(channelCount));
        ASSERT_EQ(channelCount, aux.GetChannelCount());
        nn::atk::ChannelIndex* pAuxChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount));
        NN_ABORT_UNLESS_NOT_NULL(pAuxChannelSetting);
        aux.GetChannelIndex(pAuxChannelSetting, channelCount);
        for (int i = 0; i < aux.GetChannelCount(); ++i)
        {
            EXPECT_EQ(static_cast<nn::atk::ChannelIndex>(i), pAuxChannelSetting[i]);
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, AuxSetChannelIndexTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (int channelCount = 1; channelCount <= nn::atk::EffectAux::ChannelCountMax; ++channelCount)
    {
        EffectAuxTest aux;

        nn::atk::ChannelIndex* pChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount));
        NN_ABORT_UNLESS_NOT_NULL(pChannelSetting);
        // 初期化
        for (int i = 0; i < channelCount; ++i)
        {
            pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
        }

        // 適切なチャンネル設定となるものについて総当たり検査
        while (pChannelSetting[channelCount - 1] != nn::atk::ChannelIndex_Count)
        {
            // テストケースとして適切か(チャンネル設定の被りがないか)確認
            bool isValidChannelSetting = true;
            for (int i = 0; i < channelCount; ++i)
            {
                for (int j = i + 1; j < channelCount; ++j)
                {
                    isValidChannelSetting &= (pChannelSetting[i] != pChannelSetting[j]);
                }
            }

            // テスト本体
            if (isValidChannelSetting)
            {
                EXPECT_TRUE(aux.SetChannelIndex(pChannelSetting, channelCount));

                ASSERT_EQ(channelCount, aux.GetChannelCount());
                nn::atk::ChannelIndex* pAuxChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount));
                NN_ABORT_UNLESS_NOT_NULL(pAuxChannelSetting);
                aux.GetChannelIndex(pAuxChannelSetting, channelCount);
                for (int i = 0; i < channelCount; ++i)
                {
                    EXPECT_EQ(pChannelSetting[i], pAuxChannelSetting[i]);
                }
                g_Allocator.Free(reinterpret_cast<void*>(pAuxChannelSetting));
            }

            // テストケースの更新
            for (int i = 0; i < channelCount; ++i)
            {
                pChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(pChannelSetting[i] + 1);
                // 繰り上がりしない場合
                if (pChannelSetting[i] != nn::atk::ChannelIndex_Count || i == channelCount - 1)
                {
                    break;
                }

                pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
            }
        }

        g_Allocator.Free(reinterpret_cast<void*>(pChannelSetting));
    }

    g_Allocator.Finalize();
}

TEST(Effect, AuxAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for (int channelCount = 1; channelCount <= nn::atk::EffectAux::ChannelCountMax; ++channelCount)
    {
        EffectAuxTest aux;
        EffectAuxTest auxForFinalMix;

        // 使用するチャンネル設定の初期化
        void* bufferForValidChannelSetting = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * channelCount);
        nn::atk::ChannelIndex* pValidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(bufferForValidChannelSetting);
        NN_ABORT_UNLESS_NOT_NULL(pValidChannelSetting);
        for (int i = 0; i < channelCount; ++i)
        {
            pValidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
        }

        // Append 前に変更できることをテストする
        EXPECT_TRUE(aux.SetChannelCount(channelCount));
        EXPECT_TRUE(aux.SetChannelIndex(pValidChannelSetting, channelCount));
        EXPECT_TRUE(auxForFinalMix.SetChannelCount(channelCount));
        EXPECT_TRUE(auxForFinalMix.SetChannelIndex(pValidChannelSetting, channelCount));

        // エフェクトの Append
        size_t auxBufferSize = nn::util::align_up(nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&aux), nn::audio::MemoryPoolType::SizeGranularity);
        void* auxBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, auxBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(auxBuffer);
        nn::audio::MemoryPoolType memoryPoolForAux;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForAux, auxBuffer, auxBufferSize);

        size_t auxBufferSizeForFinalMix = nn::util::align_up(nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&auxForFinalMix), nn::audio::MemoryPoolType::SizeGranularity);
        void* auxBufferForFinalMix = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, auxBufferSizeForFinalMix, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(auxBufferForFinalMix);
        nn::audio::MemoryPoolType memoryPoolForAuxFinalMix;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForAuxFinalMix, auxBufferForFinalMix, auxBufferSizeForFinalMix);

        aux.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &aux, auxBuffer, auxBufferSize));

        auxForFinalMix.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffectToFinalMix(&auxForFinalMix, auxBufferForFinalMix, auxBufferSizeForFinalMix));

        // Append で送ったコマンドが処理されるのを待つ
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(aux);
        nnt::atk::effectUtil::WaitForEffectAppend(auxForFinalMix);

        // Append 完了後に変更ができないことをテストする
        EXPECT_FALSE(aux.SetChannelCount(channelCount));
        EXPECT_FALSE(aux.SetChannelIndex(pValidChannelSetting, channelCount));

        EXPECT_FALSE(auxForFinalMix.SetChannelCount(channelCount));
        EXPECT_FALSE(auxForFinalMix.SetChannelIndex(pValidChannelSetting, channelCount));

        aux.SetEnabled(false);
        auxForFinalMix.SetEnabled(false);
        nnt::atk::effectUtil::UpdateAndProcess(soundArchivePlayer);
        nnt::atk::effectUtil::WaitForEffectClear(aux);
        nnt::atk::effectUtil::WaitForEffectClear(auxForFinalMix);
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &aux);
        nn::atk::SoundSystem::RemoveEffectFromFinalMix(&auxForFinalMix);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForAux);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForAuxFinalMix);

        // Remove 後に変更できることをテストする
        EXPECT_TRUE(aux.SetChannelCount(channelCount));
        EXPECT_TRUE(aux.SetChannelIndex(pValidChannelSetting, channelCount));
        EXPECT_TRUE(auxForFinalMix.SetChannelCount(channelCount));
        EXPECT_TRUE(auxForFinalMix.SetChannelIndex(pValidChannelSetting, channelCount));

        g_Allocator.Free(auxBuffer);
        g_Allocator.Free(auxBufferForFinalMix);
        g_Allocator.Free(bufferForValidChannelSetting);
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
