﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/mem.h>
#include <nnt/atkUtil/testAtk_Util.h>

//---------------------------------------------------------------------------
//! @brief        Atk テスト用ヒープ
//---------------------------------------------------------------------------
class TestHeap
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    TestHeap()
    {
        m_pAllocator     = NULL;
        m_Counter        = 0;
        m_AllocedSize    = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    explicit TestHeap( nn::mem::StandardAllocator* allocator )
    {
        m_pAllocator  = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~TestHeap()
    {
        Finalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        nn::mem::StandardAllocator インスタンスを設定します。
    //---------------------------------------------------------------------------
    void SetAllocator( nn::mem::StandardAllocator* allocator )
    {
        m_pAllocator  = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        nn::mem::StandardAllocator インスタンスを取得します。
    //---------------------------------------------------------------------------
    nn::mem::StandardAllocator* GetAllocator() const
    {
        return m_pAllocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを確保します。
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size, size_t alignment = 128 )
    {
        void* ptr = nnt::atk::util::AllocateUninitializedMemory( *m_pAllocator, size, alignment );
        NN_ASSERT( ptr );
        m_Counter++;
        m_AllocedSize += size;

        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを解放します。
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr )
    {
        NN_ASSERT( ptr );

        if ( m_Counter > 0 )
        {
            m_Counter--;
        }
        else
        {
            NN_ABORT( "[ATKUT] Heap Free Error! " );
        }

        return m_pAllocator->Free( ptr );
//      return _aligned_free( ptr );
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリ確保カウンターを取得します。
    //---------------------------------------------------------------------------
    int GetCounter() const
    {
        return m_Counter;
    }

    //---------------------------------------------------------------------------
    //! @brief        確保したメモリサイズを取得します。
    //---------------------------------------------------------------------------
    size_t GetAllocedSize() const
    {
        return m_AllocedSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        終了処理
    //---------------------------------------------------------------------------
    void Finalize()
    {
        m_Counter = 0;
        m_AllocedSize = 0;
    }

private:
    nn::mem::StandardAllocator* m_pAllocator;
    int                         m_Counter;
    size_t                      m_AllocedSize;
};
