﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h> // Debug

#include <nn/nn_Assert.h>

#include <opus.h>
#include <opus_private.h>

#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_UtilOpusPacketMaker.h>

namespace nnt {
namespace codec {
namespace util {
namespace {

enum OpusFrameCode
{
    OpusFrameCode_OneFrame,
    OpusFrameCode_TwoFrameEqualSizes,
    OpusFrameCode_TwoFrameDifferentSizes,
    OpusFrameCode_ArbitraryCount,
    OpusFrameCode_Invalid
};

uint8_t GetConfigurationCodeOfCodingModeCelt(
    nn::codec::OpusBandwidth bandwidth,
    int frameInMicroSeconds) NN_NOEXCEPT
{
    NN_ASSERT(bandwidth == nn::codec::OpusBandwidth_NarrowBand
        || bandwidth == nn::codec::OpusBandwidth_WideBand
        || bandwidth == nn::codec::OpusBandwidth_SuperWideBand
        || bandwidth == nn::codec::OpusBandwidth_FullBand);
    NN_ASSERT(frameInMicroSeconds == 2500
        || frameInMicroSeconds == 5000
        || frameInMicroSeconds == 10000
        || frameInMicroSeconds == 20000);
    const int frameBase = 2500;
    const uint8_t codingModeCode = 1u;
    uint8_t bandwidthCode;

    switch(bandwidth)
    {
    case nn::codec::OpusBandwidth_NarrowBand:
        {
            bandwidthCode = 0u;
        }
        break;
    case nn::codec::OpusBandwidth_WideBand:
        {
            bandwidthCode = 1u;
        }
        break;
    case nn::codec::OpusBandwidth_SuperWideBand:
        {
            bandwidthCode = 2u;
        }
        break;
    case nn::codec::OpusBandwidth_FullBand:
        {
            bandwidthCode = 3u;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    const auto frameCode = static_cast<uint8_t>(Log2<uint32_t>(frameInMicroSeconds / frameBase));

    return (codingModeCode << 4) | (bandwidthCode << 2) | frameCode;
}

uint8_t GetConfigurationCodeOfCodingModeSilk(
    nn::codec::OpusBandwidth bandwidth,
    int frameInMicroSeconds) NN_NOEXCEPT
{
    NN_ASSERT(bandwidth == nn::codec::OpusBandwidth_NarrowBand
        || bandwidth == nn::codec::OpusBandwidth_MediumBand
        || bandwidth == nn::codec::OpusBandwidth_WideBand);
    NN_ASSERT(frameInMicroSeconds == 10000
        || frameInMicroSeconds == 20000
        || frameInMicroSeconds == 40000
        || frameInMicroSeconds == 60000);
    const int frameBase = 20000;
    const uint8_t codingModeCode = 0u;
    uint8_t bandwidthCode;

    switch(bandwidth)
    {
    case nn::codec::OpusBandwidth_NarrowBand:
        {
            bandwidthCode = 0u;
        }
        break;
    case nn::codec::OpusBandwidth_MediumBand:
        {
            bandwidthCode = 1u;
        }
        break;
    case nn::codec::OpusBandwidth_WideBand:
        {
            bandwidthCode = 2u;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    const auto frameCode = static_cast<uint8_t>(frameInMicroSeconds / frameBase);
    return (codingModeCode << 4) | (bandwidthCode << 2) | frameCode;
}

uint8_t GetConfigurationCodeOfCodingModeHybrid(
    nn::codec::OpusBandwidth bandwidth,
    int frameInMicroSeconds) NN_NOEXCEPT
{
    NN_ASSERT(bandwidth == nn::codec::OpusBandwidth_SuperWideBand
        || bandwidth == nn::codec::OpusBandwidth_FullBand);
    NN_ASSERT(frameInMicroSeconds == 10000
        || frameInMicroSeconds == 20000);
    const int frameBase = 20000;
    const uint8_t codingModeCode = 0u;
    uint8_t bandwidthCode;

    switch(bandwidth)
    {
    case nn::codec::OpusBandwidth_SuperWideBand:
        {
            bandwidthCode = 6u;
        }
        break;
    case nn::codec::OpusBandwidth_FullBand:
        {
            bandwidthCode = 7u;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    const auto frameCode = static_cast<uint8_t>(frameInMicroSeconds / frameBase);
    return (codingModeCode << 4) | (bandwidthCode << 1) | frameCode;
}

uint8_t GetConfigurationCode(
    nn::codec::OpusCodingMode codingMode,
    int frameInMicroSeconds,
    nn::codec::OpusBandwidth bandwidth) NN_NOEXCEPT
{
    NN_ASSERT(codingMode == nn::codec::OpusCodingMode_Celt
        || codingMode == nn::codec::OpusCodingMode_Hybrid
        || codingMode == nn::codec::OpusCodingMode_Silk);
    NN_ASSERT(frameInMicroSeconds == 2500
        || frameInMicroSeconds == 5000
        || frameInMicroSeconds == 10000
        || frameInMicroSeconds == 20000
        || frameInMicroSeconds == 40000
        || frameInMicroSeconds == 60000);
    NN_ASSERT(bandwidth == nn::codec::OpusBandwidth_NarrowBand
        || bandwidth == nn::codec::OpusBandwidth_MediumBand
        || bandwidth == nn::codec::OpusBandwidth_WideBand
        || bandwidth == nn::codec::OpusBandwidth_SuperWideBand
        || bandwidth == nn::codec::OpusBandwidth_FullBand);
    switch(codingMode)
    {
    case nn::codec::OpusCodingMode_Celt:
        {
            return GetConfigurationCodeOfCodingModeCelt(bandwidth, frameInMicroSeconds);
        }
        break;
    case nn::codec::OpusCodingMode_Silk:
        {
            return GetConfigurationCodeOfCodingModeSilk(bandwidth, frameInMicroSeconds);
        }
        break;
    case nn::codec::OpusCodingMode_Hybrid:
        {
            return GetConfigurationCodeOfCodingModeHybrid(bandwidth, frameInMicroSeconds);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

uint8_t GetFrameCode(
    int frameCount,
    const std::size_t sizes[]) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(frameCount, 1, 48);
    switch(frameCount)
    {
    case 1:
        return 0u;
    case 2:
        return (sizes[0] == sizes[1]) ? 1u : 2u;
    default:
        return 3u;
    }
}

}

bool GetOpusPacketSize(
    std::size_t* pOutputDataSize,
    const struct OpusPacketConfiguration& configuration) NN_NOEXCEPT
{
    NN_ASSERT_GREATER(configuration.streamCount, 0);
    *pOutputDataSize = configuration.streamCount * 4 - 1;
    return true;
}

bool MakeOpusPacket(
    std::size_t* pOutputDataSize,
    uint8_t* outputBuffer,
    std::size_t outputSize,
    const struct OpusPacketConfiguration& configuration) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(configuration.channelCount, 1, 2);
    NN_ASSERT_GREATER(configuration.streamCount, 0);
    NN_ASSERT(configuration.frameInMicroSeconds == 2500
        || configuration.frameInMicroSeconds == 5000
        || configuration.frameInMicroSeconds == 10000
        || configuration.frameInMicroSeconds == 20000
        || configuration.frameInMicroSeconds == 40000
        || configuration.frameInMicroSeconds == 60000);
    NN_ASSERT(configuration.codingMode == nn::codec::OpusCodingMode_Celt
        || configuration.codingMode == nn::codec::OpusCodingMode_Hybrid
        || configuration.codingMode == nn::codec::OpusCodingMode_Silk);
    NN_ASSERT(configuration.bandwidth == nn::codec::OpusBandwidth_NarrowBand
        || configuration.bandwidth == nn::codec::OpusBandwidth_MediumBand
        || configuration.bandwidth == nn::codec::OpusBandwidth_WideBand
        || configuration.bandwidth == nn::codec::OpusBandwidth_SuperWideBand
        || configuration.bandwidth == nn::codec::OpusBandwidth_FullBand);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    std::size_t requiredBufferSize = 0;
    NN_ASSERT_EQUAL(GetOpusPacketSize(&requiredBufferSize, configuration), true);
    if (outputSize < requiredBufferSize + headerSize)
    {
        return false;
    }
    struct nn::codec::detail::OpusPacketInternal::Header header =
    {
        static_cast<uint32_t>(requiredBufferSize), // パケットサイズ
        static_cast<uint32_t>(0u) // finalRange
    };
    nn::codec::detail::OpusPacketInternal::SetHeaderInPacket(outputBuffer, header);
    auto p = outputBuffer + headerSize;

    const auto config = GetConfigurationCode(
        configuration.codingMode,
        configuration.frameInMicroSeconds,
        configuration.bandwidth);

    const auto frameCode = GetFrameCode(
        configuration.frameCount,
        configuration.frameSizes
    );

    const int streamCount = configuration.streamCount;
    for (auto stream = 0; stream < streamCount; ++stream)
    {
        p[0] = static_cast<uint8_t>(config << 3) | frameCode | (configuration.channelCount == 1 ? 0u : 4u);
        if (stream != streamCount - 1)
        {
            p[1] = 2;
            ++p;
        }
        p[1] = 255;
        p[2] = 255;
        p += 3;
    }
    *pOutputDataSize = requiredBufferSize + headerSize;
    return true;

// マルチフレームを未サポートなので、コメントアウトしておく。
// ToDo: マルチフレームをサポートする
//    switch(frameCode)
//    {
//    case OpusFrameCode_OneFrame:
//        {
//        }
//        break;
//    case OpusFrameCode_TwoFrameEqualSizes:
//        {
//            NN_ASSERT_NOT_EQUAL(configuration.frameSizes[0] & 1, 0);
//        }
//        break;
//    case OpusFrameCode_TwoFrameDifferentSizes:
//        {
//            const auto packetSize = configuration.frameSizes[0];
//            auto frameSequenceSize = 1;
//            if (packetSize <= 251)
//            {
//                p[1] = static_cast<uint8_t>(packetSize);
//            }
//            else
//            {
//                p[2] = static_cast<uint8_t>(packetSize - 252) / 4;
//                p[1] = static_cast<uint8_t>(packetSize - p[2] * 4);
//                ++frameSequenceSize;
//            }
//            if (configuration.frameSizes[0] - frameSequenceSize - packetSize < 0)
//            {
//                return false;
//            }
//            *pOutputDataSize += 2;
//        }
//        break;
//    case OpusFrameCode_ArbitraryCount:
//        {
//            union
//            {
//                uint8_t frameCountByte;
//                struct
//                {
//                    uint8_t frameCount: 6;
//                    uint8_t padding: 1;
//                    uint8_t vbr: 1;
//                } bits;
//            };
//            if (configuration.frameCount > 48)
//            {
//                return false;
//            }
//            if (configuration.paddingSize > 254u)
//            {
//                return false;
//            }
//            frameCodeCount = static_cast<uint8_t>(configuration.frameCount);
//            bits.padding = configuration.paddingSize == 0 ? 0  : 1;
//            bits.vbr = configuration.vbr == true ? 1 : 0;
//
//            *pOutputDataSize += 255; // ??
//        }
//        break;
//    default: NN_UNEXPECTED_DEFAULT;
//    }
}

bool DefaultOpusPacket(struct OpusPacketConfiguration* pConfiguration) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pConfiguration);

    std::memset(pConfiguration, 0, sizeof(*pConfiguration));

    pConfiguration->channelCount = 1; // Mono
    pConfiguration->frameInMicroSeconds = 20000; // 20ms
    pConfiguration->codingMode = nn::codec::OpusCodingMode_Celt; // celt
    pConfiguration->bandwidth = nn::codec::OpusBandwidth_FullBand; // -20kHz
    pConfiguration->frameCount = 1;
    pConfiguration->streamCount = 1;
    pConfiguration->frameSizes[0] = 1;
    pConfiguration->vbr = 0; // "cbr"
    pConfiguration->paddingSize = 0;
    return true;
}

nn::codec::OpusResult GetOpusPacketBandwidth(nn::codec::OpusBandwidth* pBandwidth, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(opusPacketSize);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    const int bandwidth = opus_packet_get_bandwidth(&pOpusPacket[headerSize]);
    *pBandwidth = ConvertBandwidth(bandwidth);
    return nn::codec::OpusResult_Success;
}

nn::codec::OpusResult GetOpusPacketCodingMode(nn::codec::OpusCodingMode* pCodingMode, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(opusPacketSize);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    const uint8_t config = pOpusPacket[headerSize] >> 3;
    if (config < 12)
    {
        *pCodingMode = nn::codec::OpusCodingMode_Silk;
    }
    else if (config < 16)
    {
        *pCodingMode = nn::codec::OpusCodingMode_Hybrid;
    }
    else
    {
        *pCodingMode = nn::codec::OpusCodingMode_Celt;
    }
    return nn::codec::OpusResult_Success;
}

nn::codec::OpusResult GetOpusPacketChannelCount(int* pChannelCount, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(opusPacketSize);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    *pChannelCount = opus_packet_get_nb_channels(&pOpusPacket[headerSize]);
    return nn::codec::OpusResult_Success;
}

nn::codec::OpusResult GetOpusPacketFrameInMicroSeconds(int* pframeInMicroSeconds, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(opusPacketSize);
    const int sampleRate = 48000;
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    *pframeInMicroSeconds = static_cast<int>(opus_packet_get_samples_per_frame(&pOpusPacket[headerSize], sampleRate) *  static_cast<int64_t>(1000 * 1000) / sampleRate);
    return nn::codec::OpusResult_Success;
}

nn::codec::OpusResult GetOpusPacketFrameCount(int* pFrameCount, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    auto result = opus_packet_get_nb_frames(pOpusPacket + headerSize, static_cast<opus_int32>(opusPacketSize));
    if (result > 0)
    {
        *pFrameCount = result;
        return nn::codec::OpusResult_Success;
    }
    return ConvertResult(result);
}

nn::codec::OpusResult GetOpusPacketPaddingSize(int* pPaddingSize, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(pOpusPacket);
    NN_UNUSED(opusPacketSize);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    *pPaddingSize = -1;
    return nn::codec::OpusResult_UnsupportedFormat; // TBD
}

nn::codec::OpusResult GetOpusPacketVbr(bool* pVbr, const uint8_t* pOpusPacket, std::size_t opusPacketSize) NN_NOEXCEPT
{
    NN_UNUSED(pOpusPacket);
    NN_UNUSED(opusPacketSize);
    const auto headerSize = sizeof(nn::codec::detail::OpusPacketInternal::Header);
    NN_ASSERT_GREATER(opusPacketSize, headerSize);
    *pVbr = false;
    return nn::codec::OpusResult_UnsupportedFormat; // TBD
}

}}} // nnt::codec::util
