﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/atkUtil/testAtk_AtkCommonSetup.h>
#include <nnt/atkUtil/testAtk_FsCommonSetup.h>
#include <nnt/atkUtil/testAtk_Util.h>

namespace nnt { namespace atk { namespace util {

void AtkCommonSetup::InitializeParam::SetSoundSystemParam(nn::atk::SoundSystem::SoundSystemParam& param) NN_NOEXCEPT
{
    m_SoundSystemParam = param;
}

const nn::atk::SoundSystem::SoundSystemParam& AtkCommonSetup::InitializeParam::GetSoundSystemParam() const NN_NOEXCEPT
{
    return m_SoundSystemParam;
}

nn::atk::SoundSystem::SoundSystemParam& AtkCommonSetup::InitializeParam::GetSoundSystemParam() NN_NOEXCEPT
{
    return m_SoundSystemParam;
}

void AtkCommonSetup::InitializeParam::SetSoundHeapSize(size_t size) NN_NOEXCEPT
{
    m_SoundHeapSize = size;
}

const size_t AtkCommonSetup::InitializeParam::GetSoundHeapSize() const NN_NOEXCEPT
{
    return m_SoundHeapSize;
}

void AtkCommonSetup::InitializeParam::SetSoundArchivePath(char* path) NN_NOEXCEPT
{
    m_SoundArchivePath = path;
}

const char* AtkCommonSetup::InitializeParam::GetSoundArchivePath() const NN_NOEXCEPT
{
    return m_SoundArchivePath;
}

AtkCommonSetup::AtkCommonSetup()
    : m_IsInitializedSoundSystem(false)
{
}

AtkCommonSetup::~AtkCommonSetup()
{
}

////////////////////////////////////////////////////////////////////////////////////
// Atk Initialize
////////////////////////////////////////////////////////////////////////////////////
void AtkCommonSetup::Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    Initialize(param, allocator);
}

void AtkCommonSetup::Initialize(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    InitializeSoundSystem(param.GetSoundSystemParam(), allocator);
    InitializeSoundHeap(param.GetSoundHeapSize(), allocator);
    InitializeSoundArchive(param.GetSoundArchivePath(), allocator);
    InitializeSoundDataManager(allocator);
    InitializeSoundArchivePlayer(allocator);
}

void AtkCommonSetup::InitializeSoundSystem(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    nn::atk::SoundSystem::SoundSystemParam param;
    InitializeSoundSystem(param, allocator);
}

void AtkCommonSetup::InitializeSoundSystem(const nn::atk::SoundSystem::SoundSystemParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    if(nn::atk::SoundSystem::IsInitialized())
    {
        return;
    }

    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
    m_pMemoryForSoundSystem = AllocateUninitializedMemory(allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);

    bool result = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystem),
        memSizeForSoundSystem);
    NN_ABORT_UNLESS(result, "cannot initialize SoundSystem");
    NN_ABORT_UNLESS(nn::atk::SoundSystem::IsInitialized(), "Initialize state is invalid");

    m_IsInitializedSoundSystem = true;
}

void AtkCommonSetup::InitializeSoundArchive(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    InitializeSoundArchive(nullptr, allocator);
}

void AtkCommonSetup::InitializeSoundArchive(const char* soundArchivePath, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    if(soundArchivePath == nullptr)
    {
        char commonSoundArchivePath[128];
        NN_ABORT_UNLESS(nnt::atk::util::GetCommonSoundArchivePath(commonSoundArchivePath, sizeof(commonSoundArchivePath), nnt::atk::util::FsCommonSetup::RomMountName), "Cannot get sound archive path.");
        InitializeFsSoundArchive(commonSoundArchivePath, allocator);
    }
    else
    {
        InitializeFsSoundArchive(soundArchivePath, allocator);
    }
}

void AtkCommonSetup::InitializeFsSoundArchive(const char* soundArchivePath, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    // SoundArchive ファイルのオープン
    NN_ABORT_UNLESS(m_FsSoundArchive.Open(soundArchivePath),
                    "cannot open SoundArchive(%s)\n", soundArchivePath);
    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = m_FsSoundArchive.GetHeaderSize();

    m_pMemoryForInfoBlock = AllocateUninitializedMemory(allocator, infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize);
    NN_ABORT_UNLESS(m_FsSoundArchive.LoadHeader(m_pMemoryForInfoBlock, infoBlockSize), "cannot load InfoBlock");
    m_pMemoryForLabelStringData = nullptr;
}

void AtkCommonSetup::InitializeSoundDataManager(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    std::size_t memSizeForSoundDataManager = m_SoundDataManager.GetRequiredMemSize(&m_FsSoundArchive);

    m_pMemoryForSoundDataManager = AllocateUninitializedMemory(allocator, memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize);
    NN_ABORT_UNLESS(m_SoundDataManager.Initialize(
                    &m_FsSoundArchive,
                    m_pMemoryForSoundDataManager,
                    memSizeForSoundDataManager),
                    "cannot initialize SoundDataManager");
}

void AtkCommonSetup::InitializeSoundArchivePlayer(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    // SoundArchivePlayer の初期化
    size_t memSizeForSoundArchivePlayer = m_SoundArchivePlayer.GetRequiredMemSize(&m_FsSoundArchive);
    m_pMemoryForSoundArchivePlayer = AllocateUninitializedMemory(allocator, memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);

    size_t memSizeForStreamBuffer = m_SoundArchivePlayer.GetRequiredStreamBufferSize(&m_FsSoundArchive);
    size_t alignedMemSizeForStreamBuffer = nn::util::align_up( memSizeForStreamBuffer, nn::audio::MemoryPoolType::SizeGranularity );
    m_pMemoryForStreamBuffer = AllocateUninitializedMemory(allocator, alignedMemSizeForStreamBuffer, nn::audio::MemoryPoolType::AddressAlignment);
    nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForStreamBuffer, m_pMemoryForStreamBuffer, alignedMemSizeForStreamBuffer );

    NN_ABORT_UNLESS(m_SoundArchivePlayer.Initialize(
                    &m_FsSoundArchive,
                    &m_SoundDataManager,
                    m_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
                    m_pMemoryForStreamBuffer, memSizeForStreamBuffer),
                    "cannot initialize SoundArchivePlayer");
}

void AtkCommonSetup::InitializeSoundHeap(size_t soundHeapSize, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    m_pMemoryForSoundHeap = AllocateUninitializedMemory(allocator, soundHeapSize);
    NN_ABORT_UNLESS(m_SoundHeap.Create(m_pMemoryForSoundHeap, soundHeapSize), "cannot create SoundHeap");
}

////////////////////////////////////////////////////////////////////////////////////
// Atk Finalize
////////////////////////////////////////////////////////////////////////////////////

void AtkCommonSetup::Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    FinalizeSoundArchivePlayer(allocator);
    FinalizeSoundDataManager(allocator);
    FinalizeSoundArchive(allocator);
    FinalizeSoundHeap(allocator);
    FinalizeSoundSystem(allocator);
}

void AtkCommonSetup::FinalizeSoundSystem(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    if(!m_IsInitializedSoundSystem)
    {
        return;
    }

    nn::atk::SoundSystem::Finalize();
    allocator.Free(m_pMemoryForSoundSystem);
}

void AtkCommonSetup::FinalizeSoundArchive(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    FinalizeFsSoundArchive(allocator);
}

void AtkCommonSetup::FinalizeFsSoundArchive(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    m_FsSoundArchive.Close();
    if( m_pMemoryForLabelStringData != nullptr )
    {
        allocator.Free(m_pMemoryForLabelStringData);
    }
    allocator.Free(m_pMemoryForInfoBlock);
}

void AtkCommonSetup::FinalizeSoundDataManager(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    m_SoundDataManager.Finalize();
    allocator.Free(m_pMemoryForSoundDataManager);
}

void AtkCommonSetup::FinalizeSoundArchivePlayer(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    m_SoundArchivePlayer.Finalize();
    nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPoolForStreamBuffer );
    allocator.Free(m_pMemoryForStreamBuffer);
    allocator.Free(m_pMemoryForSoundArchivePlayer);
}

void AtkCommonSetup::FinalizeSoundHeap(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    m_SoundHeap.Destroy();
    allocator.Free(m_pMemoryForSoundHeap);
}

////////////////////////////////////////////////////////////////////////////////////
// Atk Other Process
////////////////////////////////////////////////////////////////////////////////////
void AtkCommonSetup::LoadData(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel) NN_NOEXCEPT
{
    NN_UNUSED(itemLabel);
    NN_ABORT_UNLESS(m_SoundDataManager.LoadData(itemId, &m_SoundHeap), "LoadData(%s) failed.", itemLabel);
}

void AtkCommonSetup::LoadData(const char* itemLabel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_SoundDataManager.LoadData(itemLabel, &m_SoundHeap), "LoadData(%s) failed.", itemLabel);
}

void AtkCommonSetup::LoadLabelStringData(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
{
    const size_t size = m_FsSoundArchive.GetLabelStringDataSize();
    m_pMemoryForLabelStringData = AllocateUninitializedMemory( allocator, size, nn::atk::FsSoundArchive::BufferAlignSize );
    NN_ABORT_UNLESS_NOT_NULL(m_pMemoryForLabelStringData);
    m_FsSoundArchive.LoadLabelStringData( m_pMemoryForLabelStringData, size );
}

nn::atk::SoundArchive& AtkCommonSetup::GetSoundArchive() NN_NOEXCEPT
{
    return m_FsSoundArchive;
}

nn::atk::SoundDataManager& AtkCommonSetup::GetSoundDataManager() NN_NOEXCEPT
{
    return m_SoundDataManager;
}

nn::atk::SoundArchivePlayer& AtkCommonSetup::GetSoundArchivePlayer() NN_NOEXCEPT
{
    return m_SoundArchivePlayer;
}

nn::atk::SoundHeap& AtkCommonSetup::GetSoundHeap() NN_NOEXCEPT
{
    return m_SoundHeap;
}

}}}
