﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/sf_ObjectImplFactory.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/sf/detail/sf_AutogenInterfaceIncludes.h>

#define MY_EXPECT_EQ(a, b) EXPECT_TRUE((a) == (b))

class IFoo : public nn::sf::IServiceObject
{
private:

    virtual int sync_Bar(int n) NN_NOEXCEPT = 0;

public:

    int Bar(int n) NN_NOEXCEPT
    {
        return this->sync_Bar(n);
    }

};

namespace {

using namespace nn::sf::impl;

// 1MiB の領域を先頭から使っていくアロケート関数
// 状態無しアロケータの実装に使用する
int64_t g_AllocationBuffer[1024 * 1024];
int64_t* g_AllocationHead = g_AllocationBuffer;
void* AllocateForTest(size_t size)
{
    auto n = (size - 1) / sizeof(int64_t) + 1;
    auto ret = g_AllocationHead;
    g_AllocationHead += n;
    return ret;
}
void DeallocateForTest(void* p, size_t size)
{
    NN_UNUSED(p);
    NN_UNUSED(size);
}

// Foo 実装の定義
// ※ SF を標準的に使用する分には自動生成コードを使用するが、
//    自動生成系などに左右されないため、手動で書いたものを直接利用する。

template <class Impl>
class FooTemplateObjectBase : public Impl, public IFoo
{
private:

    virtual int sync_Bar(int n) NN_NOEXCEPT NN_OVERRIDE
    {
        return static_cast<Impl*>(this)->Bar(n);
    }

protected:

    using Impl::Impl;

};

template <typename Impl, class AllocationPolicy>
class FooTemplate
    : public nn::sf::ObjectImplFactory<FooTemplateObjectBase<Impl>, AllocationPolicy>
{
};

struct FooImpl
{
private:

    int& m_N;

public:

    explicit FooImpl(int& n) NN_NOEXCEPT
        : m_N(n)
    {
    }

    ~FooImpl() NN_NOEXCEPT
    {
        this->m_N = -1;
    }

    int Bar(int n) NN_NOEXCEPT
    {
        return n + m_N;
    }

    void SetN(int n) NN_NOEXCEPT
    {
        this->m_N = n;
    }

};

struct Padding
{
    int padding;
};

// 状態無しのアロケータ向けのテストのための名前空間
namespace StatelessAllocatorTest {

// 状態無しのアロケータ
// - AllocateForTest を実装に用いる
// - 内部でアロケート回数などのカウンタを持つ
// - T でテンプレートとしているのは T ごとに型インスタンスを作成するため
template <typename T>
struct StatelessAllocatorBase
{
    static int AllocateCount;
    static int DeallocateCount;
    static void* LastAllocateAddress;
    static void* LastDeallocateAddress;
    static size_t LastAllocateSize;
    static size_t LastDeallocateSize;
    void* Allocate(size_t size)
    {
        void* ret = AllocateForTest(size);
        ++AllocateCount;
        LastAllocateAddress = ret;
        LastAllocateSize = size;
        return ret;
    }
    void Deallocate(void* p, size_t size)
    {
        DeallocateForTest(p, size);
        ++DeallocateCount;
        LastDeallocateAddress = p;
        LastDeallocateSize = size;
    }
    static void Reset()
    {
        AllocateCount = 0;
        DeallocateCount = 0;
        ResetLast();
    }
    static void ResetLast()
    {
        LastAllocateAddress = 0;
        LastDeallocateAddress = 0;
        LastAllocateSize = 0;
        LastDeallocateSize = 0;
    }
};

template <typename T>
int StatelessAllocatorBase<T>::AllocateCount;

template <typename T>
int StatelessAllocatorBase<T>::DeallocateCount;

template <typename T>
void* StatelessAllocatorBase<T>::LastAllocateAddress;

template <typename T>
void* StatelessAllocatorBase<T>::LastDeallocateAddress;

template <typename T>
size_t StatelessAllocatorBase<T>::LastAllocateSize;

template <typename T>
size_t StatelessAllocatorBase<T>::LastDeallocateSize;

typedef StatelessAllocatorBase<void> StatelessAllocator;

// FooTemplate の直使用(typedef)による Foo 実装
// 全型共通の StatelessAllocator を使用
typedef FooTemplate<FooImpl, StatelessAllocationPolicy<StatelessAllocator>> Foo1;

// FooTemplate の継承による Foo 実装
// 全型共通の StatelessAllocator を使用
class Foo2 : public FooTemplate<Foo2, StatelessAllocationPolicy<StatelessAllocator>>, public FooImpl
{
public:
    explicit Foo2(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

// FooTemplate の継承による Foo 実装
// 全型共通の StatelessAllocator を使用
// (FooTemplate が継承の先頭ではない → Foo3* と FooTemplate* とでオフセットがずれても大丈夫なことを確認)
class Foo3 : public Padding, public FooTemplate<Foo3, StatelessAllocationPolicy<StatelessAllocator>>, public FooImpl
{
public:
    explicit Foo3(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

template <typename T>
struct StatelessAllocatorT : public StatelessAllocatorBase<T>
{
    void* Allocate(size_t size)
    {
        MY_EXPECT_EQ(sizeof(T), size);
        return StatelessAllocatorBase<T>().Allocate(size);
    }
    void Deallocate(void* p, size_t size)
    {
        MY_EXPECT_EQ(sizeof(T), size);
        return StatelessAllocatorBase<T>().Deallocate(p, size);
    }
};

// 型ごとの StatelessAllocator を使用
typedef FooTemplate<FooImpl, StatelessTypedAllocationPolicy<StatelessAllocatorT>> Foo1T;

// 型ごとの StatelessAllocator を使用
class Foo2T : public FooTemplate<Foo2T, StatelessTypedAllocationPolicy<StatelessAllocatorT>>, public FooImpl
{
public:
    explicit Foo2T(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

// 型ごとの StatelessAllocator を使用
class Foo3T : public Padding, public FooTemplate<Foo3T, StatelessTypedAllocationPolicy<StatelessAllocatorT>>, public FooImpl
{
public:
    explicit Foo3T(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

template <typename Foo>
void TestFoo()
{
    typedef typename Foo::StatelessAllocator Allocator;
    Allocator::Reset();
    for (auto i = 0; i < 10; ++i)
    {
        Allocator::ResetLast();
        // これまでのアロケート・デアロケート回数が i と一致することをテスト
        MY_EXPECT_EQ(i, Allocator::AllocateCount);
        MY_EXPECT_EQ(i, Allocator::DeallocateCount);
        void* allocated;
        int n = i;
        {
            auto impl = Foo::Create(n);
            auto p = nn::sf::SharedPointer<IFoo>(impl, false);
            // 前回のテストからアロケートが一回行われたことをテスト
            MY_EXPECT_EQ(i + 1, Allocator::AllocateCount);
            // 前回のテストからデアロケートが行われていないことのテスト
            MY_EXPECT_EQ(i, Allocator::DeallocateCount);
            // 今回のアロケートでアロケートされたアドレスを保存
            // あとでこのアドレスに対してデアロケートが呼ばれることをテスト
            allocated = Allocator::LastAllocateAddress;
            // 今回のアロケートサイズをテスト
            MY_EXPECT_EQ(sizeof(typename Foo::Object), Allocator::LastAllocateSize);
            // 正しくアロケートされており、オブジェクトへの操作ができることテスト
            MY_EXPECT_EQ(i + 20, p->Bar(20));
            impl->SetN(10);
            MY_EXPECT_EQ(10 + 20, p->Bar(20));
            // まだ FooImpl のデストラクタが動いていないことのチェック (デストラクタで n = -1 される)
            MY_EXPECT_EQ(10, n);
        }
        // FooImpl のデストラクタが動いたことのチェック
        MY_EXPECT_EQ(-1, n);
        // アロケート・デアロケート回数が (i + 1) と一致することをテスト
        MY_EXPECT_EQ(i + 1, Allocator::AllocateCount);
        MY_EXPECT_EQ(i + 1, Allocator::DeallocateCount);
        // 保存されたアロケートアドレスから変わっていないことのチェック
        MY_EXPECT_EQ(allocated, Allocator::LastAllocateAddress);
        // 保存されたアロケートアドレスが直前でデアロケートされたことのチェック
        MY_EXPECT_EQ(allocated, Allocator::LastDeallocateAddress);
        // 直前でデアロケートされたサイズのチェック
        MY_EXPECT_EQ(sizeof(typename Foo::Object), Allocator::LastDeallocateSize);
    }
}

TEST(sf, Impl_ImplAdapterWithStatelessAllocator)
{
    // { 直使用, 継承, パディング付き } * { 共通アロケータ, 型ごとのアロケータ } の組み合わせでテスト
    TestFoo<Foo1>();
    TestFoo<Foo2>();
    TestFoo<Foo3>();
    TestFoo<Foo1T>();
    TestFoo<Foo2T>();
    TestFoo<Foo3T>();
}

TEST(sf, Impl_ImplRefCount)
{
    StatelessAllocator::Reset();
    int n = 0;
    {
        auto p = nn::sf::SharedPointer<IFoo>(Foo1::Create(n), false);
        // 参照カウント == 1
        MY_EXPECT_EQ(1, StatelessAllocator::AllocateCount);
        MY_EXPECT_EQ(0, StatelessAllocator::DeallocateCount);
        p.Get()->AddReference();
        // 参照カウント == 2
        MY_EXPECT_EQ(1, StatelessAllocator::AllocateCount);
        MY_EXPECT_EQ(0, StatelessAllocator::DeallocateCount);
        p.Get()->Release();
        // 参照カウント == 1
        MY_EXPECT_EQ(1, StatelessAllocator::AllocateCount);
        MY_EXPECT_EQ(0, StatelessAllocator::DeallocateCount);
    }
    // 参照カウント == 0
    MY_EXPECT_EQ(1, StatelessAllocator::AllocateCount);
    MY_EXPECT_EQ(1, StatelessAllocator::DeallocateCount);
}

} // StatelessAllocatorTest

// 状態ありのアロケータ向けのテストのための名前空間
// (テスト内容は状態無しアロケータの場合と同等)
namespace StatefulAllocatorTest {

struct StatefulAllocator
{
    int AllocateCount;
    int DeallocateCount;
    void* LastAllocateAddress;
    void* LastDeallocateAddress;
    size_t LastAllocateSize;
    size_t LastDeallocateSize;
    void* Allocate(size_t size)
    {
        void* ret = AllocateForTest(size);
        ++AllocateCount;
        LastAllocateAddress = ret;
        LastAllocateSize = size;
        return ret;
    }
    void Deallocate(void* p, size_t size)
    {
        DeallocateForTest(p, size);
        ++DeallocateCount;
        LastDeallocateAddress = p;
        LastDeallocateSize = size;
    }
    void Reset()
    {
        AllocateCount = 0;
        DeallocateCount = 0;
        ResetLast();
    }
    void ResetLast()
    {
        LastAllocateAddress = 0;
        LastDeallocateAddress = 0;
        LastAllocateSize = 0;
        LastDeallocateSize = 0;
    }
    StatefulAllocator()
    {
        Reset();
    }
};

typedef FooTemplate<FooImpl, StatefulAllocationPolicy<StatefulAllocator>> Foo1;

class Foo2 : public FooTemplate<Foo2, StatefulAllocationPolicy<StatefulAllocator>>, public FooImpl
{
public:
    explicit Foo2(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

class Foo3 : public Padding, public FooTemplate<Foo3, StatefulAllocationPolicy<StatefulAllocator>>, public FooImpl
{
public:
    explicit Foo3(int& n) NN_NOEXCEPT : FooImpl(n) {}
};

template <class T>
struct StatefulAllocatorT : public StatefulAllocator
{
    void* Allocate(size_t size)
    {
        MY_EXPECT_EQ(sizeof(T), size);
        void* ret = AllocateForTest(size);
        ++AllocateCount;
        LastAllocateAddress = ret;
        LastAllocateSize = size;
        return ret;
    }
    void Deallocate(void* p, size_t size)
    {
        MY_EXPECT_EQ(sizeof(T), size);
        DeallocateForTest(p, size);
        ++DeallocateCount;
        LastDeallocateAddress = p;
        LastDeallocateSize = size;
    }
    void Reset()
    {
        AllocateCount = 0;
        DeallocateCount = 0;
        ResetLast();
    }
    void ResetLast()
    {
        LastAllocateAddress = 0;
        LastDeallocateAddress = 0;
        LastAllocateSize = 0;
        LastDeallocateSize = 0;
    }
    StatefulAllocatorT()
    {
        Reset();
    }
};

template <typename Foo>
void TestFoo()
{
    typename Foo::Allocator a;
    for (auto i = 0; i < 10; ++i)
    {
        a.ResetLast();
        MY_EXPECT_EQ(i, a.AllocateCount);
        MY_EXPECT_EQ(i, a.DeallocateCount);
        int n = i;
        void* allocated;
        {
            auto impl = Foo::Create(&a, n);
            auto p = nn::sf::SharedPointer<IFoo>(impl, false);
            MY_EXPECT_EQ(i + 1, a.AllocateCount);
            MY_EXPECT_EQ(i, a.DeallocateCount);
            allocated = a.LastAllocateAddress;
            MY_EXPECT_EQ(sizeof(typename Foo::Object), a.LastAllocateSize);
            MY_EXPECT_EQ(i + 20, p->Bar(20));
            impl->SetN(10);
            MY_EXPECT_EQ(10 + 20, p->Bar(20));
            MY_EXPECT_EQ(10, n);
        }
        MY_EXPECT_EQ(-1, n);
        MY_EXPECT_EQ(i + 1, a.AllocateCount);
        MY_EXPECT_EQ(i + 1, a.DeallocateCount);
        MY_EXPECT_EQ(allocated, a.LastAllocateAddress);
        MY_EXPECT_EQ(allocated, a.LastDeallocateAddress);
        MY_EXPECT_EQ(sizeof(typename Foo::Object), a.LastDeallocateSize);
    }
}

TEST(sf, Impl_ImplAdapterWithStatefulAllocator)
{
    TestFoo<Foo1>();
    TestFoo<Foo2>();
    TestFoo<Foo3>();
}

#if 0
// 例外が有効なときのみにテスト
// TODO: 上記条件で実装分岐ができるようになるまでは #if 0 固定

struct FooImplException
{
};
struct FooImplWithException : public FooImpl
{
    explicit FooImplWithException(int n)
        : FooImpl(n)
    {
        // 動的な分岐がないと、到達不可コードの警告が出るため、動的な分岐を挿入
        if (n == 0)
        {
            throw FooImplException();
        }
    }
};

class FooE : public Padding, public FooTemplate<FooE, StatefulTypedAllocationPolicy<StatefulAllocatorT>>, public FooImplWithException
{
public:
    explicit FooE(int& n) : FooImplWithException(n) {}
};

TEST(sf, Impl_ImplAdapterWithStatefulAllocatorException)
{
    // コンストラクタ中での例外時の巻き戻しの実装チェック
    // 以下のシーケンスであることをチェック
    //   - アロケート
    //   - コンストラクタ起動
    //     - ここで throw
    //   - * デストラクタは起動しない
    //   - デアロケート
    FooE::Allocator a;
    nn::sf::SharedPointer<IFoo> p;
    int n = 0;
    ASSERT_THROW(p = FooE::CreateShared(&a, n), FooImplException);
    // p が書き換わっていないことのテスト
    MY_EXPECT_EQ(0, p);
    // デストラクタが呼ばれていないことのテスト(呼ばれていたら n == -1)
    MY_EXPECT_EQ(0, n);
    // アロケートとデアロケートの回数が 1 回ずつであることのテスト
    MY_EXPECT_EQ(1, a.AllocateCount);
    MY_EXPECT_EQ(1, a.DeallocateCount);
    // アロケートアドレスとデアロケートアドレスが一致していることのテスト
    MY_EXPECT_EQ(a.LastAllocateAddress, a.LastDeallocateAddress);
}
#endif

} // StatelessAllocatorTest

} // anonymous namespace
