﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>

#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkAssert.h>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nnt/sfutil/sfutil_MultiWaitUtils.h>
#include <cstring>
#include <nn/os/os_Event.h>
#include <nn/util/util_ScopeExit.h>

using namespace nn::sf::hipc;
using namespace nnt::sfutil;

TEST(sf_hipc, ServiceResolution_Initialize)
{
    InitializeHipcServiceResolution();
    FinalizeHipcServiceResolution();
}

TEST(sf_hipc, ServiceResolution_Register)
{
    const char ServiceName[] = "12345678";
    NN_SDK_ASSERT_EQUAL(HipcServiceNameLengthMax, std::strlen(ServiceName));
    InitializeHipcServiceResolution();
    HipcServerPortHandle handle;
    ASSERT_TRUE(RegisterHipcService(&handle, 10, ServiceName).IsSuccess());
    CloseServerPortHandle(handle);
    UnregisterHipcService(ServiceName);
    FinalizeHipcServiceResolution();
}

namespace {

NN_ALIGNAS(4096) char g_ThreadStack[4096];
TestThread g_Thread;

template <typename T>
void InitializeServerThread(void (*f)(T*), T* p) NN_NOEXCEPT
{
    g_Thread.Initialize(f, p, g_ThreadStack, sizeof(g_ThreadStack), 10);
    g_Thread.Start();
}

void FinalizeServerThread() NN_NOEXCEPT
{
    g_Thread.Finalize();
}

}

namespace {

struct ArgumentForConnectTo
{
    HipcServerPortHandle portHandle;
    nn::os::Event* pShutdownEvent;
};

void ThreadForConnectTo(ArgumentForConnectTo* p)
{
    auto portHandle = p->portHandle;

    MultiWaitHodler shutdownEventHolder;
    nn::os::InitializeMultiWaitHolder(shutdownEventHolder.GetBaseForAttach(), p->pShutdownEvent->GetBase());
    MultiWaitHodler portHolder;
    AttachWaitHolderForAccept(portHolder.GetBaseForAttach(), portHandle);

    MultiWaiter multiWaiter;
    multiWaiter.Link(&shutdownEventHolder);
    multiWaiter.Link(&portHolder);

    for (;;)
    {
        auto timeout = nn::TimeSpan::FromSeconds(10);
        auto signaled = multiWaiter.TimedWaitAny(timeout, false);
        if (!signaled)
        {
            // タイムアウトによるテスト失敗
            FAIL();
            break;
        }
        if (signaled == &shutdownEventHolder)
        {
            break;
        }
        else
        {
            NN_SDK_ASSERT(signaled == &portHolder);
            HipcServerSessionHandle sessionHandle;
            ASSERT_TRUE(AcceptSession(&sessionHandle, portHandle).IsSuccess());
            CloseServerSessionHandle(sessionHandle);
        }
    }

    multiWaiter.Unlink(&portHolder);
    multiWaiter.Unlink(&shutdownEventHolder);

    CloseServerPortHandle(portHandle);
}

}

TEST(sf_hipc, ServiceResolution_ConnectTo)
{
    const char ServiceName[] = "test_Sf";
    const int MaxSessions = 10;
    for (int i = 0; i < 10; ++i)
    {
        nn::os::Event shutdownEvent(nn::os::EventClearMode_ManualClear);
        InitializeHipcServiceResolution();
        HipcServerPortHandle portHandle;
        ASSERT_TRUE(RegisterHipcService(&portHandle, MaxSessions, ServiceName).IsSuccess());
        ArgumentForConnectTo a =
        {
            portHandle,
            &shutdownEvent,
        };
        InitializeServerThread(ThreadForConnectTo, &a);
        NN_UTIL_SCOPE_EXIT
        {
            shutdownEvent.Signal();
            FinalizeServerThread();
            UnregisterHipcService(ServiceName);
            FinalizeHipcServiceResolution();
        };

        for (int j = 0; j < MaxSessions; ++j)
        {
            HipcClientSessionHandle handle;
            ASSERT_TRUE(ConnectToHipcService(&handle, ServiceName).IsSuccess());
            CloseClientSessionHandle(handle);
        }

        {
            HipcClientSessionHandle handles[MaxSessions];
            for (int j = 0; j < MaxSessions; ++j)
            {
                ASSERT_TRUE(ConnectToHipcService(&handles[j], ServiceName).IsSuccess());
            }
            for (int j = 0; j < MaxSessions; ++j)
            {
                CloseClientSessionHandle(handles[j]);
            }
        }
    }
}
