﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_MemoryResource.h>
#include <nn/sf/sf_LmemUtility.h>
#include <nn/sf/sf_MemUtility.h>

// SF インターフェイス
#include "testSf_IAllFunctionTests.sfdl.h"

// 実装
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <memory>
#include <nn/sf/sf_StandardAllocationPolicy.h>
#include <nn/util/util_ScopeExit.h>

namespace {

typedef nn::sf::impl::ExpHeapStaticAllocator<1024 * 1024> MyStatelessAllocator;
typedef MyStatelessAllocator::Policy MyStatelessAllocationPolicy;

struct MyStatelessAllocatorInitializer
{
    MyStatelessAllocatorInitializer() NN_NOEXCEPT
    {
        MyStatelessAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyStatelessAllocatorInitializer;

class MyStatefulAllocator
    : public nn::sf::impl::ExpHeapAllocator
{
private:
    char m_Buffer[1024 * 1024];
public:
    MyStatefulAllocator() NN_NOEXCEPT
    {
        auto handle = nn::lmem::CreateExpHeap(m_Buffer, sizeof(m_Buffer), nn::lmem::CreationOption_NoOption);
        ExpHeapAllocator::Attach(handle);
    }
} g_MyStatefulAllocator;
typedef MyStatefulAllocator::Policy MyStatefulAllocationPolicy;

class TestSessionImpl
{
private:
    int m_Value;
    int* m_P;
    int m_DestroyValue;
public:
    TestSessionImpl() NN_NOEXCEPT
        : m_Value(0)
        , m_P(nullptr)
    {
    }
    explicit TestSessionImpl(int x) NN_NOEXCEPT
        : m_Value(x)
        , m_P(nullptr)
    {
    }
    void SetOnDestroy(int* p, int value) NN_NOEXCEPT
    {
        this->m_P = p;
        this->m_DestroyValue = value;
    }
    ~TestSessionImpl() NN_NOEXCEPT
    {
        if (m_P)
        {
            *m_P = m_DestroyValue;
        }
    }
    int GetValue() NN_NOEXCEPT
    {
        return m_Value;
    }
    void SetValue(int x) NN_NOEXCEPT
    {
        this->m_Value = x;
    }
    static void* operator new(size_t size) NN_NOEXCEPT
    {
        return MyStatelessAllocator::Allocate(size);
    }
    static void operator delete(void* p, size_t size) NN_NOEXCEPT
    {
        MyStatelessAllocator::Deallocate(p, size);
    }
};

class SharedTestSessionImpl
    : public TestSessionImpl
    , public nn::sf::ISharedObject
{
public:
    SharedTestSessionImpl() NN_NOEXCEPT
    {
    }
    explicit SharedTestSessionImpl(int x) NN_NOEXCEPT
        : TestSessionImpl(x)
    {
    }
};

}

TEST(sf_AllInOne, Factory_UnmanagedServiceObject)
{
    {
        auto d = 0;
        {
            nn::sf::UnmanagedServiceObject<nnt::testsf::ITestSession, TestSessionImpl> x;
            auto p = x.GetShared();
            auto& impl = x.GetImpl();
            ASSERT_EQ(0, p->GetValue());
            impl.SetValue(10);
            ASSERT_EQ(10, p->GetValue());
            impl.SetOnDestroy(&d, 10);
        }
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            nn::sf::UnmanagedServiceObject<nnt::testsf::ITestSession, TestSessionImpl> x(10);
            auto p = x.GetShared();
            auto& impl = x.GetImpl();
            ASSERT_EQ(10, p->GetValue());
            impl.SetValue(20);
            ASSERT_EQ(20, p->GetValue());
            impl.SetOnDestroy(&d, 10);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_UnmanagedServiceObjectByPointer)
{
    {
        auto d = 0;
        {
            TestSessionImpl impl;
            {
                nn::sf::UnmanagedServiceObjectByPointer<nnt::testsf::ITestSession, TestSessionImpl> x(&impl);
                auto p = x.GetShared();
                ASSERT_EQ(0, x.GetValue());
                impl.SetValue(10);
                ASSERT_EQ(10, x.GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            TestSessionImpl impl(10);
            {
                nn::sf::UnmanagedServiceObjectByPointer<nnt::testsf::ITestSession, TestSessionImpl> x(&impl);
                auto p = x.GetShared();
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_Stateless_Emplaced)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    typedef TestSessionImpl Impl;
    {
        auto d = 0;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>();
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(0, p->GetValue());
                impl.SetValue(10);
                ASSERT_EQ(10, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>(10);
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_Stateful_Emplaced)
{
    typedef nn::sf::ObjectFactory<MyStatefulAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    typedef TestSessionImpl Impl;
    auto& allocator = g_MyStatefulAllocator;
    {
        auto d = 0;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>(&allocator);
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(0, p->GetValue());
                impl.SetValue(10);
                ASSERT_EQ(10, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>(&allocator, 10);
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    typedef nn::sf::StatefulObjectFactory<MyStatefulAllocationPolicy> StatefulFactory;
    StatefulFactory factory(&allocator);
    {
        auto d = 0;
        {
            auto p = factory.CreateSharedEmplaced<Interface, Impl>();
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(0, p->GetValue());
                impl.SetValue(10);
                ASSERT_EQ(10, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            auto p = factory.CreateSharedEmplaced<Interface, Impl>(10);
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_Stateless_SharedEmplaced)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    typedef SharedTestSessionImpl Impl;
    {
        auto d = 0;
        nn::sf::ISharedObject* pShared;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>();
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(0, p->GetValue());
                impl.SetValue(10);
                ASSERT_EQ(10, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
            pShared = Factory::GetEmplacedImplPointer<Impl>(p);
            pShared->AddReference();
        }
        ASSERT_EQ(0, d);
        pShared->Release();
        ASSERT_EQ(10, d);
    }
    {
        auto d = 0;
        {
            auto p = Factory::CreateSharedEmplaced<Interface, Impl>(10);
            {
                auto& impl = p.GetImpl();
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_Stateless_SharedPointer)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    typedef TestSessionImpl Impl;
    {
        auto d = 0;
        {
            auto sp = Factory::CreateSharedEmplaced<Interface, Impl>(10);
            auto& impl = *Factory::GetEmplacedImplPointer<Impl>(sp);
            auto p = Factory::CreateShared<Interface>(std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
        {
            auto sp = Factory::CreateSharedEmplaced<Interface, Impl>(10);
            auto& impl = *Factory::GetEmplacedImplPointer<Impl>(sp);
            auto p = Factory::CreateSharedEmplaced<Interface, decltype(sp)>(std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 30);
            }
            ASSERT_EQ(10, d);
        }
        ASSERT_EQ(30, d);
    }
}

TEST(sf_AllInOne, Factory_Stateful_SharedPointer)
{
    typedef nn::sf::ObjectFactory<MyStatefulAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    typedef TestSessionImpl Impl;
    auto& allocator = g_MyStatefulAllocator;
    {
        auto d = 0;
        {
            auto sp = Factory::CreateSharedEmplaced<Interface, Impl>(&allocator, 10);
            auto& impl = *Factory::GetEmplacedImplPointer<Impl>(sp);
            auto p = Factory::CreateShared<Interface>(&allocator, std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
        {
            auto sp = Factory::CreateSharedEmplaced<Interface, Impl>(&allocator, 10);
            auto& impl = *Factory::GetEmplacedImplPointer<Impl>(sp);
            auto p = Factory::CreateSharedEmplaced<Interface, decltype(sp)>(&allocator, std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 30);
            }
            ASSERT_EQ(10, d);
        }
        ASSERT_EQ(30, d);
    }
    typedef nn::sf::StatefulObjectFactory<MyStatefulAllocationPolicy> StatefulFactory;
    StatefulFactory factory(&allocator);
    {
        auto d = 0;
        {
            auto sp = factory.CreateSharedEmplaced<Interface, Impl>(10);
            auto& impl = *StatefulFactory::GetEmplacedImplPointer<Impl>(sp);
            auto p = factory.CreateShared<Interface>(std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
        {
            auto sp = factory.CreateSharedEmplaced<Interface, Impl>(10);
            auto& impl = *StatefulFactory::GetEmplacedImplPointer<Impl>(sp);
            auto p = factory.CreateSharedEmplaced<Interface, decltype(sp)>(std::move(sp));
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 30);
            }
            ASSERT_EQ(10, d);
        }
        ASSERT_EQ(30, d);
    }
}

TEST(sf_AllInOne, Factory_Stateless_StdUnique)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    {
        auto d = 0;
        {
            auto pImpl = new TestSessionImpl(10);
            std::unique_ptr<TestSessionImpl> u(pImpl);
            auto p = Factory::CreateShared<Interface>(std::move(u));
            {
                ASSERT_TRUE(!u);
                ASSERT_EQ(10, p->GetValue());
                pImpl->SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                pImpl->SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

TEST(sf_AllInOne, Factory_Stateful_StdUnique)
{
    typedef nn::sf::ObjectFactory<MyStatefulAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    auto& allocator = g_MyStatefulAllocator;
    {
        auto d = 0;
        {
            auto pImpl = new TestSessionImpl(10);
            std::unique_ptr<TestSessionImpl> u(pImpl);
            auto p = Factory::CreateShared<Interface>(&allocator, std::move(u));
            {
                ASSERT_TRUE(!u);
                ASSERT_EQ(10, p->GetValue());
                pImpl->SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                pImpl->SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
    typedef nn::sf::StatefulObjectFactory<MyStatefulAllocationPolicy> StatefulFactory;
    StatefulFactory factory(&allocator);
    {
        auto d = 0;
        {
            auto pImpl = new TestSessionImpl(10);
            std::unique_ptr<TestSessionImpl> u(pImpl);
            auto p = factory.CreateShared<Interface>(std::move(u));
            {
                ASSERT_TRUE(!u);
                ASSERT_EQ(10, p->GetValue());
                pImpl->SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                pImpl->SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(10, d);
    }
}

// TODO:
// TEST(sf_AllInOne, Factory_Stateless/ful_StdShared)
//   std::shared_ptr が内部アロケートをするため、後回し
// TEST(sf_AllInOne, Factory_Stateless/ful_Pointer)
//   煩雑で、かつ、需要が少なそうなので後回し

TEST(sf_AllInOne, Factory_Stateless_PointerWithoutManagement)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    {
        TestSessionImpl impl(10);
        auto d = 0;
        {
            auto p = Factory::CreateSharedWithoutManagement<Interface>(&impl);
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(0, d);
    }
}

TEST(sf_AllInOne, Factory_Stateful_PointerWithoutManagement)
{
    typedef nn::sf::ObjectFactory<MyStatefulAllocationPolicy> Factory;
    typedef nnt::testsf::ITestSession Interface;
    auto& allocator = g_MyStatefulAllocator;
    {
        TestSessionImpl impl(10);
        auto d = 0;
        {
            auto p = Factory::CreateSharedWithoutManagement<Interface>(&allocator, &impl);
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(0, d);
    }
    typedef nn::sf::StatefulObjectFactory<MyStatefulAllocationPolicy> StatefulFactory;
    StatefulFactory factory(&allocator);
    {
        TestSessionImpl impl(10);
        auto d = 0;
        {
            auto p = factory.CreateSharedWithoutManagement<Interface>(&impl);
            {
                ASSERT_EQ(10, p->GetValue());
                impl.SetValue(20);
                ASSERT_EQ(20, p->GetValue());
                impl.SetOnDestroy(&d, 10);
            }
            ASSERT_EQ(0, d);
        }
        ASSERT_EQ(0, d);
    }
}

class NullMemoryResouce
    : public nn::MemoryResource
{
protected:

    virtual void* do_allocate(std::size_t, std::size_t) NN_NOEXCEPT
    {
        return nullptr;
    }

    virtual void do_deallocate(void*, std::size_t, std::size_t) NN_NOEXCEPT
    {
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
    {
        return this == &other;
    }

};

TEST(sf_AllInOne, Factory_SetDefaultAllocator)
{
    EXPECT_TRUE(nn::sf::GetNewDeleteMemoryResource() == nn::sf::GetCurrentEffectiveMemoryResource());
    {
        EXPECT_TRUE(nullptr == nn::sf::GetGlobalDefaultMemoryResource());
        NullMemoryResouce mr1;
        EXPECT_TRUE(nullptr == nn::sf::SetGlobalDefaultMemoryResource(&mr1));
        EXPECT_TRUE(&mr1 == nn::sf::GetGlobalDefaultMemoryResource());
        EXPECT_TRUE(&mr1 == nn::sf::GetCurrentEffectiveMemoryResource());
        EXPECT_TRUE(&mr1 == nn::sf::SetGlobalDefaultMemoryResource(nullptr));
        EXPECT_TRUE(nullptr == nn::sf::GetGlobalDefaultMemoryResource());
        EXPECT_TRUE(nn::sf::GetNewDeleteMemoryResource() == nn::sf::GetCurrentEffectiveMemoryResource());
    }
    {
        EXPECT_TRUE(nullptr == nn::sf::GetCurrentMemoryResource());
        NullMemoryResouce mr1;
        NullMemoryResouce mr2;
        EXPECT_TRUE(nullptr == nn::sf::SetCurrentMemoryResource(&mr2));
        EXPECT_TRUE(&mr2 == nn::sf::GetCurrentMemoryResource());
        EXPECT_TRUE(&mr2 == nn::sf::GetCurrentEffectiveMemoryResource());
        nn::sf::SetGlobalDefaultMemoryResource(&mr1);
        EXPECT_TRUE(&mr2 == nn::sf::GetCurrentEffectiveMemoryResource());
        EXPECT_TRUE(&mr2 == nn::sf::SetCurrentMemoryResource(nullptr));
        EXPECT_TRUE(&mr1 == nn::sf::GetCurrentEffectiveMemoryResource());
        nn::sf::SetGlobalDefaultMemoryResource(nullptr);
    }
    {
        NullMemoryResouce mr1;
        NullMemoryResouce mr2;
        {
            nn::sf::ScopedCurrentMemoryResourceSetter rs1(&mr1);
            EXPECT_TRUE(&mr1 == nn::sf::GetCurrentMemoryResource());
            EXPECT_TRUE(&mr1 == nn::sf::GetCurrentEffectiveMemoryResource());
            {
                nn::sf::ScopedCurrentMemoryResourceSetter rs2(&mr2);
                EXPECT_TRUE(&mr2 == nn::sf::GetCurrentMemoryResource());
                EXPECT_TRUE(&mr2 == nn::sf::GetCurrentEffectiveMemoryResource());
            }
            EXPECT_TRUE(&mr1 == nn::sf::GetCurrentMemoryResource());
            EXPECT_TRUE(&mr1 == nn::sf::GetCurrentEffectiveMemoryResource());
        }
        EXPECT_TRUE(nullptr == nn::sf::GetCurrentMemoryResource());
        EXPECT_TRUE(nn::sf::GetNewDeleteMemoryResource() == nn::sf::GetCurrentEffectiveMemoryResource());
    }
}

class CountuingMemoryResouce
    : public nn::MemoryResource
{
public:

    nn::MemoryResource* pMemoryResource;
    int allocatedCount;
    int deallocatedCount;
    std::size_t lastAllocatedSize;
    std::size_t lastDellocatedSize;

    explicit CountuingMemoryResouce(nn::MemoryResource* pMemoryResource = nn::sf::GetNewDeleteMemoryResource()) NN_NOEXCEPT
        : pMemoryResource(pMemoryResource)
        , allocatedCount(0)
        , deallocatedCount(0)
        , lastAllocatedSize(0)
        , lastDellocatedSize(0)
    {
    }

protected:

    virtual void* do_allocate(std::size_t size, std::size_t) NN_NOEXCEPT
    {
        ++allocatedCount;
        lastAllocatedSize = size;
        return ::operator new(size, std::nothrow);
    }

    virtual void do_deallocate(void* p, std::size_t size, std::size_t) NN_NOEXCEPT
    {
        ++deallocatedCount;
        lastDellocatedSize = size;
        return ::operator delete(p, std::nothrow);
    }

    virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
    {
        return this == &other;
    }

};

nn::sf::SharedPointer<nnt::testsf::ITestSession> CreateFoo() NN_NOEXCEPT
{
    return nn::sf::CreateSharedObjectEmplaced<nnt::testsf::ITestSession, TestSessionImpl>();
}

TEST(sf_AllInOne, Factory_AllocateByDefaultAllocator)
{
    {
        CountuingMemoryResouce mr1;
        {
            auto p1 = CreateFoo();
        }
        EXPECT_EQ(0, mr1.allocatedCount);
        EXPECT_EQ(0, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
    {
        CountuingMemoryResouce mr1;
        {
            nn::sf::SetGlobalDefaultMemoryResource(&mr1);
            auto p1 = CreateFoo();
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
        {
            auto p2 = CreateFoo();
            nn::sf::SetGlobalDefaultMemoryResource(nullptr);
        }
        EXPECT_EQ(2, mr1.allocatedCount);
        EXPECT_EQ(2, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
    {
        CountuingMemoryResouce mr1;
        {
            nn::sf::SetCurrentMemoryResource(&mr1);
            auto p1 = CreateFoo();
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
        {
            auto p2 = CreateFoo();
            nn::sf::SetCurrentMemoryResource(nullptr);
        }
        EXPECT_EQ(2, mr1.allocatedCount);
        EXPECT_EQ(2, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}

nn::sf::SharedPointer<nnt::testsf::ITestSession> CreateFooWithMemoryResource(nn::MemoryResource* pMemoryResource) NN_NOEXCEPT
{
    return nn::sf::CreateSharedObjectEmplaced<nnt::testsf::ITestSession, TestSessionImpl>(pMemoryResource);
}

TEST(sf_AllInOne, Factory_AllocateByMemoryResource)
{
    {
        CountuingMemoryResouce mr1;
        {
            auto p1 = CreateFooWithMemoryResource(&mr1);
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}

namespace {

struct MyMemoryResourceStaticAllocatorTag {};
typedef nn::sf::MemoryResourceStaticAllocator<MyMemoryResourceStaticAllocatorTag> MyMemoryResourceStaticAllocator;

}

nn::sf::SharedPointer<nnt::testsf::ITestSession> CreateFooWithMemoryResourceStatic() NN_NOEXCEPT
{
    return nn::sf::ObjectFactory<MyMemoryResourceStaticAllocator::Policy>::CreateSharedEmplaced<nnt::testsf::ITestSession, TestSessionImpl>();
}

TEST(sf_AllInOne, Factory_AllocateByMemoryResourceStatic)
{
    {
        CountuingMemoryResouce mr1;
        MyMemoryResourceStaticAllocator::Initialize(&mr1);
        {
            auto p1 = CreateFooWithMemoryResourceStatic();
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}

TEST(sf_AllInOne, Factory_AllocateByExpHeapMemoryResource)
{
    {
        const auto BufferSize = 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        auto heapHandle = nn::lmem::CreateExpHeap(buffer.get(), BufferSize, nn::lmem::CreationOption_ThreadSafe);
        NN_UTIL_SCOPE_EXIT
        {
            nn::lmem::DestroyExpHeap(heapHandle);
        };
        nn::sf::ExpHeapMemoryResource lmemmr1(heapHandle);
        CountuingMemoryResouce mr1(&lmemmr1);
        {
            auto p1 = CreateFooWithMemoryResource(&mr1);
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}

TEST(sf_AllInOne, Factory_AllocateByUnitHeapMemoryResource)
{
    {
        const auto BufferSize = 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        auto heapHandle = nn::lmem::CreateUnitHeap(buffer.get(), BufferSize, 64, nn::lmem::CreationOption_ThreadSafe);
        NN_UTIL_SCOPE_EXIT
        {
            nn::lmem::DestroyUnitHeap(heapHandle);
        };
        nn::sf::UnitHeapMemoryResource lmemmr1(heapHandle);
        CountuingMemoryResouce mr1(&lmemmr1);
        {
            auto p1 = CreateFooWithMemoryResource(&mr1);
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}

TEST(sf_AllInOne, Factory_AllocateByStandardAllocatorMemoryResource)
{
    {
        const auto BufferSize = 1024 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        nn::mem::StandardAllocator allocator;
        allocator.Initialize(buffer.get(), BufferSize);
        nn::sf::StandardAllocatorMemoryResource memmr1(&allocator);
        CountuingMemoryResouce mr1(&memmr1);
        {
            auto p1 = CreateFooWithMemoryResource(&mr1);
            EXPECT_EQ(1, mr1.allocatedCount);
            EXPECT_EQ(0, mr1.deallocatedCount);
        }
        EXPECT_EQ(1, mr1.allocatedCount);
        EXPECT_EQ(1, mr1.deallocatedCount);
        EXPECT_EQ(mr1.lastAllocatedSize, mr1.lastDellocatedSize);
    }
}
