﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/settings/settings_SystemApplication.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用初回起動設定 A
const ::nn::settings::system::InitialLaunchSettings InitialLaunchSettingsA = {
    { { 0x00000100 } },
    {
        0,
        {
            {
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            }
        }
    }
};

//!< テスト用初回起動設定 B
const ::nn::settings::system::InitialLaunchSettings InitialLaunchSettingsB = {
    { { 0x00010001 } },
    {
        65535,
        {
            {
                0xEF, 0xCD, 0xAB, 0x89, 0x67, 0x45, 0x23, 0x01,
                0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
            }
        }
    }
};

} // namespace

//!< 携帯モード時誤操作防止画面の有無の設定を取得可能か
TEST(SystemApplicationSuite, LockScreenFlagTest1)
{
    ::nn::settings::system::GetLockScreenFlag();
}

//!< 携帯モード時誤操作防止画面の有無の設定を変更可能か
TEST(SystemApplicationSuite, LockScreenFlagTest2)
{
    ::nn::settings::system::SetLockScreenFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetLockScreenFlag());

    ::nn::settings::system::SetLockScreenFlag(true);

    EXPECT_TRUE(::nn::settings::system::GetLockScreenFlag());

    ::nn::settings::system::SetLockScreenFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetLockScreenFlag());
}

//!< カラーセット ID の設定を取得可能か
TEST(SystemApplicationSuite, ColorSetIdTest1)
{
    ::nn::settings::system::GetColorSetId();
}

//!< カラーセット ID の設定を変更可能か
TEST(SystemApplicationSuite, ColorSetIdTest2)
{
    ::nn::settings::system::SetColorSetId(0xBADBABE);

    EXPECT_EQ(0xBADBABE, ::nn::settings::system::GetColorSetId());

    ::nn::settings::system::SetColorSetId(0xACEDACE);

    EXPECT_EQ(0xACEDACE, ::nn::settings::system::GetColorSetId());

    ::nn::settings::system::SetColorSetId(0xBADBABE);

    EXPECT_EQ(0xBADBABE, ::nn::settings::system::GetColorSetId());
}

//!< バッテリー残量のパーセンテージ表示を行うか否か表す値の設定を変更可能か
TEST(SystemApplicationSuite, BatteryPercentageFlagTest1)
{
    ::nn::settings::system::SetBatteryPercentageFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetBatteryPercentageFlag());

    ::nn::settings::system::SetBatteryPercentageFlag(true);

    EXPECT_TRUE(::nn::settings::system::GetBatteryPercentageFlag());

    ::nn::settings::system::SetBatteryPercentageFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetBatteryPercentageFlag());
}

//!< Quest メニューを起動するか否かを表す値を取得可能か
TEST(SystemApplicationSuite, QuestFlagTest1)
{
    ::nn::settings::GetQuestFlag();

    ::nn::settings::system::GetQuestFlag();
}

//!< Quest メニューを起動するか否かを表す値を変更可能か
TEST(SystemApplicationSuite, QuestFlagTest2)
{
    ::nn::settings::system::SetQuestFlag(false);

    EXPECT_FALSE(::nn::settings::GetQuestFlag());

    EXPECT_FALSE(::nn::settings::system::GetQuestFlag());

    ::nn::settings::system::SetQuestFlag(true);

    EXPECT_TRUE(::nn::settings::GetQuestFlag());

    EXPECT_TRUE(::nn::settings::system::GetQuestFlag());

    ::nn::settings::system::SetQuestFlag(false);

    EXPECT_FALSE(::nn::settings::GetQuestFlag());

    EXPECT_FALSE(::nn::settings::system::GetQuestFlag());
}

//!< 初回起動設定を設定可能か
TEST(SystemApplicationSuite, InitialLaunchSettingsTest1)
{
    ::nn::settings::system::InitialLaunchSettings settings = {};

    ::nn::settings::system::SetInitialLaunchSettings(settings);
    ::nn::settings::system::GetInitialLaunchSettings(&settings);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsCompleted>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsUserAddable>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::HasTimeStamp>());
    ::nn::time::SteadyClockTimePoint timeStamp = {};
    EXPECT_EQ(timeStamp.value, settings.timeStamp.value);
    EXPECT_EQ(timeStamp.sourceId, settings.timeStamp.sourceId);

    ::nn::settings::system::SetInitialLaunchSettings(InitialLaunchSettingsA);
    ::nn::settings::system::GetInitialLaunchSettings(&settings);
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsCompleted>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsUserAddable>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::HasTimeStamp>());
    timeStamp = InitialLaunchSettingsA.timeStamp;
    EXPECT_EQ(timeStamp.value, settings.timeStamp.value);
    EXPECT_EQ(timeStamp.sourceId, settings.timeStamp.sourceId);

    ::nn::settings::system::SetInitialLaunchSettings(InitialLaunchSettingsB);
    ::nn::settings::system::GetInitialLaunchSettings(&settings);
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsCompleted>());
    EXPECT_FALSE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::IsUserAddable>());
    EXPECT_TRUE(
        settings.flags.Test<
            ::nn::settings::system::InitialLaunchFlag::HasTimeStamp>());
    timeStamp = InitialLaunchSettingsB.timeStamp;
    EXPECT_EQ(timeStamp.value, settings.timeStamp.value);
    EXPECT_EQ(timeStamp.sourceId, settings.timeStamp.sourceId);
}

//!< 本体の識別名を取得可能か
TEST(SystemApplicationSuite, DeviceNickNameTest1)
{
    ::nn::settings::system::DeviceNickName name = {};

    ::nn::settings::system::GetDeviceNickName(&name);
}

//!< 本体の識別名を設定可能か
TEST(SystemApplicationSuite, DeviceNickNameTest2)
{
    ::nn::settings::system::DeviceNickName name = {};

    for (size_t i = 0; i < sizeof(name.string) - 1; ++i)
    {
        name.string[i] = static_cast<char>('A' + (i % 26));
    }

    ::nn::settings::system::SetDeviceNickName(name);

    for (size_t i = 0; i < sizeof(name.string); ++i)
    {
        name.string[i] = '0';
    }

    ::nn::settings::system::GetDeviceNickName(&name);

    for (size_t i = 0; i < sizeof(name.string) - 1; ++i)
    {
        EXPECT_EQ(static_cast<char>('A' + (i % 26)), name.string[i]);
    }

    EXPECT_EQ('\0', name.string[sizeof(name.string) - 1]);

    for (size_t i = 0; i < sizeof(name.string) - 1; ++i)
    {
        name.string[i] = static_cast<char>('a' + (i % 26));
    }

    ::nn::settings::system::SetDeviceNickName(name);

    for (size_t i = 0; i < sizeof(name.string); ++i)
    {
        name.string[i] = '0';
    }

    ::nn::settings::system::GetDeviceNickName(&name);

    for (size_t i = 0; i < sizeof(name.string) - 1; ++i)
    {
        EXPECT_EQ(static_cast<char>('a' + (i % 26)), name.string[i]);
    }

    EXPECT_EQ('\0', name.string[sizeof(name.string) - 1]);
}

//!< アプレット起動の制御フラグを設定可能か
TEST(SystemApplicationSuite, AppletLaunchFlagsTest1)
{
    ::nn::settings::system::AppletLaunchFlagSet flags = {};

    ::nn::settings::system::SetAppletLaunchFlags(flags);

    flags.Reset();

    ::nn::settings::system::GetAppletLaunchFlags(&flags);

    EXPECT_FALSE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Gift>());
    EXPECT_FALSE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Migration>());

    flags.Reset();
    flags.Set<::nn::settings::system::AppletLaunchFlag::Gift>();

    ::nn::settings::system::SetAppletLaunchFlags(flags);

    flags.Reset();

    ::nn::settings::system::GetAppletLaunchFlags(&flags);

    EXPECT_TRUE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Gift>());
    EXPECT_FALSE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Migration>());

    flags.Reset();
    flags.Set<::nn::settings::system::AppletLaunchFlag::Migration>();

    ::nn::settings::system::SetAppletLaunchFlags(flags);

    flags.Reset();

    ::nn::settings::system::GetAppletLaunchFlags(&flags);

    EXPECT_FALSE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Gift>());
    EXPECT_TRUE(
        flags.Test<::nn::settings::system::AppletLaunchFlag::Migration>());
}
