﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_News.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用通知設定 A
const ::nn::settings::system::NotificationSettings NotificationSettingsA = {
    { { 0x00000101 } },
    static_cast<int32_t>(::nn::settings::system::NotificationVolume_Low),
    { 1, 2 },
    { 3, 4 },
};

//!< テスト用通知設定 B
const ::nn::settings::system::NotificationSettings NotificationSettingsB = {
    { { 0x00000202 } },
    static_cast<int32_t>(::nn::settings::system::NotificationVolume_Mute),
    { 5, 6 },
    { 7, 8 },
};

//!< テスト用アカウント毎の通知設定 A
const ::nn::settings::system::AccountNotificationSettings
    AccountNotificationSettingsA =
{
    { { 1, 2 } },
    { { 0x0101 } },
    static_cast<int8_t>(
        ::nn::settings::system::FriendPresenceOverlayPermission_NoDisplay),
};

//!< テスト用アカウント毎の通知設定 B
const ::nn::settings::system::AccountNotificationSettings
    AccountNotificationSettingsB =
{
    { { 3, 4 } },
    { { 0x0002 } },
    static_cast<int8_t>(
        ::nn::settings::system::
            FriendPresenceOverlayPermission_FavoriteFriends),
};

::nn::settings::system::AccountNotificationSettings
    g_AccountNotificationSettings[
        ::nn::settings::system::AccountNotificationSettingsCountMax];

//!< アカウント毎の通知設定が等しいことを期待します。
void ExpectEqualAccountNotificationSettings(
    const ::nn::settings::system::AccountNotificationSettings& lhs,
    const ::nn::settings::system::AccountNotificationSettings& rhs) NN_NOEXCEPT;

} // namespace

//!< アプリケーション自動配信が有効か否かを表す値の設定を変更可能か
TEST(NewsSuite, AutomaticApplicationDownloadFlagTest1)
{
    ::nn::settings::system::SetAutomaticApplicationDownloadFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetAutomaticApplicationDownloadFlag());

    ::nn::settings::system::SetAutomaticApplicationDownloadFlag(true);

    EXPECT_TRUE(::nn::settings::system::GetAutomaticApplicationDownloadFlag());

    ::nn::settings::system::SetAutomaticApplicationDownloadFlag(false);

    EXPECT_FALSE(::nn::settings::system::GetAutomaticApplicationDownloadFlag());
}

//!< 通知設定を設定可能か
TEST(NewsSuite, NotificationSettingsTest1)
{
    ::nn::settings::system::SetNotificationSettings(NotificationSettingsA);

    ::nn::settings::system::SetNotificationSettings(NotificationSettingsB);

    ::nn::settings::system::NotificationSettings settings = {};

    ::nn::settings::system::GetNotificationSettings(&settings);

    EXPECT_EQ(
        NotificationSettingsB.flags.Test<
            ::nn::settings::system::NotificationFlag::Ringtone>(),
        settings.flags.Test<
            ::nn::settings::system::NotificationFlag::Ringtone>());

    EXPECT_EQ(
        NotificationSettingsB.flags.Test<
            ::nn::settings::system::NotificationFlag::DownloadCompletion>(),
        settings.flags.Test<
            ::nn::settings::system::NotificationFlag::DownloadCompletion>());

    EXPECT_EQ(
        NotificationSettingsB.flags.Test<
            ::nn::settings::system::NotificationFlag::EnablesNews>(),
        settings.flags.Test<
            ::nn::settings::system::NotificationFlag::EnablesNews>());

    EXPECT_EQ(
        NotificationSettingsB.flags.Test<
            ::nn::settings::system::NotificationFlag::IncomingLamp>(),
        settings.flags.Test<
            ::nn::settings::system::NotificationFlag::IncomingLamp>());

    EXPECT_EQ(NotificationSettingsB.volume, settings.volume);

    EXPECT_EQ(NotificationSettingsB.headTime.hour, settings.headTime.hour);

    EXPECT_EQ(NotificationSettingsB.headTime.minute, settings.headTime.minute);

    EXPECT_EQ(NotificationSettingsB.tailTime.hour, settings.tailTime.hour);

    EXPECT_EQ(NotificationSettingsB.tailTime.minute, settings.tailTime.minute);
}

//!< アカウント毎の通知設定を設定可能か
TEST(NewsSuite, AccountNotificationSettingsTest1)
{
    ::nn::settings::system::SetAccountNotificationSettings(
        &AccountNotificationSettingsA, 1);

    ::nn::settings::system::AccountNotificationSettings* settings =
          &g_AccountNotificationSettings[0];

    const auto countMax =
        ::nn::settings::system::AccountNotificationSettingsCountMax;

    EXPECT_EQ(
        1,
        ::nn::settings::system::GetAccountNotificationSettings(
            settings, countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualAccountNotificationSettings(
            AccountNotificationSettingsA, settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] =
            (i % 2 == 1) ? AccountNotificationSettingsA
                         : AccountNotificationSettingsB;
    }

    ::nn::settings::system::SetAccountNotificationSettings(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::AccountNotificationSettings();
    }

    EXPECT_EQ(
        countMax,
        ::nn::settings::system::GetAccountNotificationSettings(
            settings, countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualAccountNotificationSettings(
            (i % 2 == 1) ? AccountNotificationSettingsA
                         : AccountNotificationSettingsB,
            settings[i]);
    }

    ::nn::settings::system::SetAccountNotificationSettings(nullptr, 0);

    EXPECT_EQ(
        0,
        ::nn::settings::system::GetAccountNotificationSettings(
            settings, countMax));
}

namespace {

void ExpectEqualAccountNotificationSettings(
    const ::nn::settings::system::AccountNotificationSettings& lhs,
    const ::nn::settings::system::AccountNotificationSettings& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.uid, rhs.uid);
    EXPECT_EQ(
        lhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::FriendOnline>(),
        rhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::FriendOnline>());
    EXPECT_EQ(
        lhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::FriendRequest>(),
        rhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::FriendRequest>());
    EXPECT_EQ(
        lhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::CoralInvitation>(),
        rhs.flags.Test<
            ::nn::settings::system::AccountNotificationFlag::CoralInvitation>()
        );
    EXPECT_EQ(
        lhs.friendPresenceOverlayPermission,
        rhs.friendPresenceOverlayPermission);
}

} // namespace
